#!/usr/bin/env python3
"""
External API Test Script

Quick test script for external systems to validate their API key setup
and test connectivity to Krystal EA external endpoints.

Usage:
    python test_external_api.py --url https://krystal-ea.com --api-key your-api-key
    python test_external_api.py  # Uses environment variables

Requirements:
    pip install httpx python-dotenv
"""

import argparse
import asyncio
import json
import os
import sys
from datetime import datetime
from typing import Dict, Any

try:
    import httpx
    from dotenv import load_dotenv
except ImportError:
    print("❌ Missing dependencies. Install with: pip install httpx python-dotenv")
    sys.exit(1)

# Load environment variables
load_dotenv()


async def test_api_key_auth(base_url: str, api_key: str) -> bool:
    """Test API key authentication with a minimal request."""
    url = f"{base_url}/api/v1/sap/receive-process-orders"
    headers = {
        'Content-Type': 'application/json',
        'X-API-Key': api_key
    }
    
    # Minimal test payload (will fail validation but should pass auth)
    test_payload = {"test": "auth"}
    
    async with httpx.AsyncClient(timeout=10.0) as client:
        try:
            response = await client.post(url, json=test_payload, headers=headers)
            
            if response.status_code == 401:
                error_data = response.json() if response.content else {}
                if "API key" in error_data.get("detail", ""):
                    return False
            
            # Any non-401 response means auth passed
            # (400 is expected due to invalid payload)
            return True
            
        except httpx.ConnectError:
            print(f"❌ Connection failed to {base_url}")
            return False
        except Exception as e:
            print(f"❌ Request failed: {e}")
            return False


async def test_health_endpoint(base_url: str) -> bool:
    """Test health endpoint availability."""
    url = f"{base_url}/health"
    
    async with httpx.AsyncClient(timeout=10.0) as client:
        try:
            response = await client.get(url)
            if response.status_code == 200:
                health_data = response.json()
                print(f"✅ Health check passed: {health_data.get('status', 'unknown')}")
                return True
            else:
                print(f"⚠️ Health endpoint returned {response.status_code}")
                return False
        except httpx.ConnectError:
            print(f"❌ Cannot connect to {base_url}")
            return False
        except Exception as e:
            print(f"❌ Health check failed: {e}")
            return False


async def test_invalid_api_key(base_url: str) -> bool:
    """Test that invalid API key is properly rejected."""
    url = f"{base_url}/api/v1/sap/receive-process-orders"
    headers = {
        'Content-Type': 'application/json',
        'X-API-Key': 'invalid-key-12345'
    }
    
    test_payload = {"test": "invalid_auth"}
    
    async with httpx.AsyncClient(timeout=10.0) as client:
        try:
            response = await client.post(url, json=test_payload, headers=headers)
            
            if response.status_code == 401:
                error_data = response.json() if response.content else {}
                if "Invalid API key" in error_data.get("detail", ""):
                    print("✅ Invalid API key properly rejected")
                    return True
            
            print(f"⚠️ Expected 401 for invalid key, got {response.status_code}")
            return False
            
        except Exception as e:
            print(f"❌ Invalid key test failed: {e}")
            return False


async def test_missing_api_key(base_url: str) -> bool:
    """Test that missing API key is properly rejected."""
    url = f"{base_url}/api/v1/sap/receive-process-orders"
    headers = {
        'Content-Type': 'application/json'
        # No X-API-Key header
    }
    
    test_payload = {"test": "no_auth"}
    
    async with httpx.AsyncClient(timeout=10.0) as client:
        try:
            response = await client.post(url, json=test_payload, headers=headers)
            
            if response.status_code == 401:
                error_data = response.json() if response.content else {}
                if "API key required" in error_data.get("detail", ""):
                    print("✅ Missing API key properly rejected")
                    return True
            
            print(f"⚠️ Expected 401 for missing key, got {response.status_code}")
            return False
            
        except Exception as e:
            print(f"❌ Missing key test failed: {e}")
            return False


def validate_api_key(api_key: str) -> bool:
    """Validate API key format."""
    if not api_key:
        print("❌ API key is empty")
        return False
    
    if len(api_key) < 32:
        print(f"❌ API key too short: {len(api_key)} characters (minimum 32)")
        return False
    
    if len(api_key) < 64:
        print(f"⚠️ API key length: {len(api_key)} characters (64+ recommended)")
    else:
        print(f"✅ API key length: {len(api_key)} characters")
    
    return True


def validate_base_url(base_url: str) -> bool:
    """Validate base URL format."""
    if not base_url:
        print("❌ Base URL is empty")
        return False
    
    if not base_url.startswith(('http://', 'https://')):
        print("❌ Base URL must start with http:// or https://")
        return False
    
    if base_url.startswith('http://') and 'localhost' not in base_url:
        print("⚠️ Using HTTP for non-localhost URL (HTTPS recommended)")
    
    return True


async def main():
    """Main test function."""
    parser = argparse.ArgumentParser(description='Test Krystal EA External API connectivity and authentication')
    parser.add_argument('--url', help='Base URL (e.g., https://krystal-ea.com)')
    parser.add_argument('--api-key', help='API key for authentication')
    parser.add_argument('--verbose', '-v', action='store_true', help='Verbose output')
    
    args = parser.parse_args()
    
    # Get configuration
    base_url = args.url or os.getenv('KRYSTAL_EA_BASE_URL', 'http://localhost:8000')
    api_key = args.api_key or os.getenv('SAP_API_KEY')
    
    # Remove trailing slash
    base_url = base_url.rstrip('/')
    
    print("🧪 Krystal EA External API Test")
    print("=" * 40)
    print(f"📡 Base URL: {base_url}")
    print(f"🔑 API Key: {api_key[:8] + '...' + api_key[-4:] if api_key and len(api_key) >= 12 else 'Not provided'}")
    print()
    
    # Validate inputs
    if not validate_base_url(base_url):
        return 1
    
    if not api_key:
        print("❌ No API key provided. Set SAP_API_KEY environment variable or use --api-key")
        return 1
    
    if not validate_api_key(api_key):
        return 1
    
    print()
    
    # Run tests
    tests_passed = 0
    total_tests = 5
    
    print("🔍 Running connectivity and authentication tests...")
    print()
    
    # Test 1: Health endpoint
    print("1️⃣ Testing health endpoint...")
    if await test_health_endpoint(base_url):
        tests_passed += 1
    print()
    
    # Test 2: Missing API key rejection
    print("2️⃣ Testing missing API key rejection...")
    if await test_missing_api_key(base_url):
        tests_passed += 1
    print()
    
    # Test 3: Invalid API key rejection
    print("3️⃣ Testing invalid API key rejection...")
    if await test_invalid_api_key(base_url):
        tests_passed += 1
    print()
    
    # Test 4: Valid API key authentication
    print("4️⃣ Testing valid API key authentication...")
    if await test_api_key_auth(base_url, api_key):
        print("✅ Valid API key accepted")
        tests_passed += 1
    else:
        print("❌ Valid API key rejected - check your API key")
    print()
    
    # Test 5: HTTPS check (for production)
    print("5️⃣ Testing security configuration...")
    if base_url.startswith('https://'):
        print("✅ Using HTTPS (secure)")
        tests_passed += 1
    elif 'localhost' in base_url or '127.0.0.1' in base_url:
        print("✅ Using HTTP for localhost (acceptable for development)")
        tests_passed += 1
    else:
        print("❌ Using HTTP for remote URL (security risk)")
    print()
    
    # Summary
    print("📊 Test Results")
    print("=" * 40)
    print(f"✅ Passed: {tests_passed}/{total_tests}")
    print(f"❌ Failed: {total_tests - tests_passed}/{total_tests}")
    
    if tests_passed == total_tests:
        print()
        print("🎉 All tests passed! Your external API setup is ready.")
        print()
        print("Next steps:")
        print("- Use the API key in your BTP middleware configuration")
        print("- Implement proper error handling in your integration")
        print("- Monitor rate limits and response times")
        print("- See EXTERNAL_API_DOCUMENTATION.md for complete API reference")
        return 0
    else:
        print()
        print("⚠️ Some tests failed. Please check your configuration.")
        print()
        print("Common issues:")
        print("- Incorrect API key (check SAP_API_KEY environment variable)")
        print("- Wrong base URL (check KRYSTAL_EA_BASE_URL)")
        print("- Network connectivity issues")
        print("- Krystal EA server not running")
        return 1


if __name__ == "__main__":
    try:
        exit_code = asyncio.run(main())
        sys.exit(exit_code)
    except KeyboardInterrupt:
        print("\n❌ Test interrupted by user")
        sys.exit(1)
    except Exception as e:
        print(f"\n❌ Unexpected error: {e}")
        sys.exit(1)