#!/usr/bin/env python3
"""
Utility script to generate secure API keys for SAP integration.
"""

import secrets
import string
import hashlib
from datetime import datetime


def generate_secure_api_key(length: int = 64) -> str:
    """
    Generate a cryptographically secure API key.
    
    Args:
        length: Length of the API key (minimum 32)
        
    Returns:
        Secure API key string
    """
    if length < 32:
        raise ValueError("API key length must be at least 32 characters")
    
    # Use URL-safe characters (letters, digits, -, _)
    alphabet = string.ascii_letters + string.digits + '-_'
    api_key = ''.join(secrets.choice(alphabet) for _ in range(length))
    
    return api_key


def generate_api_key_with_prefix(prefix: str = "krystal", length: int = 64) -> str:
    """
    Generate API key with a specific prefix.
    
    Args:
        prefix: Prefix for the API key
        length: Total length including prefix
        
    Returns:
        API key with prefix
    """
    if length < 32:
        raise ValueError("API key length must be at least 32 characters")
    
    # Calculate remaining length after prefix and separator
    remaining_length = length - len(prefix) - 1  # -1 for underscore
    
    if remaining_length < 16:
        raise ValueError(f"Remaining length after prefix too short: {remaining_length}")
    
    key_part = generate_secure_api_key(remaining_length)
    return f"{prefix}_{key_part}"


def generate_timestamped_api_key(length: int = 64) -> str:
    """
    Generate API key with timestamp component.
    
    Args:
        length: Total length of the API key
        
    Returns:
        API key with timestamp component
    """
    timestamp = datetime.utcnow().strftime("%Y%m%d")
    remaining_length = length - len(timestamp) - 1  # -1 for underscore
    
    if remaining_length < 24:
        remaining_length = 24  # Minimum secure length
        length = len(timestamp) + 1 + remaining_length
    
    key_part = generate_secure_api_key(remaining_length)
    return f"{timestamp}_{key_part}"


def validate_api_key_strength(api_key: str) -> dict:
    """
    Validate the strength of an API key.
    
    Args:
        api_key: API key to validate
        
    Returns:
        Dictionary with validation results
    """
    results = {
        "length": len(api_key),
        "min_length_ok": len(api_key) >= 32,
        "has_letters": any(c.isalpha() for c in api_key),
        "has_digits": any(c.isdigit() for c in api_key),
        "has_special": any(c in '-_' for c in api_key),
        "entropy_bits": len(api_key) * 6,  # Approximate for base64-like encoding
        "strength": "weak"
    }
    
    # Determine strength
    if results["length"] >= 64 and results["has_letters"] and results["has_digits"]:
        results["strength"] = "very_strong"
    elif results["length"] >= 48 and results["has_letters"] and results["has_digits"]:
        results["strength"] = "strong"
    elif results["length"] >= 32 and results["has_letters"]:
        results["strength"] = "medium"
    
    return results


def main():
    """Main function to generate and display API keys."""
    print("🔐 SAP Integration API Key Generator")
    print("=" * 50)
    
    # Generate different types of API keys
    print("\n1. Standard Secure API Key (64 characters):")
    standard_key = generate_secure_api_key(64)
    print(f"   {standard_key}")
    
    print("\n2. Prefixed API Key (Krystal SAP):")
    prefixed_key = generate_api_key_with_prefix("krystal-sap", 64)
    print(f"   {prefixed_key}")
    
    print("\n3. Timestamped API Key:")
    timestamped_key = generate_timestamped_api_key(64)
    print(f"   {timestamped_key}")
    
    print("\n4. Extra Long API Key (128 characters):")
    long_key = generate_secure_api_key(128)
    print(f"   {long_key}")
    
    # Validate the standard key
    print(f"\n📊 Key Strength Analysis (Standard Key):")
    validation = validate_api_key_strength(standard_key)
    print(f"   Length: {validation['length']} characters")
    print(f"   Minimum Length OK: {validation['min_length_ok']}")
    print(f"   Has Letters: {validation['has_letters']}")
    print(f"   Has Digits: {validation['has_digits']}")
    print(f"   Has Special Chars: {validation['has_special']}")
    print(f"   Entropy: ~{validation['entropy_bits']} bits")
    print(f"   Strength: {validation['strength'].upper()}")
    
    # Generate .env format
    print(f"\n📝 .env Configuration:")
    print(f"SAP_API_KEY={standard_key}")
    print(f"API_KEY_HEADER=X-API-Key")
    
    # Security recommendations
    print(f"\n🛡️  Security Recommendations:")
    print(f"   • Store the API key securely (environment variables)")
    print(f"   • Use HTTPS for all API communications")
    print(f"   • Rotate the API key regularly (quarterly)")
    print(f"   • Monitor API key usage in logs")
    print(f"   • Use different keys for different environments")
    print(f"   • Never commit API keys to version control")
    
    # BTP Configuration
    print(f"\n🔧 BTP Configuration:")
    print(f"   Configure your BTP middleware to send this API key:")
    print(f"   Header: X-API-Key")
    print(f"   Value: {standard_key}")
    
    print(f"\n✅ API Key Generation Complete!")


if __name__ == "__main__":
    main()