from fastapi import APIRouter, HTTPException, Depends, Request, Form, Cookie, Query
from fastapi.responses import JSONResponse, HTMLResponse
from typing import Optional, List, Dict
import mysql.connector
from mysql.connector import Error
from datetime import datetime
import secrets
import json
import html as html_module
import bcrypt
import math
import re
import os
from backend.main import (
    get_db_connection, get_session, require_auth,
    week_number, rand_number, sessions, DB_CONFIG,
    create_access_token, verify_token, get_current_user as get_current_user_from_token
)
from pydantic import BaseModel, Field, EmailStr
from typing import Optional, List, Dict
from fastapi import status

# Module-level storage for user CSRF tokens (keyed by user_id)
user_csrf_tokens: Dict[str, str] = {}

# Response models for API documentation
class SuccessResponse(BaseModel):
    """Standard success response"""
    success: bool = Field(True, description="Indicates the operation was successful")
    message: Optional[str] = Field(None, description="Success message")
    
class ErrorResponse(BaseModel):
    """Standard error response"""
    detail: str = Field(..., description="Error message describing what went wrong")
    
class LoginResponse(BaseModel):
    """Login success response"""
    success: bool = Field(True, description="Login successful")
    redirect: str = Field(..., description="Suggested redirect URL based on user role")
    message: str = Field(..., description="Success message")
    access_token: str = Field(..., description="JWT token for Authorization header")
    token_type: str = Field("bearer", description="Token type")
    csrf_token: str = Field(..., description="CSRF protection token")
    
    class Config:
        schema_extra = {
            "example": {
                "success": True,
                "redirect": "blue.html",
                "message": "Login successful",
                "access_token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
                "token_type": "bearer",
                "csrf_token": "MHCyO_ObDkEfhxW6qUhouu1I71ayQdsRQ780YWvi2lk"
            }
        }

# Pydantic models for routes
class LoginRequest(BaseModel):
    """Request model for user authentication"""
    email: EmailStr = Field(..., description="User's email address", example="user@example.com")
    password: str = Field(..., description="User's password", min_length=1, example="password123")
    
    class Config:
        schema_extra = {
            "example": {
                "email": "beryl.ohuru@gmail.com",
                "password": "password"
            }
        }

class BackendRequest(BaseModel):
    """Request model for submitting batch data to the backend"""
    tableData: List[Dict] = Field(..., description="List of items with their weights and batch information", 
                                  example=[{"item": "Water", "actual_weight": "50.5", "bn": "BATCH001"}])
    operator: EmailStr = Field(..., description="Email of the operator who performed the weighing", 
                              example="operator@example.com")
    analyst: EmailStr = Field(..., description="Email of the analyst assigned to review this batch", 
                             example="analyst@example.com")
    teamLead: EmailStr = Field(..., description="Email of the team lead assigned to review this batch", 
                               example="teamlead@example.com")
    product: str = Field(..., description="Product name/code (must exist in products table)", 
                        example="vaseline_extra_strength")
    factory: str = Field(..., description="Factory identifier", example="NT")
    csrf_token: str = Field(..., description="CSRF protection token obtained from login response", 
                           example="MHCyO_ObDkEfhxW6qUhouu1I71ayQdsRQ780YWvi2lk")
    
    class Config:
        schema_extra = {
            "example": {
                "tableData": [
                    {"item": "Water", "actual_weight": "50.5", "bn": "BATCH001", 
                     "date_manufacturer": "2025-01-15", "expiry_date": "2026-01-15", "mdn": "MDN123"}
                ],
                "operator": "beryl.ohuru@gmail.com",
                "analyst": "polycap@krystal-ea.com",
                "teamLead": "keesi@krystal-ea.com",
                "product": "vaseline_extra_strength",
                "factory": "NT",
                "csrf_token": "your_csrf_token_here"
            }
        }

class BatchItemsRequest(BaseModel):
    """Request model for querying batch items with optional filters"""
    serial_no: Optional[str] = Field(None, description="Serial number of the batch item to retrieve. If not provided, returns all items matching other filters.", 
                                     example="2026_1_5_1")
    shift: Optional[str] = Field(None, description="Shift name (e.g., 'Day Shift', 'Night Shift'). If not provided, returns items from all shifts.", 
                                 example="Day Shift")
    operator: Optional[str] = Field(None, description="Email of the operator. If not provided and user is analyst/team_lead, returns their assigned items.", 
                                   example="operator@example.com")
    
    class Config:
        schema_extra = {
            "example": {
                "serial_no": "2026_1_5_1",
                "shift": "Day Shift",
                "operator": "beryl.ohuru@gmail.com"
            }
        }

class BatchItemsBulkRequest(BaseModel):
    """Request model for querying multiple batch items at once"""
    items: List[BatchItemsRequest] = Field(..., description="List of batch item queries to execute", 
                                           min_items=1, max_items=100)
    
    class Config:
        schema_extra = {
            "example": {
                "items": [
                    {"serial_no": "2026_1_5_1", "shift": "Day Shift", "operator": "operator@example.com"},
                    {"serial_no": "2026_1_5_2", "shift": "Day Shift", "operator": "operator@example.com"}
                ]
            }
        }

class ApproveItemRequest(BaseModel):
    """Request model for approving or rejecting a batch item"""
    serial_no: str = Field(..., description="Serial number of the batch item", example="2026_1_5_1")
    operator: EmailStr = Field(..., description="Email of the operator who created the item", 
                              example="operator@example.com")
    shift: str = Field(..., description="Shift name", example="Day Shift")
    analyst: EmailStr = Field(..., description="Email of the analyst assigned to review", 
                             example="analyst@example.com")
    type: str = Field(..., description="Approval type: 'analyst' or 'tl' (team lead)", 
                     example="analyst", pattern="^(analyst|tl)$")
    action: str = Field(..., description="Action to take: 'approve' or 'reject'", 
                       example="approve", pattern="^(approve|reject)$")
    csrf_token: str = Field(..., description="CSRF protection token", example="your_csrf_token_here")
    
    class Config:
        schema_extra = {
            "example": {
                "serial_no": "2026_1_5_1",
                "operator": "operator@example.com",
                "shift": "Day Shift",
                "analyst": "analyst@example.com",
                "type": "analyst",
                "action": "approve",
                "csrf_token": "your_csrf_token_here"
            }
        }

class ListItemsRequest(BaseModel):
    """Request model for listing items from a product table"""
    data: str = Field(..., description="Batch number to filter items by", example="BATCH001", min_length=1)
    csrf_token: str = Field(..., description="CSRF protection token", example="your_csrf_token_here")
    
    class Config:
        schema_extra = {
            "example": {
                "data": "BATCH001",
                "csrf_token": "your_csrf_token_here"
            }
        }

class UpdateRowRequest(BaseModel):
    """Request model for updating a specific field in a product table row"""
    value: str = Field(..., description="New value to set for the column", example="60")
    column: str = Field(..., description="Column name to update (must be in whitelist: ingredient, expected_weight, weight_variance, to_be_measured, mdn, batch_no)", 
                       example="expected_weight", pattern="^(ingredient|expected_weight|weight_variance|to_be_measured|mdn|batch_no)$")
    table: str = Field(..., description="Product table name (must exist in products table)", 
                      example="test", pattern="^[a-zA-Z0-9_]+$")
    id: int = Field(..., description="Row ID to update", example=70, gt=0)
    csrf_token: str = Field(..., description="CSRF protection token", example="your_csrf_token_here")
    
    class Config:
        schema_extra = {
            "example": {
                "value": "60",
                "column": "expected_weight",
                "table": "test",
                "id": 70,
                "csrf_token": "your_csrf_token_here"
            }
        }

class AreaRequest(BaseModel):
    """Request model for creating or updating an area of operation"""
    name: str = Field(..., description="Area name (must be unique)", example="Food", min_length=1, max_length=100)
    code: str = Field(..., description="Area code (must be unique)", example="K013", min_length=1, max_length=50)
    
    class Config:
        schema_extra = {
            "example": {
                "name": "Food",
                "code": "K013"
            }
        }

class AddUserRequest(BaseModel):
    """Request model for adding a new user to the system"""
    name: str = Field(..., description="User's full name", example="John Doe", min_length=1, max_length=255)
    email: EmailStr = Field(..., description="User's email address (must be unique)", example="john.doe@example.com")
    password: str = Field(..., description="User's password (will be hashed before storage)", 
                         example="SecurePassword123", min_length=6)
    title: str = Field(..., description="User's role/title: admin, super_admin, analyst, head_of_analyst, team_lead, head_of_team_lead, operator, user", 
                      example="analyst", pattern="^(admin|super_admin|analyst|head_of_analyst|team_lead|head_of_team_lead|operator|user)$")
    shift: str = Field(..., description="User's shift assignment", example="Day Shift")
    areas: Optional[str] = Field("", description="Comma-separated list of area names the user has access to", 
                                 example="Food,BW,PC")
    
    class Config:
        schema_extra = {
            "example": {
                "name": "John Doe",
                "email": "john.doe@example.com",
                "password": "SecurePassword123",
                "title": "analyst",
                "shift": "Day Shift",
                "areas": "Food,BW"
            }
        }

class UpdateUserRequest(BaseModel):
    """Request model for updating an existing user's information"""
    name: str = Field(..., description="User's full name", example="John Doe", min_length=1, max_length=255)
    title: str = Field(..., description="User's role/title", example="analyst")
    shift: str = Field(..., description="User's shift assignment", example="Day Shift")
    status: Optional[int] = Field(1, description="User status: 1 = active, 0 = inactive", example=1, ge=0, le=1)
    areas: Optional[str] = Field("", description="Comma-separated list of area names", example="Food,BW")
    password: Optional[str] = Field(None, description="New password (only used if change_password is true)", 
                                   example="NewPassword123", min_length=6)
    change_password: Optional[bool] = Field(False, description="Set to true to change the user's password", example=False)
    
    class Config:
        schema_extra = {
            "example": {
                "name": "John Doe",
                "title": "analyst",
                "shift": "Day Shift",
                "status": 1,
                "areas": "Food,BW",
                "password": None,
                "change_password": False
            }
        }

router = APIRouter()

# Pagination helper function
def paginate_results(items: List[Dict], page: int = 1, limit: int = 50) -> Dict:
    """Paginate a list of items and return paginated response with metadata"""
    # Validate page and limit
    page = max(1, page)  # Page must be at least 1
    limit = max(1, min(100, limit))  # Limit between 1 and 100, default 50
    
    total_items = len(items)
    total_pages = math.ceil(total_items / limit) if total_items > 0 else 0
    offset = (page - 1) * limit
    
    # Get paginated items
    paginated_items = items[offset:offset + limit]
    
    return {
        "items": paginated_items,
        "pagination": {
            "current_page": page,
            "per_page": limit,
            "total_items": total_items,
            "total_pages": total_pages,
            "has_next": page < total_pages,
            "has_previous": page > 1
        }
    }

# Login endpoint
@router.post(
    "/api/login",
    summary="User Authentication",
    description="""
    Authenticate a user with email and password.
    
    **Features:**
    - Supports both hashed (bcrypt) and plain text passwords (legacy migration)
    - Automatically migrates plain text passwords to hashed on successful login
    - Returns JWT access token and CSRF token for subsequent API calls
    - Case-insensitive email matching
    
    **Response includes:**
    - `access_token`: JWT token for Authorization header (Bearer token)
    - `csrf_token`: CSRF protection token for form submissions
    - `redirect`: Suggested redirect URL based on user role
    
    **Security:**
    - Passwords are validated securely
    - JWT tokens expire after 24 hours (configurable)
    - CSRF tokens are stored per user for protection
    """,
    response_model=LoginResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Login successful",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "redirect": "blue.html",
                        "message": "Login successful",
                        "access_token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
                        "token_type": "bearer",
                        "csrf_token": "MHCyO_ObDkEfhxW6qUhouu1I71ayQdsRQ780YWvi2lk"
                    }
                }
            }
        },
        401: {
            "description": "Authentication failed - Invalid email or password",
            "content": {
                "application/json": {
                    "example": {
                        "detail": "Invalid credentials"
                    }
                }
            }
        },
        500: {
            "description": "Internal server error - Database connection failed or unexpected error",
            "content": {
                "application/json": {
                    "example": {
                        "detail": "Login error: Database connection failed"
                    }
                }
            }
        }
    },
    tags=["Authentication"]
)
async def login(request: Request, login_data: LoginRequest):
    """
    Authenticate user and return JWT token.
    
    **Example Request:**
    ```json
    {
        "email": "beryl.ohuru@gmail.com",
        "password": "password"
    }
    ```
    
    **Example Response:**
    ```json
    {
        "success": true,
        "redirect": "blue.html",
        "message": "Login successful",
        "access_token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
        "token_type": "bearer",
        "csrf_token": "MHCyO_ObDkEfhxW6qUhouu1I71ayQdsRQ780YWvi2lk"
    }
    ```
    """
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Debug logging
        print(f"Login attempt for email: {login_data.email}")
        
        # Normalize email (trim and lowercase for case-insensitive matching)
        normalized_email = login_data.email.strip().lower()
        
        cursor.execute(
            "SELECT * FROM users WHERE LOWER(TRIM(email)) = %s",
            (normalized_email,)
        )
        user = cursor.fetchone()
        
        if not user:
            print(f"User not found: {login_data.email} (normalized: {normalized_email})")
            raise HTTPException(status_code=401, detail="Invalid credentials")
        
        # Get password from database
        input_password = login_data.password.strip()
        db_password = (user['password'] or '').strip()
        
        # Debug: Check password comparison (don't log actual password)
        print(f"User found: {user.get('email')}")
        print(f"Input password length: {len(input_password)}, DB password length: {len(db_password)}")
        
        # Support both hashed passwords (new) and plain text (legacy - for migration)
        password_valid = False
        
        # Check if password is hashed (starts with $2y$, $2a$, or $2b$ for bcrypt)
        if db_password.startswith('$2y$') or db_password.startswith('$2a$') or db_password.startswith('$2b$'):
            # Password is hashed - use bcrypt to verify
            try:
                password_valid = bcrypt.checkpw(input_password.encode('utf-8'), db_password.encode('utf-8'))
                print(f"Password is hashed, verification result: {password_valid}")
            except Exception as e:
                print(f"Error verifying hashed password: {e}")
                password_valid = False
        else:
            # Legacy plain text password - check directly (for migration period)
            password_valid = (input_password == db_password)
            print(f"Password is plain text, match: {password_valid}")
            
            # If plain text password matches, hash it and update database (migration)
            if password_valid and user.get('user_id'):
                try:
                    hashed_password = bcrypt.hashpw(input_password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
                    update_cursor = conn.cursor()
                    update_cursor.execute(
                        "UPDATE users SET password = %s WHERE user_id = %s",
                        (hashed_password, user['user_id'])
                    )
                    conn.commit()
                    update_cursor.close()
                    print(f"Migrated plain text password to hashed for user: {login_data.email}")
                except Exception as e:
                    print(f"Warning: Failed to migrate password to hashed: {e}")
                    # Don't fail login if migration fails
        
        if not password_valid:
            print(f"Password mismatch for user: {login_data.email}")
            raise HTTPException(status_code=401, detail="Invalid credentials")
        
        # Create JWT token with user data
        token_data = {
            'id': user['user_id'],
            'name': user['email'],
            'email': user['email'],
            'imglink': user.get('imglink', ''),
            'shift': user.get('shift', ''),
            'title': user.get('title', ''),
            'user_name': user.get('name', ''),
        }
        
        access_token = create_access_token(data=token_data)
        
        print(f"DEBUG: Created JWT token for user {user.get('email')}")
        
        # Also create session for backward compatibility (optional)
        # Use user_id as part of session key for consistency with JWT
        session_id = secrets.token_urlsafe(32)
        sessions[session_id] = token_data.copy()
        csrf_token = secrets.token_urlsafe(32)
        sessions[session_id]['csrf_token'] = csrf_token
        # Store CSRF token by user_id for JWT-based authentication
        user_csrf_tokens[user['user_id']] = csrf_token
        print(f"DEBUG: Stored CSRF token for user {user['user_id']}: {csrf_token[:20]}...")
        
        # Determine redirect URL
        title = user.get('title', '')
        if title in ['team_lead', 'analyst']:
            redirect_url = "analyst-tl.html"
        else:
            redirect_url = "blue.html"
        
        response_data = {
            "success": True,
            "redirect": redirect_url,
            "message": "Login successful",
            "access_token": access_token,  # JWT token
            "token_type": "bearer",
            "csrf_token": csrf_token  # CSRF token for form protection
        }
        
        print(f"DEBUG: Login response data: {response_data}")
        print(f"DEBUG: CSRF token in response: {csrf_token}")
        
        response = JSONResponse(response_data)
        
        return response
        
    except HTTPException as e:
        # Log the error for debugging
        print(f"HTTPException in login: {e.status_code} - {e.detail}")
        raise
    except Exception as e:
        # Log the error for debugging
        print(f"Exception in login: {type(e).__name__} - {str(e)}")
        import traceback
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=f"Login error: {str(e)}")
    finally:
        if cursor:
            cursor.close()
        if conn:
            conn.close()

# Logout endpoint
@router.get(
    "/api/logout",
    summary="User Logout",
    description="Log out the current user and clear session data. Note: JWT tokens remain valid until expiration (client should discard them).",
    response_model=SuccessResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Logout successful",
            "content": {
                "application/json": {
                    "example": {
                        "success": True
                    }
                }
            }
        },
        500: {
            "description": "Internal server error",
            "content": {
                "application/json": {
                    "example": {
                        "detail": "Logout error: Session cleanup failed"
                    }
                }
            }
        }
    },
    tags=["Authentication"]
)
async def logout(request: Request):
    """
    Log out the current user.
    
    Clears session data and removes session cookies.
    
    **Example Response:**
    ```json
    {
        "success": true
    }
    ```
    """
    session_id = request.cookies.get('session_id')
    if session_id and session_id in sessions:
        del sessions[session_id]
    response = JSONResponse({"success": True})
    response.delete_cookie("session_id")
    return response

# Backend data insertion endpoint
@router.post(
    "/api/backend",
    summary="Submit Batch Data",
    description="""
    Submit batch weighing data from the frontend to the backend.
    
    **Purpose:**
    - Stores batch item data (weights, batch numbers, dates) in the database
    - Creates records for operator, analyst, and team lead assignments
    - Generates unique serial numbers for tracking
    
    **Authorization:**
    - Requires JWT authentication
    - Requires valid CSRF token
    
    **Data Flow:**
    1. Validates CSRF token
    2. Creates product table if it doesn't exist
    3. Inserts batch items with serial numbers
    4. Links items to operator, analyst, and team lead
    
    **Security:**
    - CSRF protection via token validation
    - JWT authentication required
    - Input validation on all fields
    """,
    response_description="Batch data successfully submitted with serial numbers",
    tags=["Data Operations"]
)
async def backend_data(request: Request, backend_data: BackendRequest):
    """
    Submit batch weighing data to the backend.
    
    **Example Request:**
    ```json
    {
        "tableData": [
            {
                "item": "Water",
                "actual_weight": "50.5",
                "bn": "BATCH001",
                "date_manufacturer": "2025-01-15",
                "expiry_date": "2026-01-15",
                "mdn": "MDN123"
            }
        ],
        "operator": "beryl.ohuru@gmail.com",
        "analyst": "polycap@krystal-ea.com",
        "teamLead": "keesi@krystal-ea.com",
        "product": "vaseline_extra_strength",
        "factory": "NT",
        "csrf_token": "your_csrf_token_here"
    }
    ```
    
    **Example Response:**
    ```json
    {
        "success": true,
        "message": "Data submitted successfully",
        "serial_numbers": ["2026_1_5_1", "2026_1_5_2"]
    }
    ```
    """
    # Get current user for CSRF validation
    user = require_auth(request)
    user_id = user.get('id') or user.get('user_id')
    
    # Validate CSRF token - check both session and user-based storage
    session_id, session = get_session(request)
    csrf_valid = False
    
    print(f"DEBUG: Validating CSRF token for user_id: {user_id}")
    print(f"DEBUG: Received CSRF token: {backend_data.csrf_token[:20]}...")
    print(f"DEBUG: Session CSRF token: {session.get('csrf_token', 'None')[:20] if session.get('csrf_token') else 'None'}...")
    print(f"DEBUG: User CSRF tokens available: {list(user_csrf_tokens.keys())}")
    
    # Check session-based CSRF token
    if backend_data.csrf_token == session.get('csrf_token'):
        csrf_valid = True
        print("DEBUG: CSRF token validated via session")
    # Check user-based CSRF token (for JWT authentication)
    elif user_id in user_csrf_tokens:
        if backend_data.csrf_token == user_csrf_tokens[user_id]:
            csrf_valid = True
            print(f"DEBUG: CSRF token validated via user storage: {user_csrf_tokens[user_id][:20]}...")
        else:
            print(f"DEBUG: CSRF token mismatch. Expected: {user_csrf_tokens[user_id][:20]}..., Got: {backend_data.csrf_token[:20]}...")
    else:
        print(f"DEBUG: User ID {user_id} not found in user_csrf_tokens")
    
    if not csrf_valid:
        raise HTTPException(status_code=403, detail="CSRF token validation failed")
    
    conn = get_db_connection()
    cursor = conn.cursor()
    
    try:
        conn.start_transaction()
        
        # Get expected weights
        weight_array = {}
        cursor.execute(f"SELECT ingredient, expected_weight FROM `{backend_data.product}`")
        for row in cursor.fetchall():
            weight_array[row[0]] = row[1]
        
        # Prepare insert statement
        insert_query = """INSERT INTO `data` (
            product, items, batch_no, expected_weight, actual_weight, 
            scan_time, operator, analyst, scan_date, week, 
            units, serial_no, status, mdn, shift, 
            date_manufacturer, expiry_date, teamLead, analyst_status, factory, tl_status
        ) VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)"""
        
        # Calculate date/time values
        now = datetime.now()
        date = now.strftime("%Y-%m-%d")
        time = now.strftime("%H:%M:%S")
        yr = now.year
        week = week_number(date)
        day_of_week = now.weekday()
        hour = now.hour
        shift = 'Day Shift' if 6 <= hour < 18 else 'Night Shift'
        shiftcode = '1' if shift == 'Day Shift' else '2'
        serial = f"{yr}_{week}_{day_of_week}_{shiftcode}"
        
        success_count = 0
        for item_data in backend_data.tableData:
            item = item_data.get('item', '').strip()
            actual_weight = item_data.get('actual_weight', '').strip()
            dt_manufacture = item_data.get('date_manufacturer', '').strip() or None
            # Always use auto-generated batch number based on date/time.
            # QR-scanned batch numbers are not stored in the backend.
            bn = f"{date}_{time.replace(':', '')}"
            expiry_date = item_data.get('expiry_date', '').strip() or None
            mdn = item_data.get('mdn', '').strip()
            expectedweight = weight_array.get(item, '')
            units = 'kg'
            status = '1'
            analyst_status = '0'
            tl_status = '0'
            
            cursor.execute(insert_query, (
                backend_data.product, item, bn, expectedweight, actual_weight,
                time, backend_data.operator, backend_data.analyst, date, week,
                units, serial, status, mdn, shift,
                dt_manufacture, expiry_date, backend_data.teamLead, analyst_status,
                backend_data.factory, tl_status
            ))
            success_count += 1
        
        conn.commit()
        return {"status": "done"}
        
    except Exception as e:
        conn.rollback()
        raise HTTPException(status_code=500, detail=f"Failed to update data: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Get table endpoint
@router.get(
    "/api/get_table",
    summary="Get Product Table Data",
    description="""
    Retrieve product table data by product ID.
    
    **Returns:**
    - Product details (name, table name)
    - Paginated list of ingredients with weights, batch numbers, dates, MDN
    
    **Security:**
    - Requires authentication
    - ID validation (must be > 0)
    - Table name validation (regex, existence check)
    - XSS prevention (HTML escaping)
    
    **Pagination:**
    - Default: 50 items per page
    - Maximum: 100 items per page
    - Page numbers start at 1
    """,
    response_description="Product table data in JSON format with pagination",
    tags=["Data Operations"]
)
async def get_table(
    request: Request, 
    id: int = Query(..., description="Product ID from products table", gt=0, example=1),
    page: int = Query(1, ge=1, description="Page number (starts at 1)", example=1),
    limit: int = Query(50, ge=1, le=100, description="Items per page (max 100)", example=50)
):
    """
    Get product table data - Returns JSON for frontend to render.
    
    **Example Request:**
    ```
    GET /api/get_table?id=1&page=1&limit=50
    ```
    
    **Example Response:**
    ```json
    {
        "product": {
            "product_name": "vaseline_extra_strength",
            "table_name": "test"
        },
        "ingredients": [
            {
                "id": 1,
                "ingredient": "Water",
                "actual_weight": "50.5",
                "expected_weight": "50.0",
                "batch_no": "BATCH001",
                "mdn": "MDN123",
                "date_manufacturer": "2025-01-15",
                "expiry_date": "2026-01-15"
            }
        ],
        "pagination": {
            "current_page": 1,
            "per_page": 50,
            "total_items": 100,
            "total_pages": 2,
            "has_next": true,
            "has_previous": false
        }
    }
    ```
    """
    session = require_auth(request)
    
    # Validate ID is positive
    if id <= 0:
        raise HTTPException(status_code=400, detail="Invalid product ID")
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Use parameterized query (safe from SQL injection)
        cursor.execute("SELECT * FROM products WHERE id = %s", (id,))
        product = cursor.fetchone()
        
        if not product:
            return JSONResponse({
                "success": False,
                "message": "Product not found",
                "product_id": id
            }, status_code=404)
        
        table_name = product['name']
        
        # Validate table name to prevent SQL injection
        # Only allow alphanumeric and underscore characters
        import re
        if not re.match(r'^[a-zA-Z0-9_]+$', table_name):
            raise HTTPException(status_code=400, detail="Invalid product table name")
        
        # Verify table exists in products list (additional security)
        cursor.execute("SELECT name FROM products WHERE name = %s", (table_name,))
        if not cursor.fetchone():
            raise HTTPException(status_code=404, detail="Product table not found")
        
        # Use parameterized query with table name validation
        # Note: MySQL doesn't support parameterized table names, so we validate instead
        cursor.execute(f"SELECT * FROM `{table_name}`")
        all_rows = cursor.fetchall()
        
        # Convert rows to list of dictionaries
        all_ingredients = []
        for row in all_rows:
            all_ingredients.append({
                "id": row.get('id'),
                "ingredient": row.get('ingredient', ''),
                "mdn": row.get('mdn', ''),
                "batch_no": row.get('batch_no', ''),
                "expected_weight": str(row.get('expected_weight', '')) if row.get('expected_weight') is not None else '',
                "weight_variance": str(row.get('weight_variance', '')) if row.get('weight_variance') is not None else '',
                "updated_by": row.get('updated_by', ''),
                "to_be_measured": row.get('to_be_measured', 0) if 'to_be_measured' in row else None
            })
        
        # Apply pagination
        paginated = paginate_results(all_ingredients, page, limit)
        
        return JSONResponse({
            "success": True,
            "product_id": id,
            "product_name": table_name,
            "ingredients": paginated["items"],
            "pagination": paginated["pagination"]
        })
        
    except HTTPException:
        raise
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Database error: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Get batch items endpoint
@router.post("/api/get_batch_items")
async def get_batch_items(
    request: Request, 
    batch_data: BatchItemsRequest,
    page: int = Query(1, ge=1, description="Page number (starts at 1)"),
    limit: int = Query(50, ge=1, le=100, description="Items per page (max 100)")
):
    """Get batch items by serial number, shift, and operator with hierarchical role-based authorization"""
    # Require authentication
    user = require_auth(request)
    user_email = user.get('email') or user.get('name')
    user_id = user.get('id') or user.get('user_id')
    user_title = user.get('title', '').lower()
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Get user's area of operation from database
        user_area_of_operation = None
        if user_id:
            cursor.execute(
                "SELECT area_of_operation FROM users WHERE user_id = %s",
                (user_id,)
            )
            user_row = cursor.fetchone()
            if user_row:
                user_area_of_operation = user_row.get('area_of_operation') or ''
        
        # Sanitize inputs (basic validation)
        serial_no = batch_data.serial_no.strip() if batch_data.serial_no else None
        shift = batch_data.shift.strip() if batch_data.shift else None
        operator = batch_data.operator.strip() if batch_data.operator else None
        
        # Validate email format for operator if provided
        import re
        email_pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
        if operator and not re.match(email_pattern, operator):
            raise HTTPException(status_code=400, detail="Invalid operator email format")
        
        # Special handling for analysts and team leads: if operator not provided,
        # they can query all their assigned items (filtered by analyst/teamLead field, not operator)
        # For regular operators, operator is required
        if not operator:
            if user_title not in ['admin', 'super admin', 'analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead']:
                # Regular operators must provide operator
                raise HTTPException(
                    status_code=400, 
                    detail="Operator is required for your role. Analysts and Team Leads can omit operator to see their assigned items."
                )
        
        # Build base query with optional filters
        where_conditions = []
        query_params = []
        
        if serial_no:
            where_conditions.append("serial_no = %s")
            query_params.append(serial_no)
        
        if shift:
            where_conditions.append("shift = %s")
            query_params.append(shift)
        
        if operator:
            where_conditions.append("operator = %s")
            query_params.append(operator)
        
        # Build WHERE clause
        if not where_conditions:
            # If no filters, at least need role-based authorization
            where_clause = "1=1"  # Always true, will be filtered by authorization
        else:
            where_clause = " AND ".join(where_conditions)
        
        # Build base query
        base_query = f"""SELECT serial_no, shift, operator, scan_date, analyst, teamLead,
                     CASE 
                         WHEN tl_status = 1 AND analyst_status = 1 THEN 'Approved'
                         WHEN analyst_status = 0 THEN 'Pending Analyst'
                         WHEN tl_status = 0 AND analyst_status = 1 THEN 'Pending Team Lead'
                     END AS status 
              FROM data 
              WHERE {where_clause}"""
        
        # Apply role-based authorization filters
        authorization_conditions = []
        
        # Admin and Super Admin: Can see everything (no additional filter)
        if user_title in ['admin', 'super admin']:
            # No additional authorization check needed
            pass
        
        # Head of Analyst: Can see all items assigned to analysts in their area of operation
        elif user_title == 'head_of_analyst':
            if not user_area_of_operation:
                raise HTTPException(
                    status_code=403,
                    detail="Unauthorized: Area of operation not configured for your role"
                )
            # Get all analysts in the user's area of operation
            areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
            if not areas_list:
                raise HTTPException(
                    status_code=403,
                    detail="Unauthorized: No valid areas of operation configured"
                )
            # Get analysts whose area_of_operation overlaps with the head's areas
            # Build LIKE conditions for each area (handles comma-separated values)
            area_conditions = []
            analyst_params = []
            for area in areas_list:
                area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                analyst_params.extend([f'%{area}%', area])
            
            # Get analysts in the same area (assuming area_of_operation is stored in users table)
            analyst_query = f"SELECT email FROM users WHERE title = 'analyst' AND ({' OR '.join(area_conditions)})"
            cursor.execute(analyst_query, tuple(analyst_params))
            analyst_emails = [row['email'] for row in cursor.fetchall()]
            if analyst_emails:
                analyst_placeholders = ','.join(['%s'] * len(analyst_emails))
                authorization_conditions.append(f"analyst IN ({analyst_placeholders})")
                query_params.extend(analyst_emails)
            else:
                # No analysts in this area, return empty result
                return JSONResponse({
                    "success": False,
                    "message": "No items found"
                })
        
        # Analyst: Can only see items assigned to them
        elif user_title == 'analyst':
            authorization_conditions.append("analyst = %s")
            query_params.append(user_email)
        
        # Head of Team Lead: Can see all items assigned to team leads in their area of operation
        elif user_title == 'head_of_team_lead':
            if not user_area_of_operation:
                raise HTTPException(
                    status_code=403,
                    detail="Unauthorized: Area of operation not configured for your role"
                )
            # Get all team leads in the user's area of operation
            areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
            if not areas_list:
                raise HTTPException(
                    status_code=403,
                    detail="Unauthorized: No valid areas of operation configured"
                )
            # Get team leads whose area_of_operation overlaps with the head's areas
            # Build LIKE conditions for each area (handles comma-separated values)
            area_conditions = []
            tl_params = []
            for area in areas_list:
                area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                tl_params.extend([f'%{area}%', area])
            
            # Get team leads in the same area (assuming area_of_operation is stored in users table)
            tl_query = f"SELECT email FROM users WHERE title = 'team_lead' AND ({' OR '.join(area_conditions)})"
            cursor.execute(tl_query, tuple(tl_params))
            team_lead_emails = [row['email'] for row in cursor.fetchall()]
            if team_lead_emails:
                tl_placeholders = ','.join(['%s'] * len(team_lead_emails))
                authorization_conditions.append(f"teamLead IN ({tl_placeholders})")
                query_params.extend(team_lead_emails)
            else:
                # No team leads in this area, return empty result
                return JSONResponse({
                    "success": False,
                    "message": "No items found"
                })
        
        # Team Lead: Can only see items assigned to them
        elif user_title == 'team_lead':
            authorization_conditions.append("teamLead = %s")
            query_params.append(user_email)
        
        # Operators: Can only see their own data
        else:
            # Regular users/operators can only query their own operator data
            # Operator must be provided and must match their email
            if not operator:
                raise HTTPException(
                    status_code=400,
                    detail="Operator is required for your role"
                )
            if operator != user_email:
                raise HTTPException(
                    status_code=403,
                    detail="Unauthorized: You can only view your own batch items"
                )
        
        # Build final query with authorization conditions
        if authorization_conditions:
            final_query = f"{base_query} AND ({' OR '.join(authorization_conditions)}) ORDER BY scan_date"
        else:
            final_query = f"{base_query} ORDER BY scan_date"
        
        cursor.execute(final_query, tuple(query_params))
        all_items = cursor.fetchall()
        
        if not all_items:
            # Generic message to prevent information disclosure
            return JSONResponse({
                "success": False,
                "message": "No items found",
                "pagination": {
                    "current_page": page,
                    "per_page": limit,
                    "total_items": 0,
                    "total_pages": 0,
                    "has_next": False,
                    "has_previous": False
                }
            })
        
        # Apply pagination
        paginated = paginate_results(all_items, page, limit)
        
        return JSONResponse({
            "success": True,
            "items": paginated["items"],
            "pagination": paginated["pagination"]
        })
        
    except HTTPException:
        raise
    except Exception as e:
        # Don't expose database errors to client
        print(f"ERROR in get_batch_items: {str(e)}")
        return JSONResponse({
            "success": False,
            "message": "An error occurred while fetching batch items"
        }, status_code=500)
    finally:
        cursor.close()
        conn.close()

# Get multiple batch items endpoint (bulk query)
@router.post("/api/get_batch_items_bulk")
async def get_batch_items_bulk(
    request: Request, 
    bulk_request: BatchItemsBulkRequest,
    page: int = Query(1, ge=1, description="Page number (starts at 1)"),
    limit: int = Query(50, ge=1, le=100, description="Items per page (max 100)")
):
    """Get batch items for multiple serial numbers, shifts, and operators at once with hierarchical role-based authorization"""
    # Require authentication
    user = require_auth(request)
    user_email = user.get('email') or user.get('name')
    user_id = user.get('id') or user.get('user_id')
    user_title = user.get('title', '').lower()
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Get user's area of operation from database
        user_area_of_operation = None
        if user_id:
            cursor.execute(
                "SELECT area_of_operation FROM users WHERE user_id = %s",
                (user_id,)
            )
            user_row = cursor.fetchone()
            if user_row:
                user_area_of_operation = user_row.get('area_of_operation') or ''
        
        # Validate input
        if not bulk_request.items or len(bulk_request.items) == 0:
            raise HTTPException(status_code=400, detail="No batch items provided")
        
        # Limit the number of items to prevent abuse
        if len(bulk_request.items) > 100:
            raise HTTPException(status_code=400, detail="Maximum 100 batch items allowed per request")
        
        # Validate email format for all operators
        import re
        email_pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
        
        all_results = []
        
        # Process each batch item request
        for batch_data in bulk_request.items:
            # Sanitize inputs (serial_no, shift, and operator are all optional)
            serial_no = batch_data.serial_no.strip() if batch_data.serial_no else None
            shift = batch_data.shift.strip() if batch_data.shift else None
            operator = batch_data.operator.strip() if batch_data.operator else None
            
            # Validate email format for operator if provided
            if operator and not re.match(email_pattern, operator):
                continue  # Skip invalid emails
            
            # Special handling: analysts and team leads can omit operator to see their assigned items
            # For regular operators, operator is required
            if not operator:
                if user_title not in ['admin', 'super admin', 'analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead']:
                    continue  # Skip - operator required for this role
            
            # Apply role-based authorization checks
            authorization_passed = False
            
            # Admin and Super Admin: Can see everything
            if user_title in ['admin', 'super admin']:
                authorization_passed = True
            
            # Head of Analyst: Can see all items assigned to analysts in their area
            elif user_title == 'head_of_analyst':
                if not user_area_of_operation:
                    continue  # Skip if no area configured
                areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
                if areas_list:
                    area_conditions = []
                    analyst_params = []
                    for area in areas_list:
                        area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                        analyst_params.extend([f'%{area}%', area])
                    analyst_query = f"SELECT email FROM users WHERE title = 'analyst' AND ({' OR '.join(area_conditions)})"
                    cursor.execute(analyst_query, tuple(analyst_params))
                    analyst_emails = [row['email'] for row in cursor.fetchall()]
                    # Check if operator's analyst is in the list (we'll check this in the query)
                    authorization_passed = True  # Will filter in query
            
            # Analyst: Can only see items assigned to them
            elif user_title == 'analyst':
                # Will check analyst field in query
                authorization_passed = True
            
            # Head of Team Lead: Can see all items assigned to team leads in their area
            elif user_title == 'head_of_team_lead':
                if not user_area_of_operation:
                    continue  # Skip if no area configured
                areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
                if areas_list:
                    area_conditions = []
                    tl_params = []
                    for area in areas_list:
                        area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                        tl_params.extend([f'%{area}%', area])
                    tl_query = f"SELECT email FROM users WHERE title = 'team_lead' AND ({' OR '.join(area_conditions)})"
                    cursor.execute(tl_query, tuple(tl_params))
                    team_lead_emails = [row['email'] for row in cursor.fetchall()]
                    authorization_passed = True  # Will filter in query
            
            # Team Lead: Can only see items assigned to them
            elif user_title == 'team_lead':
                authorization_passed = True  # Will filter in query
            
            # Operators: Can only see their own data
            else:
                # Operator must be provided and must match their email
                if not operator:
                    continue  # Skip - operator required
                if operator != user_email:
                    continue  # Skip if not operator's own data
                authorization_passed = True
            
            if not authorization_passed:
                continue  # Skip unauthorized items
            
            # Build query with optional filters
            where_conditions = []
            query_params = []
            
            if serial_no:
                where_conditions.append("serial_no = %s")
                query_params.append(serial_no)
            
            if shift:
                where_conditions.append("shift = %s")
                query_params.append(shift)
            
            if operator:
                where_conditions.append("operator = %s")
                query_params.append(operator)
            
            # Build WHERE clause
            if not where_conditions:
                # If no filters, at least need role-based authorization
                where_clause = "1=1"  # Always true, will be filtered by authorization
            else:
                where_clause = " AND ".join(where_conditions)
            
            # Build base query
            base_query = f"""SELECT serial_no, shift, operator, scan_date, analyst, teamLead,
                         CASE 
                             WHEN tl_status = 1 AND analyst_status = 1 THEN 'Approved'
                             WHEN analyst_status = 0 THEN 'Pending Analyst'
                             WHEN tl_status = 0 AND analyst_status = 1 THEN 'Pending Team Lead'
                         END AS status 
                  FROM data 
                  WHERE {where_clause}"""
            
            authorization_conditions = []
            
            # Apply role-based filters
            if user_title == 'head_of_analyst':
                if user_area_of_operation:
                    areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
                    if areas_list:
                        area_conditions = []
                        analyst_params = []
                        for area in areas_list:
                            area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                            analyst_params.extend([f'%{area}%', area])
                        analyst_query = f"SELECT email FROM users WHERE title = 'analyst' AND ({' OR '.join(area_conditions)})"
                        cursor.execute(analyst_query, tuple(analyst_params))
                        analyst_emails = [row['email'] for row in cursor.fetchall()]
                        if analyst_emails:
                            analyst_placeholders = ','.join(['%s'] * len(analyst_emails))
                            authorization_conditions.append(f"analyst IN ({analyst_placeholders})")
                            query_params.extend(analyst_emails)
            
            elif user_title == 'analyst':
                authorization_conditions.append("analyst = %s")
                query_params.append(user_email)
            
            elif user_title == 'head_of_team_lead':
                if user_area_of_operation:
                    areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
                    if areas_list:
                        area_conditions = []
                        tl_params = []
                        for area in areas_list:
                            area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                            tl_params.extend([f'%{area}%', area])
                        tl_query = f"SELECT email FROM users WHERE title = 'team_lead' AND ({' OR '.join(area_conditions)})"
                        cursor.execute(tl_query, tuple(tl_params))
                        team_lead_emails = [row['email'] for row in cursor.fetchall()]
                        if team_lead_emails:
                            tl_placeholders = ','.join(['%s'] * len(team_lead_emails))
                            authorization_conditions.append(f"teamLead IN ({tl_placeholders})")
                            query_params.extend(team_lead_emails)
            
            elif user_title == 'team_lead':
                authorization_conditions.append("teamLead = %s")
                query_params.append(user_email)
            
            # Build final query
            if authorization_conditions:
                final_query = f"{base_query} AND ({' OR '.join(authorization_conditions)}) ORDER BY scan_date"
            else:
                final_query = f"{base_query} ORDER BY scan_date"
            
            cursor.execute(final_query, tuple(query_params))
            items = cursor.fetchall()
            
            if items:
                all_results.extend(items)
        
        if not all_results:
            return JSONResponse({
                "success": False,
                "message": "No items found",
                "pagination": {
                    "current_page": page,
                    "per_page": limit,
                    "total_items": 0,
                    "total_pages": 0,
                    "has_next": False,
                    "has_previous": False
                }
            })
        
        # Apply pagination
        paginated = paginate_results(all_results, page, limit)
        
        return JSONResponse({
            "success": True,
            "items": paginated["items"],
            "pagination": paginated["pagination"]
        })
        
    except HTTPException:
        raise
    except Exception as e:
        # Don't expose database errors to client
        print(f"ERROR in get_batch_items_bulk: {str(e)}")
        return JSONResponse({
            "success": False,
            "message": "An error occurred while fetching batch items"
        }, status_code=500)
    finally:
        cursor.close()
        conn.close()

# Send data processed endpoint
@router.get("/api/send_data_processed")
async def send_data_processed(request: Request, id: str):
    """Get processed data for batch with authentication and role-based authorization"""
    # Require authentication
    user = require_auth(request)
    user_email = user.get('email') or user.get('name')
    user_id = user.get('id') or user.get('user_id')
    user_title = user.get('title', '').lower()
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Validate batch number input
        if not id or not id.strip():
            raise HTTPException(status_code=400, detail="Batch number is required")
        
        batch_no = id.strip()
        
        # Get batch data
        cursor.execute("SELECT * FROM data WHERE batch_no = %s LIMIT 1", (batch_no,))
        first_row = cursor.fetchone()
        
        if not first_row:
            raise HTTPException(status_code=404, detail="Batch not found")
        
        row_operator = first_row.get('operator') or ''
        row_batch_no = first_row.get('batch_no') or ''
        row_analyst = first_row.get('analyst') or ''
        row_shift = first_row.get('shift') or ''
        product = first_row.get('product') or ''
        
        # Get user's area of operation from database for heads
        user_area_of_operation = None
        if user_id and user_title in ['head_of_analyst', 'head_of_team_lead']:
            cursor.execute(
                "SELECT area_of_operation FROM users WHERE user_id = %s",
                (user_id,)
            )
            user_row = cursor.fetchone()
            if user_row:
                user_area_of_operation = user_row.get('area_of_operation') or ''
        
        # Apply role-based authorization
        authorized = False
        
        # Admin and Super Admin: Can access all batches
        if user_title in ['admin', 'super admin']:
            authorized = True
        
        # Head of Analyst: Can access batches assigned to analysts in their area
        elif user_title == 'head_of_analyst':
            if not user_area_of_operation:
                raise HTTPException(
                    status_code=403,
                    detail="Unauthorized: Area of operation not configured for your role"
                )
            # Get all analysts in the user's area of operation
            areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
            if areas_list:
                area_conditions = []
                analyst_params = []
                for area in areas_list:
                    area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                    analyst_params.extend([f'%{area}%', area])
                analyst_query = f"SELECT email FROM users WHERE title = 'analyst' AND ({' OR '.join(area_conditions)})"
                cursor.execute(analyst_query, tuple(analyst_params))
                analyst_emails = [row['email'] for row in cursor.fetchall()]
                if row_analyst in analyst_emails:
                    authorized = True
        
        # Analyst: Can only access batches assigned to them
        elif user_title == 'analyst':
            # Check if analyst field matches user email (case-insensitive)
            if row_analyst and row_analyst.lower() == user_email.lower():
                authorized = True
        
        # Head of Team Lead: Can access batches assigned to team leads in their area
        elif user_title == 'head_of_team_lead':
            if not user_area_of_operation:
                raise HTTPException(
                    status_code=403,
                    detail="Unauthorized: Area of operation not configured for your role"
                )
            # Get all team leads in the user's area of operation
            areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
            if areas_list:
                area_conditions = []
                tl_params = []
                for area in areas_list:
                    area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                    tl_params.extend([f'%{area}%', area])
                tl_query = f"SELECT email FROM users WHERE title = 'team_lead' AND ({' OR '.join(area_conditions)})"
                cursor.execute(tl_query, tuple(tl_params))
                team_lead_emails = [row['email'] for row in cursor.fetchall()]
                # Check if batch's teamLead is in the list
                cursor.execute("SELECT teamLead FROM data WHERE batch_no = %s LIMIT 1", (batch_no,))
                batch_tl = cursor.fetchone()
                if batch_tl and batch_tl.get('teamLead') in team_lead_emails:
                    authorized = True
        
        # Team Lead: Can only access batches assigned to them
        elif user_title == 'team_lead':
            cursor.execute("SELECT teamLead FROM data WHERE batch_no = %s LIMIT 1", (batch_no,))
            batch_tl = cursor.fetchone()
            if batch_tl and batch_tl.get('teamLead') == user_email:
                authorized = True
        
        # Operators: Can only access their own batches
        else:
            if row_operator == user_email:
                authorized = True
        
        if not authorized:
            raise HTTPException(
                status_code=403,
                detail="Unauthorized: You do not have permission to access this batch"
            )
        
        now = datetime.now()
        date = now.strftime("%Y-%m-%d")
        time = now.strftime("%I:%M:%S %p")
        
        number = rand_number(4)
        number2 = rand_number(3)
        
        operator = f'NAME OF OPERATOR:- {row_operator}'
        batch_no_label = f'BATCH NO. {row_batch_no}'
        analyst = f'NAME OF ANALYST:- {row_analyst}'
        f_date = f'DATE:- {date}'
        week = f'WEEK:- {week_number(date)}'
        f_time = f'TIME:- {time}'
        shift = f'SHIFT:- {row_shift}'
        f_product = f'BATCH CONTROL SHEET- Product {product}'
        
        code = f'NRBPCFM{number}'
        
        div = []
        div.append(['', '', '', '', '', '', '', '', '', '', ''])
        div.append(['', '', '', f_product, '', '', '', '', '', 'Ref:', code])
        div.append(['', '', '', '', '', '', '', '', '', 'Issue no.', number2])
        div.append(['', '', '', '', '', '', '', '', '', 'Date:', date])
        div.append(['', '', '', '', '', '', '', '', '', 'Storage:', '3 yrs'])
        div.append(['', '', '', '', '', '', '', '', '', '', ''])
        div.append([operator, '', '', batch_no_label, '', '', '', '', '', '', ''])
        div.append([analyst, '', f_date, '', week, ''])
        div.append(['', f_time, shift, ''])
        div.append(['PHASE DESCRIPTION', '', '', '', '', '', '', '', '', '', ''])
        div.append(['S/NO', 'MDN', 'INGREDIENT/ MATERIAL NAME', 'BATCH NO.', 'DATE OF MNFG.', 'EXPIRY DATE', 'EXPECTED WEIGHT AS PER BOM  (KG)', 'TICK ON ORDER OF ADDITION', 'ACTUAL WEIGHT DISPENSED (KG)', 'Time', 'SUPPLIER'])
        
        cursor.execute("SELECT * FROM data WHERE batch_no = %s", (batch_no,))
        rows = cursor.fetchall()
        
        count = 0
        total = 0.0
        
        for row in rows:
            count += 1
            a = float(row.get('actual_weight') or 0)
            total += a
            b = float(row.get('expected_weight') or 0)
            dif = a - b
            div.append([
                count, 
                row.get('mdn') or '', 
                row.get('items') or '', 
                row.get('serial_no') or '', 
                '',
                '', 
                row.get('expected_weight') or '', 
                dif, 
                row.get('actual_weight') or '', 
                time, 
                ''
            ])
        
        div.append(['', '', '', '', '', 'Total QTY', total, '', '', '', ''])
        
        return JSONResponse(div)
        
    except HTTPException:
        raise
    except Exception as e:
        # Log the full error for debugging
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in send_data_processed: {error_detail}")
        print(f"  User: {user_email}, Title: {user_title}, Batch: {id}")
        raise HTTPException(status_code=500, detail=f"An error occurred while processing batch data: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# List items endpoint
@router.post("/api/list_items")
async def list_items(request: Request, list_data: ListItemsRequest):
    """Get ingredient list for a product with authentication and CSRF protection"""
    # Require authentication
    user = require_auth(request)
    user_id = user.get('id') or user.get('user_id')
    
    # Validate CSRF token - check both session and user-based storage
    session_id, session = get_session(request)
    csrf_valid = False
    
    # Check session-based CSRF token
    if list_data.csrf_token == session.get('csrf_token'):
        csrf_valid = True
    # Check user-based CSRF token (for JWT authentication)
    elif user_id in user_csrf_tokens:
        if list_data.csrf_token == user_csrf_tokens[user_id]:
            csrf_valid = True
    
    if not csrf_valid:
        raise HTTPException(status_code=403, detail="CSRF token validation failed")
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Validate and sanitize table name to prevent SQL injection
        table = list_data.data.strip().lower()
        
        # Validate table name format (only alphanumeric and underscores)
        import re
        if not re.match(r'^[a-zA-Z0-9_]+$', table):
            raise HTTPException(status_code=400, detail="Invalid product/table name format")
        
        # Verify table exists in products list (additional security)
        cursor.execute("SELECT name FROM products WHERE name = %s", (table,))
        if not cursor.fetchone():
            raise HTTPException(status_code=404, detail="Product table not found")
        
        # Query ingredients from the validated table
        cursor.execute(f"SELECT ingredient, expected_weight, weight_variance, to_be_measured FROM `{table}`")
        rows = cursor.fetchall()
        
        # Convert to JSON format
        ingredients = []
        for row in rows:
            ingredient_data = {
                "ingredient": str(row.get('ingredient') or ''),
                "expected_weight": str(row.get('expected_weight') or ''),
                "weight_variance": str(row.get('weight_variance') or ''),
                "to_be_measured": str(row.get('to_be_measured') or '')
            }
            ingredients.append(ingredient_data)
        
        return JSONResponse({
            "success": True,
            "product": table,
            "ingredients": ingredients,
            "count": len(ingredients)
        })
        
    except HTTPException:
        raise
    except Exception as e:
        # Log the error for debugging
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in list_items: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred while fetching ingredients: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Update row endpoint
@router.put("/api/update_row")
async def update_row(request: Request, update_data: UpdateRowRequest):
    """Update a product row with authentication, CSRF protection, and input validation"""
    # Require authentication
    user = require_auth(request)
    user_email = user.get('email') or user.get('name')
    user_id = user.get('id') or user.get('user_id')
    
    # Validate CSRF token - check both session and user-based storage
    session_id, session = get_session(request)
    csrf_valid = False
    
    # Check session-based CSRF token
    if update_data.csrf_token == session.get('csrf_token'):
        csrf_valid = True
    # Check user-based CSRF token (for JWT authentication)
    elif user_id in user_csrf_tokens:
        if update_data.csrf_token == user_csrf_tokens[user_id]:
            csrf_valid = True
    
    if not csrf_valid:
        raise HTTPException(status_code=403, detail="CSRF token validation failed")
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Validate and sanitize inputs
        table = update_data.table.strip().lower()
        column = update_data.column.strip()
        value = update_data.value.strip()
        row_id = update_data.id
        
        # Validate ID is positive
        if row_id <= 0:
            raise HTTPException(status_code=400, detail="Invalid row ID")
        
        # Validate table name format (only alphanumeric and underscores)
        import re
        if not re.match(r'^[a-zA-Z0-9_]+$', table):
            raise HTTPException(status_code=400, detail="Invalid table name format")
        
        # Validate column name format (only alphanumeric and underscores)
        if not re.match(r'^[a-zA-Z0-9_]+$', column):
            raise HTTPException(status_code=400, detail="Invalid column name format")
        
        # Whitelist of allowed columns that can be updated (security measure)
        allowed_columns = [
            'ingredient', 'expected_weight', 'weight_variance', 
            'to_be_measured', 'mdn', 'batch_no'
        ]
        
        if column not in allowed_columns:
            raise HTTPException(
                status_code=400, 
                detail=f"Column '{column}' is not allowed to be updated. Allowed columns: {', '.join(allowed_columns)}"
            )
        
        # Verify table exists in products list (additional security)
        cursor.execute("SELECT id, name FROM products WHERE name = %s", (table,))
        product = cursor.fetchone()
        if not product:
            # Check if table exists at all (for better error message)
            cursor.execute("SHOW TABLES LIKE %s", (table,))
            table_exists = cursor.fetchone()
            if table_exists:
                raise HTTPException(
                    status_code=404, 
                    detail=f"Product table '{table}' exists but is not registered in the products list. Please add it to the products table first."
                )
            else:
                raise HTTPException(
                    status_code=404, 
                    detail=f"Product table '{table}' not found. Please ensure the product exists and is registered in the products list."
                )
        
        # Verify the row exists before updating
        try:
            cursor.execute(f"SELECT id FROM `{table}` WHERE id = %s", (row_id,))
            row = cursor.fetchone()
            if not row:
                # Get available row IDs for better error message
                cursor.execute(f"SELECT id FROM `{table}` ORDER BY id LIMIT 10")
                available_ids = [str(r['id']) for r in cursor.fetchall()]
                available_ids_msg = f" Available row IDs: {', '.join(available_ids)}" if available_ids else " Table is empty."
                raise HTTPException(
                    status_code=404, 
                    detail=f"Row with ID {row_id} not found in product table '{table}'.{available_ids_msg}"
                )
        except Exception as e:
            # If table doesn't exist, MySQL will throw an error
            if "doesn't exist" in str(e).lower() or "unknown table" in str(e).lower():
                raise HTTPException(
                    status_code=404,
                    detail=f"Table '{table}' does not exist in the database"
                )
            raise
        
        # Perform the update with parameterized query
        # Note: MySQL doesn't support parameterized column/table names, but we've validated them above
        update_query = f"UPDATE `{table}` SET `{column}` = %s, `updated_by` = %s WHERE `id` = %s"
        cursor.execute(update_query, (value, user_email, row_id))
        conn.commit()
        
        if cursor.rowcount > 0:
            return JSONResponse({
                "success": True,
                "message": "Row updated successfully",
                "product_id": product['id'],
                "table": table,
                "column": column,
                "row_id": row_id,
                "updated_by": user_email
            })
        else:
            return JSONResponse({
                "success": False,
                "message": "No rows were updated"
            }, status_code=400)
        
    except HTTPException:
        raise
    except Exception as e:
        # Rollback on error
        conn.rollback()
        # Log the error for debugging
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in update_row: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred while updating the row: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Approve item endpoint
@router.post("/api/approve_item")
async def approve_item(request: Request, approve_data: ApproveItemRequest):
    """Approve or reject an item with authentication, CSRF protection, and proper authorization"""
    # Require authentication
    user = require_auth(request)
    current_user = user.get('email') or user.get('name')
    user_id = user.get('id') or user.get('user_id')
    user_title = user.get('title', '').lower()
    
    # Validate CSRF token - check both session and user-based storage
    session_id, session = get_session(request)
    csrf_valid = False
    
    # Check session-based CSRF token
    if approve_data.csrf_token == session.get('csrf_token'):
        csrf_valid = True
    # Check user-based CSRF token (for JWT authentication)
    elif user_id in user_csrf_tokens:
        if approve_data.csrf_token == user_csrf_tokens[user_id]:
            csrf_valid = True
    
    if not csrf_valid:
        raise HTTPException(status_code=403, detail="CSRF token validation failed")
    
    # Validate input
    if approve_data.action not in ['approve', 'reject']:
        raise HTTPException(status_code=400, detail="Invalid action. Must be 'approve' or 'reject'")
    
    if approve_data.type not in ['analyst', 'tl']:
        raise HTTPException(status_code=400, detail="Invalid type. Must be 'analyst' or 'tl'")
    
    # Validate user has the required role
    if approve_data.type == 'analyst' and user_title not in ['analyst', 'head_of_analyst']:
        raise HTTPException(
            status_code=403, 
            detail="Unauthorized: Only analysts and head of analysts can approve/reject as analyst"
        )
    
    if approve_data.type == 'tl' and user_title not in ['team_lead', 'head_of_team_lead']:
        raise HTTPException(
            status_code=403, 
            detail="Unauthorized: Only team leads and head of team leads can approve/reject as team lead"
        )
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Get user's area of operation from database for heads
        user_area_of_operation = None
        if user_id and user_title in ['head_of_analyst', 'head_of_team_lead']:
            cursor.execute(
                "SELECT area_of_operation FROM users WHERE user_id = %s",
                (user_id,)
            )
            user_row = cursor.fetchone()
            if user_row:
                user_area_of_operation = user_row.get('area_of_operation') or ''
        
        # Check if item exists and verify assignment from database (not from request)
        cursor.execute(
            "SELECT * FROM data WHERE serial_no = %s AND operator = %s AND shift = %s LIMIT 1",
            (approve_data.serial_no, approve_data.operator, approve_data.shift)
        )
        item = cursor.fetchone()
        
        if not item:
            raise HTTPException(status_code=404, detail="Item not found")
        
        # Block admins from approving/rejecting - they have view-only access
        if user_title in ['admin', 'super admin']:
            raise HTTPException(
                status_code=403,
                detail="Unauthorized: Admins have view-only access and cannot approve or reject items"
            )
        
        # Verify authorization from database (not from request data)
        authorized = False
        
        if approve_data.type == 'analyst':
            item_analyst = item.get('analyst') or ''
            
            # Regular Analyst: Can only approve items assigned to them
            if user_title == 'analyst':
                if item_analyst.lower() == current_user.lower():
                    authorized = True
                else:
                    raise HTTPException(
                        status_code=403, 
                        detail=f"Unauthorized: This item is assigned to analyst '{item_analyst}', not you"
                    )
            
            # Head of Analyst: Can approve items assigned to them OR any analyst in their area
            elif user_title == 'head_of_analyst':
                # If assigned to them directly
                if item_analyst.lower() == current_user.lower():
                    authorized = True
                else:
                    # Check if assigned analyst is in their area of operation
                    if not user_area_of_operation:
                        raise HTTPException(
                            status_code=403,
                            detail="Unauthorized: Area of operation not configured for your role"
                        )
                    
                    # Get all analysts in the head's area of operation
                    areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
                    if areas_list:
                        area_conditions = []
                        analyst_params = []
                        for area in areas_list:
                            area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                            analyst_params.extend([f'%{area}%', area])
                        analyst_query = f"SELECT email FROM users WHERE title = 'analyst' AND ({' OR '.join(area_conditions)})"
                        cursor.execute(analyst_query, tuple(analyst_params))
                        analyst_emails = [row['email'].lower() for row in cursor.fetchall()]
                        
                        if item_analyst.lower() in analyst_emails:
                            authorized = True
                        else:
                            raise HTTPException(
                                status_code=403,
                                detail=f"Unauthorized: Analyst '{item_analyst}' is not in your area of operation"
                            )
                    else:
                        raise HTTPException(
                            status_code=403,
                            detail="Unauthorized: No valid areas of operation configured"
                        )
        
        elif approve_data.type == 'tl':
            item_team_lead = item.get('teamLead') or ''
            
            # Verify that analyst has already approved (team lead should only review after analyst)
            if item.get('analyst_status') != 1:
                raise HTTPException(
                    status_code=403,
                    detail="Unauthorized: Item must be approved by analyst before team lead can review"
                )
            
            # Regular Team Lead: Can only approve items assigned to them
            if user_title == 'team_lead':
                if item_team_lead.lower() == current_user.lower():
                    authorized = True
                else:
                    raise HTTPException(
                        status_code=403, 
                        detail=f"Unauthorized: This item is assigned to team lead '{item_team_lead}', not you"
                    )
            
            # Head of Team Lead: Can approve items assigned to them OR any team lead in their area
            elif user_title == 'head_of_team_lead':
                # If assigned to them directly
                if item_team_lead.lower() == current_user.lower():
                    authorized = True
                else:
                    # Check if assigned team lead is in their area of operation
                    if not user_area_of_operation:
                        raise HTTPException(
                            status_code=403,
                            detail="Unauthorized: Area of operation not configured for your role"
                        )
                    
                    # Get all team leads in the head's area of operation
                    areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
                    if areas_list:
                        area_conditions = []
                        tl_params = []
                        for area in areas_list:
                            area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                            tl_params.extend([f'%{area}%', area])
                        tl_query = f"SELECT email FROM users WHERE title = 'team_lead' AND ({' OR '.join(area_conditions)})"
                        cursor.execute(tl_query, tuple(tl_params))
                        team_lead_emails = [row['email'].lower() for row in cursor.fetchall()]
                        
                        if item_team_lead.lower() in team_lead_emails:
                            authorized = True
                        else:
                            raise HTTPException(
                                status_code=403,
                                detail=f"Unauthorized: Team lead '{item_team_lead}' is not in your area of operation"
                            )
                    else:
                        raise HTTPException(
                            status_code=403,
                            detail="Unauthorized: No valid areas of operation configured"
                        )
        
        if not authorized:
            raise HTTPException(status_code=403, detail="Unauthorized: You do not have permission to approve/reject this item")
        
        current_date = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        status = 1 if approve_data.action == 'approve' else -1
        
        if approve_data.type == 'analyst':
            cursor.execute(
                """UPDATE data SET analyst_status = %s, analyst = %s, analyst_approval_date = %s 
                   WHERE serial_no = %s AND operator = %s AND shift = %s AND analyst = %s""",
                (status, current_user, current_date, approve_data.serial_no, approve_data.operator, 
                 approve_data.shift, current_user)
            )
        elif approve_data.type == 'tl':
            cursor.execute(
                """UPDATE data SET tl_status = %s, teamLead = %s, tl_approval_date = %s 
                   WHERE serial_no = %s AND operator = %s AND shift = %s AND teamLead = %s AND analyst_status = 1""",
                (status, current_user, current_date, approve_data.serial_no, approve_data.operator, 
                 approve_data.shift, current_user)
            )
        
        conn.commit()
        
        if cursor.rowcount > 0:
            action_verb = "approved" if approve_data.action == 'approve' else "rejected"
            return JSONResponse({
                "success": True, 
                "message": f"Item {action_verb} successfully",
                "action": approve_data.action,
                "type": approve_data.type,
                "serial_no": approve_data.serial_no
            })
        else:
            return JSONResponse({
                "success": False, 
                "message": "No rows updated. Item may have been modified or you may not have permission."
            }, status_code=400)
        
    except HTTPException:
        raise
    except Exception as e:
        conn.rollback()
        # Log the error for debugging
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in approve_item: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred while processing approval: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Get pending approvals endpoint
@router.get("/api/pending_approvals")
async def get_pending_approvals(
    request: Request,
    type: Optional[str] = Query(None, description="Filter by type: 'analyst', 'tl', or 'all' (default: based on user role)"),
    page: int = Query(1, ge=1, description="Page number (starts at 1)"),
    limit: int = Query(50, ge=1, le=100, description="Items per page (max 100)")
):
    """Get pending approvals based on user role with hierarchical authorization"""
    # Require authentication
    user = require_auth(request)
    user_email = (user.get('email') or user.get('name') or '').lower().strip()
    user_id = user.get('id') or user.get('user_id')
    user_title = (user.get('title') or '').lower().strip()
    
    # Validate user_email is not empty
    if not user_email:
        raise HTTPException(
            status_code=401,
            detail="Authentication failed: User email not found"
        )
    
    # Validate type parameter - only allow whitelisted values
    if type is not None:
        type = type.lower().strip()
        if type not in ['analyst', 'tl', 'all']:
            raise HTTPException(
                status_code=400,
                detail="Invalid type parameter. Must be 'analyst', 'tl', or 'all'"
            )
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Get user's area of operation from database for heads
        user_area_of_operation = None
        if user_id and user_title in ['head_of_analyst', 'head_of_team_lead']:
            cursor.execute(
                "SELECT area_of_operation FROM users WHERE user_id = %s",
                (user_id,)
            )
            user_row = cursor.fetchone()
            if user_row:
                user_area_of_operation = user_row.get('area_of_operation') or ''
        
        # Determine what to return based on user role and type parameter
        analyst_pending = []
        tl_pending = []
        
        # Admin and Super Admin: Can see all pending approvals
        if user_title in ['admin', 'super admin']:
            if type is None or type == 'all' or type == 'analyst':
                # Get all analyst pending approvals
                query = """SELECT id, factory, serial_no, shift, operator, scan_date, analyst, teamLead, product,
                          CASE 
                              WHEN tl_status = 1 AND analyst_status = 1 THEN 'Approved'
                              WHEN analyst_status = 0 THEN 'Pending Analyst'
                              WHEN tl_status = 0 AND analyst_status = 1 THEN 'Pending Team Lead'
                          END AS status
                          FROM data 
                          WHERE analyst_status = 0 
                          GROUP BY serial_no, shift, operator 
                          ORDER BY scan_date DESC"""
                cursor.execute(query)
                analyst_pending = cursor.fetchall()
            
            if type is None or type == 'all' or type == 'tl':
                # Get all team lead pending approvals (already approved by analyst)
                query = """SELECT id, factory, serial_no, shift, operator, scan_date, analyst, teamLead, product,
                          CASE 
                              WHEN tl_status = 1 AND analyst_status = 1 THEN 'Approved'
                              WHEN analyst_status = 0 THEN 'Pending Analyst'
                              WHEN tl_status = 0 AND analyst_status = 1 THEN 'Pending Team Lead'
                          END AS status
                          FROM data 
                          WHERE tl_status = 0 AND analyst_status = 1 
                          GROUP BY serial_no, shift, operator 
                          ORDER BY scan_date DESC"""
                cursor.execute(query)
                tl_pending = cursor.fetchall()
        
        # Head of Analyst: Can see pending approvals for analysts in their area
        elif user_title == 'head_of_analyst':
            if type is None or type == 'all' or type == 'analyst':
                if not user_area_of_operation:
                    raise HTTPException(
                        status_code=403,
                        detail="Unauthorized: Area of operation not configured for your role"
                    )
                
                # Get analysts in the head's area
                # Sanitize and validate area names (only alphanumeric, spaces, hyphens, underscores)
                areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
                # Validate area names to prevent injection
                valid_areas = []
                for area in areas_list:
                    # Only allow alphanumeric, spaces, hyphens, underscores, and commas
                    if re.match(r'^[a-zA-Z0-9\s\-_,]+$', area) and len(area) <= 100:
                        valid_areas.append(area)
                
                if valid_areas:
                    area_conditions = []
                    analyst_params = []
                    for area in valid_areas:
                        area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                        analyst_params.extend([f'%{area}%', area])
                    analyst_query = f"SELECT email FROM users WHERE title = 'analyst' AND ({' OR '.join(area_conditions)})"
                    cursor.execute(analyst_query, tuple(analyst_params))
                    analyst_emails = [row['email'].lower().strip() for row in cursor.fetchall() if row.get('email')]
                    
                    if analyst_emails:
                        # Use parameterized query with proper IN clause construction
                        analyst_placeholders = ','.join(['%s'] * len(analyst_emails))
                        query = f"""SELECT id, factory, serial_no, shift, operator, scan_date, analyst, teamLead, product,
                                  CASE 
                                      WHEN tl_status = 1 AND analyst_status = 1 THEN 'Approved'
                                      WHEN analyst_status = 0 THEN 'Pending Analyst'
                                      WHEN tl_status = 0 AND analyst_status = 1 THEN 'Pending Team Lead'
                                  END AS status
                                  FROM data 
                                  WHERE analyst_status = 0 AND LOWER(TRIM(analyst)) IN ({analyst_placeholders})
                                  GROUP BY serial_no, shift, operator 
                                  ORDER BY scan_date DESC"""
                        cursor.execute(query, tuple(analyst_emails))
                        analyst_pending = cursor.fetchall()
        
        # Analyst: Can only see their own pending approvals
        elif user_title == 'analyst':
            if type is None or type == 'all' or type == 'analyst':
                query = """SELECT id, factory, serial_no, shift, operator, scan_date, analyst, teamLead, product,
                          CASE 
                              WHEN tl_status = 1 AND analyst_status = 1 THEN 'Approved'
                              WHEN analyst_status = 0 THEN 'Pending Analyst'
                              WHEN tl_status = 0 AND analyst_status = 1 THEN 'Pending Team Lead'
                          END AS status
                          FROM data 
                          WHERE analyst_status = 0 AND LOWER(TRIM(analyst)) = %s 
                          GROUP BY serial_no, shift, operator 
                          ORDER BY scan_date DESC"""
                cursor.execute(query, (user_email,))
                analyst_pending = cursor.fetchall()
        
        # Head of Team Lead: Can see pending approvals for team leads in their area
        elif user_title == 'head_of_team_lead':
            if type is None or type == 'all' or type == 'tl':
                if not user_area_of_operation:
                    raise HTTPException(
                        status_code=403,
                        detail="Unauthorized: Area of operation not configured for your role"
                    )
                
                # Get team leads in the head's area
                # Sanitize and validate area names (only alphanumeric, spaces, hyphens, underscores)
                areas_list = [area.strip() for area in user_area_of_operation.split(',') if area.strip()]
                # Validate area names to prevent injection
                valid_areas = []
                for area in areas_list:
                    # Only allow alphanumeric, spaces, hyphens, underscores, and commas
                    if re.match(r'^[a-zA-Z0-9\s\-_,]+$', area) and len(area) <= 100:
                        valid_areas.append(area)
                
                if valid_areas:
                    area_conditions = []
                    tl_params = []
                    for area in valid_areas:
                        area_conditions.append("(area_of_operation LIKE %s OR area_of_operation = %s)")
                        tl_params.extend([f'%{area}%', area])
                    tl_query = f"SELECT email FROM users WHERE title = 'team_lead' AND ({' OR '.join(area_conditions)})"
                    cursor.execute(tl_query, tuple(tl_params))
                    team_lead_emails = [row['email'].lower().strip() for row in cursor.fetchall() if row.get('email')]
                    
                    if team_lead_emails:
                        # Use parameterized query with proper IN clause construction
                        tl_placeholders = ','.join(['%s'] * len(team_lead_emails))
                        query = f"""SELECT id, factory, serial_no, shift, operator, scan_date, analyst, teamLead, product,
                                  CASE 
                                      WHEN tl_status = 1 AND analyst_status = 1 THEN 'Approved'
                                      WHEN analyst_status = 0 THEN 'Pending Analyst'
                                      WHEN tl_status = 0 AND analyst_status = 1 THEN 'Pending Team Lead'
                                  END AS status
                                  FROM data 
                                  WHERE tl_status = 0 AND analyst_status = 1 AND LOWER(TRIM(teamLead)) IN ({tl_placeholders})
                                  GROUP BY serial_no, shift, operator 
                                  ORDER BY scan_date DESC"""
                        cursor.execute(query, tuple(team_lead_emails))
                        tl_pending = cursor.fetchall()
        
        # Team Lead: Can only see their own pending approvals
        elif user_title == 'team_lead':
            if type is None or type == 'all' or type == 'tl':
                query = """SELECT id, factory, serial_no, shift, operator, scan_date, analyst, teamLead, product,
                          CASE 
                              WHEN tl_status = 1 AND analyst_status = 1 THEN 'Approved'
                              WHEN analyst_status = 0 THEN 'Pending Analyst'
                              WHEN tl_status = 0 AND analyst_status = 1 THEN 'Pending Team Lead'
                          END AS status
                          FROM data 
                          WHERE tl_status = 0 AND analyst_status = 1 AND LOWER(TRIM(teamLead)) = %s 
                          GROUP BY serial_no, shift, operator 
                          ORDER BY scan_date DESC"""
                cursor.execute(query, (user_email,))
                tl_pending = cursor.fetchall()
        
        # Regular operators: Cannot see pending approvals
        else:
            raise HTTPException(
                status_code=403,
                detail="Unauthorized: Your role does not have access to pending approvals"
            )
        
        # Combine results based on type parameter
        if type == 'analyst':
            all_pending = analyst_pending
        elif type == 'tl':
            all_pending = tl_pending
        else:
            # Combine both (for 'all' or None)
            all_pending = analyst_pending + tl_pending
        
        # Apply pagination
        paginated = paginate_results(all_pending, page, limit)
        
        # Determine if user has view-only access (admins cannot approve/reject)
        view_only = user_title in ['admin', 'super admin']
        
        return JSONResponse({
            "success": True,
            "view_only": view_only,  # Indicates if user can only view (cannot approve/reject)
            "analyst_pending": analyst_pending if (type is None or type == 'all' or type == 'analyst') else [],
            "tl_pending": tl_pending if (type is None or type == 'all' or type == 'tl') else [],
            "items": paginated["items"],
            "pagination": paginated["pagination"],
            "summary": {
                "analyst_pending_count": len(analyst_pending),
                "tl_pending_count": len(tl_pending),
                "total_pending": len(all_pending)
            }
        })
        
    except HTTPException:
        raise
    except Exception as e:
        # Log the error for debugging
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in get_pending_approvals: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred while fetching pending approvals: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Get areas endpoint
@router.get("/api/get_areas")
async def get_areas(
    request: Request,
    page: int = Query(1, ge=1, description="Page number (starts at 1)"),
    limit: int = Query(50, ge=1, le=100, description="Items per page (max 100)")
):
    """Get all areas of operation from the areas_of_operation table"""
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Query from the new areas_of_operation table
        cursor.execute(
            "SELECT id, name, code, status, created_at, updated_at FROM areas_of_operation WHERE status = 1 ORDER BY name ASC"
        )
        all_areas = cursor.fetchall()
        
        # Convert datetime objects to strings for JSON serialization
        for area in all_areas:
            if area.get('created_at') and isinstance(area['created_at'], datetime):
                area['created_at'] = area['created_at'].isoformat()
            if area.get('updated_at') and isinstance(area['updated_at'], datetime):
                area['updated_at'] = area['updated_at'].isoformat()
        
        # Apply pagination
        paginated = paginate_results(all_areas, page, limit)
        
        return JSONResponse({
            "success": True,
            "areas": [item["name"] for item in paginated["items"]],  # Backward compatibility - return names
            "areas_full": paginated["items"],  # Full details with id, name, code
            "pagination": paginated["pagination"]
        })
    except mysql.connector.Error as db_error:
        # Database-specific errors
        import traceback
        error_detail = f"{str(db_error)}\n{traceback.format_exc()}"
        print(f"ERROR in get_areas (Database): {error_detail}")
        # Check if it's a "table doesn't exist" error
        if "doesn't exist" in str(db_error).lower() or "1146" in str(db_error):
            return JSONResponse({
                "success": False,
                "areas": [],
                "areas_full": [],
                "pagination": {
                    "current_page": 1,
                    "per_page": limit,
                    "total_items": 0,
                    "total_pages": 0,
                    "has_next": False,
                    "has_previous": False
                },
                "message": "Areas table not found. Please run the migration script.",
                "error": str(db_error)
            })
        else:
            raise HTTPException(status_code=500, detail=f"Database error: {str(db_error)}")
    except Exception as e:
        # Other errors
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in get_areas: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Get all areas with full details (for management)
@router.get("/api/areas")
async def get_all_areas(
    request: Request,
    page: int = Query(1, ge=1, description="Page number (starts at 1)"),
    limit: int = Query(50, ge=1, le=100, description="Items per page (max 100)"),
    include_inactive: bool = Query(False, description="Include inactive areas")
):
    """Get all areas of operation with full details (name, code, status, etc.)"""
    # Require authentication
    user = require_auth(request)
    user_title = (user.get('title') or '').lower()
    
    # Only admins can access this endpoint
    if user_title not in ['admin', 'super admin']:
        raise HTTPException(
            status_code=403,
            detail="Unauthorized: Only admins can view area details"
        )
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        if include_inactive:
            cursor.execute(
                "SELECT id, name, code, status, created_at, updated_at, created_by, updated_by FROM areas_of_operation ORDER BY name ASC"
            )
        else:
            cursor.execute(
                "SELECT id, name, code, status, created_at, updated_at, created_by, updated_by FROM areas_of_operation WHERE status = 1 ORDER BY name ASC"
            )
        all_areas = cursor.fetchall()
        
        # Convert datetime objects to strings for JSON serialization
        for area in all_areas:
            if area.get('created_at') and isinstance(area['created_at'], datetime):
                area['created_at'] = area['created_at'].isoformat()
            if area.get('updated_at') and isinstance(area['updated_at'], datetime):
                area['updated_at'] = area['updated_at'].isoformat()
        
        # Apply pagination
        paginated = paginate_results(all_areas, page, limit)
        
        return JSONResponse({
            "success": True,
            "areas": paginated["items"],
            "pagination": paginated["pagination"]
        })
    except Exception as e:
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in get_all_areas: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred while fetching areas: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Create new area
@router.post("/api/areas")
async def create_area(request: Request, area_data: AreaRequest):
    """Create a new area of operation (Admin only)"""
    # Require authentication
    user = require_auth(request)
    user_email = (user.get('email') or user.get('name') or '').lower().strip()
    user_title = (user.get('title') or '').lower()
    
    # Only admins can create areas
    if user_title not in ['admin', 'super admin']:
        raise HTTPException(
            status_code=403,
            detail="Unauthorized: Only admins can create areas"
        )
    
    # Validate input
    area_name = area_data.name.strip()
    area_code = area_data.code.strip().upper()  # Convert code to uppercase
    
    if not area_name or len(area_name) > 100:
        raise HTTPException(
            status_code=400,
            detail="Invalid area name. Must be between 1 and 100 characters"
        )
    
    if not area_code or len(area_code) > 50:
        raise HTTPException(
            status_code=400,
            detail="Invalid area code. Must be between 1 and 50 characters"
        )
    
    # Validate code format (alphanumeric and some special chars)
    if not re.match(r'^[a-zA-Z0-9\-_]+$', area_code):
        raise HTTPException(
            status_code=400,
            detail="Invalid area code format. Only alphanumeric characters, hyphens, and underscores allowed"
        )
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Check if name or code already exists
        cursor.execute(
            "SELECT id, name, code FROM areas_of_operation WHERE name = %s OR code = %s",
            (area_name, area_code)
        )
        existing = cursor.fetchone()
        
        if existing:
            if existing['name'].lower() == area_name.lower():
                raise HTTPException(
                    status_code=409,
                    detail=f"Area with name '{area_name}' already exists"
                )
            else:
                raise HTTPException(
                    status_code=409,
                    detail=f"Area with code '{area_code}' already exists"
                )
        
        # Insert new area
        cursor.execute(
            """INSERT INTO areas_of_operation (name, code, created_by, status) 
               VALUES (%s, %s, %s, 1)""",
            (area_name, area_code, user_email)
        )
        conn.commit()
        
        # Get the created area
        area_id = cursor.lastrowid
        cursor.execute(
            "SELECT id, name, code, status, created_at, updated_at, created_by FROM areas_of_operation WHERE id = %s",
            (area_id,)
        )
        new_area = cursor.fetchone()
        
        # Convert datetime objects to strings for JSON serialization
        if new_area.get('created_at') and isinstance(new_area['created_at'], datetime):
            new_area['created_at'] = new_area['created_at'].isoformat()
        if new_area.get('updated_at') and isinstance(new_area['updated_at'], datetime):
            new_area['updated_at'] = new_area['updated_at'].isoformat()
        
        return JSONResponse({
            "success": True,
            "message": "Area created successfully",
            "area": new_area
        }, status_code=201)
        
    except HTTPException:
        conn.rollback()
        raise
    except Exception as e:
        conn.rollback()
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in create_area: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred while creating area: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Update area
@router.put("/api/areas/{area_id}")
async def update_area(request: Request, area_id: int, area_data: AreaRequest):
    """Update an existing area of operation (Admin only)"""
    # Require authentication
    user = require_auth(request)
    user_email = (user.get('email') or user.get('name') or '').lower().strip()
    user_title = (user.get('title') or '').lower()
    
    # Only admins can update areas
    if user_title not in ['admin', 'super admin']:
        raise HTTPException(
            status_code=403,
            detail="Unauthorized: Only admins can update areas"
        )
    
    # Validate input
    area_name = area_data.name.strip()
    area_code = area_data.code.strip().upper()  # Convert code to uppercase
    
    if not area_name or len(area_name) > 100:
        raise HTTPException(
            status_code=400,
            detail="Invalid area name. Must be between 1 and 100 characters"
        )
    
    if not area_code or len(area_code) > 50:
        raise HTTPException(
            status_code=400,
            detail="Invalid area code. Must be between 1 and 50 characters"
        )
    
    # Validate code format
    if not re.match(r'^[a-zA-Z0-9\-_]+$', area_code):
        raise HTTPException(
            status_code=400,
            detail="Invalid area code format. Only alphanumeric characters, hyphens, and underscores allowed"
        )
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Check if area exists
        cursor.execute(
            "SELECT id, name, code FROM areas_of_operation WHERE id = %s",
            (area_id,)
        )
        existing_area = cursor.fetchone()
        
        if not existing_area:
            raise HTTPException(
                status_code=404,
                detail=f"Area with ID {area_id} not found"
            )
        
        # Check if name or code already exists (excluding current area)
        cursor.execute(
            "SELECT id, name, code FROM areas_of_operation WHERE (name = %s OR code = %s) AND id != %s",
            (area_name, area_code, area_id)
        )
        duplicate = cursor.fetchone()
        
        if duplicate:
            if duplicate['name'].lower() == area_name.lower():
                raise HTTPException(
                    status_code=409,
                    detail=f"Area with name '{area_name}' already exists"
                )
            else:
                raise HTTPException(
                    status_code=409,
                    detail=f"Area with code '{area_code}' already exists"
                )
        
        # Update area
        cursor.execute(
            """UPDATE areas_of_operation 
               SET name = %s, code = %s, updated_by = %s, updated_at = NOW()
               WHERE id = %s""",
            (area_name, area_code, user_email, area_id)
        )
        conn.commit()
        
        if cursor.rowcount == 0:
            raise HTTPException(
                status_code=400,
                detail="No changes made to the area"
            )
        
        # Get the updated area
        cursor.execute(
            "SELECT id, name, code, status, created_at, updated_at, created_by, updated_by FROM areas_of_operation WHERE id = %s",
            (area_id,)
        )
        updated_area = cursor.fetchone()
        
        # Convert datetime objects to strings for JSON serialization
        if updated_area.get('created_at') and isinstance(updated_area['created_at'], datetime):
            updated_area['created_at'] = updated_area['created_at'].isoformat()
        if updated_area.get('updated_at') and isinstance(updated_area['updated_at'], datetime):
            updated_area['updated_at'] = updated_area['updated_at'].isoformat()
        
        return JSONResponse({
            "success": True,
            "message": "Area updated successfully",
            "area": updated_area
        })
        
    except HTTPException:
        conn.rollback()
        raise
    except Exception as e:
        conn.rollback()
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in update_area: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred while updating area: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Delete/Deactivate area
@router.delete("/api/areas/{area_id}")
async def delete_area(request: Request, area_id: int, permanent: bool = Query(False, description="Permanently delete (default: deactivate)")):
    """Delete or deactivate an area of operation (Admin only)"""
    # Require authentication
    user = require_auth(request)
    user_email = (user.get('email') or user.get('name') or '').lower().strip()
    user_title = (user.get('title') or '').lower()
    
    # Only admins can delete areas
    if user_title not in ['admin', 'super admin']:
        raise HTTPException(
            status_code=403,
            detail="Unauthorized: Only admins can delete areas"
        )
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Check if area exists
        cursor.execute(
            "SELECT id, name, code, status FROM areas_of_operation WHERE id = %s",
            (area_id,)
        )
        area = cursor.fetchone()
        
        if not area:
            raise HTTPException(
                status_code=404,
                detail=f"Area with ID {area_id} not found"
            )
        
        if permanent:
            # Permanently delete
            cursor.execute(
                "DELETE FROM areas_of_operation WHERE id = %s",
                (area_id,)
            )
            message = "Area permanently deleted"
        else:
            # Soft delete (deactivate)
            cursor.execute(
                "UPDATE areas_of_operation SET status = 0, updated_by = %s, updated_at = NOW() WHERE id = %s",
                (user_email, area_id)
            )
            message = "Area deactivated successfully"
        
        conn.commit()
        
        return JSONResponse({
            "success": True,
            "message": message,
            "area_id": area_id
        })
        
    except HTTPException:
        conn.rollback()
        raise
    except Exception as e:
        conn.rollback()
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in delete_area: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred while deleting area: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Get products by hub/section and category endpoint
@router.get("/api/products_hub")
async def get_products_by_hub(
    request: Request,
    hub: str,
    category: Optional[str] = Query(None, description="Optional category filter"),
    page: int = Query(1, ge=1, description="Page number (starts at 1)"),
    limit: int = Query(50, ge=1, le=100, description="Items per page (max 100)")
):
    """Get products for a given hub/section, optionally filtered by category"""
    # Require authentication
    user = require_auth(request)
    
    # Validate and sanitize input
    section = hub.strip()
    if not section:
        raise HTTPException(
            status_code=400,
            detail="Invalid hub input. Hub name cannot be empty"
        )
    
    # Validate section length to prevent injection
    if len(section) > 100:
        raise HTTPException(
            status_code=400,
            detail="Invalid hub input. Hub name too long (max 100 characters)"
        )
    
    # Validate section format (alphanumeric, spaces, hyphens, underscores)
    if not re.match(r'^[a-zA-Z0-9\s\-_]+$', section):
        raise HTTPException(
            status_code=400,
            detail="Invalid hub format. Only alphanumeric characters, spaces, hyphens, and underscores allowed"
        )
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Validate that the hub exists in areas_of_operation table
        cursor.execute(
            "SELECT id, name, code, status FROM areas_of_operation WHERE name = %s OR code = %s LIMIT 1",
            (section, section)
        )
        area = cursor.fetchone()
        
        if not area:
            raise HTTPException(
                status_code=404,
                detail=f"Hub/section '{section}' not found in areas of operation. Please use a valid area name or code."
            )
        
        if area.get('status') != 1:
            raise HTTPException(
                status_code=400,
                detail=f"Hub/section '{section}' is inactive"
            )
        
        # Use the area name from the database for consistency
        validated_section = area.get('name')
        
        # Validate category if provided
        if category:
            category = category.strip()
            if len(category) > 100:
                raise HTTPException(
                    status_code=400,
                    detail="Invalid category. Category name too long (max 100 characters)"
                )
            if not re.match(r'^[a-zA-Z0-9\s\-_]+$', category):
                raise HTTPException(
                    status_code=400,
                    detail="Invalid category format. Only alphanumeric characters, spaces, hyphens, and underscores allowed"
                )
        
        # Query products with parameterized queries
        if category:
            cursor.execute(
                "SELECT id, name, section, category, update_by, date_updated FROM products WHERE section = %s AND category = %s ORDER BY name ASC",
                (validated_section, category)
            )
        else:
            cursor.execute(
                "SELECT id, name, section, category, update_by, date_updated FROM products WHERE section = %s ORDER BY name ASC",
                (validated_section,)
            )
        
        all_products = cursor.fetchall()
        
        # Convert datetime objects to strings for JSON serialization
        for product in all_products:
            if product.get('date_updated') and isinstance(product['date_updated'], datetime):
                product['date_updated'] = product['date_updated'].isoformat()
        
        # Apply pagination
        paginated = paginate_results(all_products, page, limit)
        
        return JSONResponse({
            "success": True,
            "hub": validated_section,
            "hub_code": area.get('code'),
            "category": category if category else None,
            "products": paginated["items"],
            "pagination": paginated["pagination"],
            "summary": {
                "total_products": len(all_products),
                "filtered_by_category": category is not None
            }
        })
        
    except HTTPException:
        raise
    except mysql.connector.Error as db_error:
        import traceback
        error_detail = f"{str(db_error)}\n{traceback.format_exc()}"
        print(f"ERROR in get_products_by_hub (Database): {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"Database error while fetching products: {str(db_error)}"
        )
    except Exception as e:
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in get_products_by_hub: {error_detail}")
        print(f"  Hub: {hub}, Category: {category}")
        raise HTTPException(
            status_code=500,
            detail=f"An error occurred while fetching products: {str(e)}"
        )
    finally:
        cursor.close()
        conn.close()

# Session endpoint
@router.get("/api/session")
async def set_session(
    request: Request,
    data: str = Query(..., description="Batch number to store in session")
):
    """Set session batch number with validation and authentication"""
    # Require authentication
    user = require_auth(request)
    user_email = (user.get('email') or user.get('name') or '').lower().strip()
    
    # Validate and sanitize input
    batch_num = data.strip() if data else ''
    
    if not batch_num:
        raise HTTPException(
            status_code=400,
            detail="Batch number cannot be empty"
        )
    
    # Validate batch number length (reasonable limit)
    if len(batch_num) > 200:
        raise HTTPException(
            status_code=400,
            detail="Batch number too long (max 200 characters)"
        )
    
    # Validate batch number format (alphanumeric, spaces, hyphens, underscores, dots, slashes)
    # This allows common batch number formats like "BATCH-001", "2026/01/15", etc.
    if not re.match(r'^[a-zA-Z0-9\s\-_./]+$', batch_num):
        raise HTTPException(
            status_code=400,
            detail="Invalid batch number format. Only alphanumeric characters, spaces, hyphens, underscores, dots, and slashes allowed"
        )
    
    try:
        # Get or create session
        session_id, session = get_session(request)
        
        # Store batch number in session
        session['batch_num'] = batch_num
        session['batch_num_set_by'] = user_email
        session['batch_num_set_at'] = datetime.now().isoformat()
        
        return JSONResponse({
            "success": True,
            "message": "Batch number set successfully",
            "batch_num": batch_num,
            "set_by": user_email,
            "set_at": session['batch_num_set_at']
        })
        
    except Exception as e:
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in set_session: {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"An error occurred while setting session batch number: {str(e)}"
        )

# Get session batch number endpoint
@router.get("/api/session/batch_num")
async def get_session_batch_num(request: Request):
    """Get the current session batch number"""
    # Require authentication
    user = require_auth(request)
    
    try:
        # Get session
        session_id, session = get_session(request)
        
        batch_num = session.get('batch_num')
        
        if batch_num:
            return JSONResponse({
                "success": True,
                "batch_num": batch_num,
                "set_by": session.get('batch_num_set_by'),
                "set_at": session.get('batch_num_set_at')
            })
        else:
            return JSONResponse({
                "success": True,
                "batch_num": None,
                "message": "No batch number set in session"
            })
            
    except Exception as e:
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in get_session_batch_num: {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"An error occurred while retrieving session batch number: {str(e)}"
        )

# Get CSRF token endpoint
@router.get("/api/csrf_token")
async def get_csrf_token(request: Request):
    """Get CSRF token for current session (JWT or session-based)"""
    user = require_auth(request)
    user_id = user.get('id') or user.get('user_id')
    
    print(f"DEBUG: Getting CSRF token for user_id: {user_id}")
    
    # Check if we have a stored CSRF token for this user
    if user_id in user_csrf_tokens:
        csrf_token = user_csrf_tokens[user_id]
        print(f"DEBUG: Returning existing CSRF token: {csrf_token[:20]}...")
    else:
        # Get or create session
        session_id, session = get_session(request)
        # Generate a CSRF token and store it in session
        csrf_token = secrets.token_urlsafe(32)
        session['csrf_token'] = csrf_token
        # Store for user-based lookup
        user_csrf_tokens[user_id] = csrf_token
        print(f"DEBUG: Generated new CSRF token for user {user_id}: {csrf_token[:20]}...")
    
    return JSONResponse({"csrf_token": csrf_token})

# User management endpoints
@router.post("/api/add_user")
async def add_user(request: Request, user_data: AddUserRequest):
    """Add a new user with validation, password hashing, and welcome email"""
    # Require authentication
    user = require_auth(request)
    user_title = (user.get('title') or '').lower()
    created_by = (user.get('email') or user.get('name') or '').lower().strip()
    
    # Only admins and super admins can add users
    if user_title not in ['admin', 'super admin']:
        raise HTTPException(
            status_code=403,
            detail="Unauthorized: Only admins can add new users"
        )
    
    # Validate and sanitize inputs
    name = user_data.name.strip()
    email = user_data.email.strip().lower()
    password = user_data.password.strip()
    title = user_data.title.strip().lower()
    shift = user_data.shift.strip()
    areas = user_data.areas.strip() if user_data.areas else ''
    
    # Validate name
    if not name or len(name) < 2:
        raise HTTPException(
            status_code=400,
            detail="Name must be at least 2 characters long"
        )
    if len(name) > 100:
        raise HTTPException(
            status_code=400,
            detail="Name too long (max 100 characters)"
        )
    if not re.match(r'^[a-zA-Z\s\-\.]+$', name):
        raise HTTPException(
            status_code=400,
            detail="Invalid name format. Only letters, spaces, hyphens, and dots allowed"
        )
    
    # Validate email
    if not email:
        raise HTTPException(
            status_code=400,
            detail="Email is required"
        )
    # Basic email format validation
    email_pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
    if not re.match(email_pattern, email):
        raise HTTPException(
            status_code=400,
            detail="Invalid email format"
        )
    if len(email) > 100:
        raise HTTPException(
            status_code=400,
            detail="Email too long (max 100 characters)"
        )
    
    # Validate password
    if not password or len(password) < 8:
        raise HTTPException(
            status_code=400,
            detail="Password must be at least 8 characters long"
        )
    if len(password) > 128:
        raise HTTPException(
            status_code=400,
            detail="Password too long (max 128 characters)"
        )
    
    # Validate title/role
    valid_titles = ['admin', 'super admin', 'user', 'analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead', 'operator']
    if title not in valid_titles:
        raise HTTPException(
            status_code=400,
            detail=f"Invalid role. Must be one of: {', '.join(valid_titles)}"
        )
    
    # Validate shift
    valid_shifts = ['A', 'B', 'shift A', 'shift B', 'Day Shift', 'Night Shift']
    if shift not in valid_shifts and shift.upper() not in ['A', 'B']:
        raise HTTPException(
            status_code=400,
            detail="Invalid shift. Must be 'A', 'B', 'shift A', 'shift B', 'Day Shift', or 'Night Shift'"
        )
    
    # Normalize shift
    if shift.upper() in ['A', 'SHIFT A']:
        shift = 'shift A'
    elif shift.upper() in ['B', 'SHIFT B']:
        shift = 'shift B'
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Check if email already exists
        cursor.execute("SELECT email FROM users WHERE email = %s", (email,))
        existing_user = cursor.fetchone()
        
        if existing_user:
            raise HTTPException(
                status_code=409,
                detail=f"Email address '{email}' already exists"
            )
        
        # Validate areas if provided
        validated_areas = ''
        if areas:
            # Split areas by comma and validate each against areas_of_operation table
            area_list = [area.strip() for area in areas.split(',') if area.strip()]
            if area_list:
                # Get valid areas from database
                area_placeholders = ','.join(['%s'] * len(area_list))
                cursor.execute(
                    f"SELECT name FROM areas_of_operation WHERE name IN ({area_placeholders}) AND status = 1",
                    tuple(area_list)
                )
                valid_areas = [row['name'] for row in cursor.fetchall()]
                
                # Check if all provided areas are valid
                invalid_areas = [area for area in area_list if area not in valid_areas]
                if invalid_areas:
                    raise HTTPException(
                        status_code=400,
                        detail=f"Invalid areas: {', '.join(invalid_areas)}. Please use valid area names from the areas of operation."
                    )
                
                validated_areas = ','.join(valid_areas)
        
        # Generate user ID
        import uuid
        user_id = str(uuid.uuid4())
        
        # Hash password using bcrypt
        password_hash = bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        
        # Insert new user
        cursor.execute(
            """INSERT INTO `users`(`email`, `password`, `user_id`, `imglink`, `title`, `name`, `shift`, `area_of_operation`, `status`) 
               VALUES (%s, %s, %s, %s, %s, %s, %s, %s, 1)""",
            (email, password_hash, user_id, 'images/admin.png', title, name, shift, validated_areas)
        )
        conn.commit()
        
        if cursor.rowcount == 0:
            raise HTTPException(
                status_code=500,
                detail="Failed to create user. No rows inserted."
            )
        
        # Create one-time credential viewing token (don't send plain text password)
        credential_token_result = None
        credential_view_url = None
        try:
            from backend.password_reset import create_credential_token
            credential_token_result = create_credential_token(user_id, email, password, expiry_hours=24)
            
            if credential_token_result.get('success'):
                # Generate secure credential viewing URL
                import os
                # Debug: Check what environment variables are available
                base_url = os.getenv('BASE_URL') or os.getenv('FRONTEND_URL')
                print(f"DEBUG: BASE_URL from env: {os.getenv('BASE_URL')}")
                print(f"DEBUG: FRONTEND_URL from env: {os.getenv('FRONTEND_URL')}")
                print(f"DEBUG: Selected base_url: {base_url}")
                
                # If not in env, try to get from request
                if not base_url:
                    try:
                        if hasattr(request, 'url') and request.url:
                            scheme = getattr(request.url, 'scheme', 'http')
                            netloc = getattr(request.url, 'netloc', 'localhost')
                            
                            # Validate scheme and netloc are not "none"
                            if scheme and scheme.lower() != 'none' and netloc and netloc.lower() != 'none':
                                base_url = f"{scheme}://{netloc}"
                                # Remove port 8000 if present (FastAPI port)
                                if ':8000' in base_url:
                                    base_url = base_url.replace(':8000', '')
                            else:
                                # Invalid URL components, use fallback
                                base_url = 'http://localhost'
                        else:
                            # No request URL available, use fallback
                            base_url = 'http://localhost'
                    except Exception as url_error:
                        # Error getting URL from request, use fallback
                        print(f"WARNING: Could not get URL from request: {url_error}")
                        base_url = 'http://localhost'
                
                # Final fallback if base_url is still empty or invalid
                if not base_url or base_url.lower() in ['none', 'http://none', 'https://none']:
                    base_url = 'http://localhost'
                
                # Ensure base_url doesn't end with /
                base_url = base_url.rstrip('/')
                credential_view_url = f"{base_url}/blue/view_credentials.php?token={credential_token_result['token']}"
                print(f"DEBUG: Created credential token for user {email}")
                print(f"DEBUG: Base URL: {base_url}")
                print(f"DEBUG: Credential view URL: {credential_view_url}")
            else:
                print(f"WARNING: Failed to create credential token: {credential_token_result.get('message')}")
        except Exception as token_error:
            print(f"WARNING: Failed to create credential token: {str(token_error)}")
            # Continue without token (fallback to old method)
        
        # Send welcome email with secure link (don't fail user creation if email fails)
        email_result = None
        try:
            from backend.email_service import send_welcome_email
            login_url = os.getenv('LOGIN_URL', 'http://localhost/blue/index.php')
            # Use secure credential link if available, otherwise fallback to plain text (not recommended)
            email_result = send_welcome_email(
                email, name, 
                password if not credential_view_url else None,  # Don't send password if we have secure link
                title, shift, login_url, 
                credential_view_url=credential_view_url  # Pass secure link
            )
            
            if not email_result.get('success'):
                print(f"WARNING: Welcome email failed for new user {email}: {email_result.get('message')}")
        except Exception as email_error:
            print(f"WARNING: Failed to send welcome email to {email}: {str(email_error)}")
            # Don't fail user creation if email fails
        
        # Get the created user
        cursor.execute(
            "SELECT user_id, email, name, title, shift, area_of_operation, status FROM users WHERE user_id = %s",
            (user_id,)
        )
        new_user = cursor.fetchone()
        
        return JSONResponse({
            "success": True,
            "message": "User created successfully",
            "user": {
                "user_id": new_user['user_id'],
                "email": new_user['email'],
                "name": new_user['name'],
                "title": new_user['title'],
                "shift": new_user['shift'],
                "area_of_operation": new_user.get('area_of_operation', ''),
                "status": new_user.get('status', 1)
            },
            "email_sent": email_result.get('success', False) if email_result else False,
            "email_message": email_result.get('message', '') if email_result else ''
        }, status_code=201)
        
    except HTTPException:
        conn.rollback()
        raise
    except mysql.connector.Error as db_error:
        conn.rollback()
        import traceback
        error_detail = f"{str(db_error)}\n{traceback.format_exc()}"
        print(f"ERROR in add_user (Database): {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"Database error while creating user: {str(db_error)}"
        )
    except Exception as e:
        conn.rollback()
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in add_user: {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"An error occurred while creating user: {str(e)}"
        )
    finally:
        cursor.close()
        conn.close()

@router.put("/api/update_user/{user_id}")
async def update_user(request: Request, user_id: str, user_data: UpdateUserRequest):
    """Update user information with validation, password hashing, and proper authorization"""
    # Require authentication
    user = require_auth(request)
    user_title = (user.get('title') or '').lower()
    updated_by = (user.get('email') or user.get('name') or '').lower().strip()
    
    # Only admins and super admins can update users
    if user_title not in ['admin', 'super admin']:
        raise HTTPException(
            status_code=403,
            detail="Unauthorized: Only admins can update users"
        )
    
    # Validate and sanitize inputs
    name = user_data.name.strip()
    title = user_data.title.strip().lower()
    shift = user_data.shift.strip()
    status = user_data.status if user_data.status is not None else 1
    areas = user_data.areas.strip() if user_data.areas else ''
    change_password = user_data.change_password or False
    password = user_data.password.strip() if user_data.password else None
    
    # Validate name
    if not name or len(name) < 2:
        raise HTTPException(
            status_code=400,
            detail="Name must be at least 2 characters long"
        )
    if len(name) > 100:
        raise HTTPException(
            status_code=400,
            detail="Name too long (max 100 characters)"
        )
    if not re.match(r'^[a-zA-Z\s\-\.]+$', name):
        raise HTTPException(
            status_code=400,
            detail="Invalid name format. Only letters, spaces, hyphens, and dots allowed"
        )
    
    # Validate title/role
    valid_titles = ['admin', 'super admin', 'user', 'analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead', 'operator']
    if title not in valid_titles:
        raise HTTPException(
            status_code=400,
            detail=f"Invalid role. Must be one of: {', '.join(valid_titles)}"
        )
    
    # Validate shift
    valid_shifts = ['A', 'B', 'shift A', 'shift B', 'Day Shift', 'Night Shift']
    if shift not in valid_shifts and shift.upper() not in ['A', 'B']:
        raise HTTPException(
            status_code=400,
            detail="Invalid shift. Must be 'A', 'B', 'shift A', 'shift B', 'Day Shift', or 'Night Shift'"
        )
    
    # Normalize shift
    if shift.upper() in ['A', 'SHIFT A']:
        shift = 'shift A'
    elif shift.upper() in ['B', 'SHIFT B']:
        shift = 'shift B'
    
    # Validate status
    if status not in [0, 1]:
        raise HTTPException(
            status_code=400,
            detail="Invalid status. Must be 0 (inactive) or 1 (active)"
        )
    
    # Validate password if changing
    if change_password:
        if not password:
            raise HTTPException(
                status_code=400,
                detail="Password is required when change_password is true"
            )
        if len(password) < 8:
            raise HTTPException(
                status_code=400,
                detail="Password must be at least 8 characters long"
            )
        if len(password) > 128:
            raise HTTPException(
                status_code=400,
                detail="Password too long (max 128 characters)"
            )
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Check if user exists
        cursor.execute("SELECT user_id, email, title FROM users WHERE user_id = %s", (user_id,))
        existing_user = cursor.fetchone()
        
        if not existing_user:
            raise HTTPException(
                status_code=404,
                detail=f"User with ID '{user_id}' not found"
            )
        
        # Prevent updating admin users (only super admin can do this)
        if existing_user.get('title', '').lower() == 'admin' and user_title != 'super admin':
            raise HTTPException(
                status_code=403,
                detail="Unauthorized: Only super admin can update admin users"
            )
        
        # Prevent changing super admin role
        if existing_user.get('title', '').lower() == 'super admin' and title != 'super admin':
            raise HTTPException(
                status_code=403,
                detail="Unauthorized: Cannot change super admin role"
            )
        
        # Validate areas if provided
        validated_areas = ''
        if areas:
            # Split areas by comma and validate each against areas_of_operation table
            area_list = [area.strip() for area in areas.split(',') if area.strip()]
            if area_list:
                # Get valid areas from database
                area_placeholders = ','.join(['%s'] * len(area_list))
                cursor.execute(
                    f"SELECT name FROM areas_of_operation WHERE name IN ({area_placeholders}) AND status = 1",
                    tuple(area_list)
                )
                valid_areas = [row['name'] for row in cursor.fetchall()]
                
                # Check if all provided areas are valid
                invalid_areas = [area for area in area_list if area not in valid_areas]
                if invalid_areas:
                    raise HTTPException(
                        status_code=400,
                        detail=f"Invalid areas: {', '.join(invalid_areas)}. Please use valid area names from the areas of operation."
                    )
                
                validated_areas = ','.join(valid_areas)
        
        # Build update query based on whether password is being changed
        if change_password and password:
            # Hash password using bcrypt
            password_hash = bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
            
            cursor.execute(
                """UPDATE `users` SET `name` = %s, `title` = %s, `password` = %s, 
                   `shift` = %s, `status` = %s, `area_of_operation` = %s 
                   WHERE `user_id` = %s""",
                (name, title, password_hash, shift, status, validated_areas, user_id)
            )
        else:
            # Update without changing password
            cursor.execute(
                """UPDATE `users` SET `name` = %s, `title` = %s, 
                   `shift` = %s, `status` = %s, `area_of_operation` = %s 
                   WHERE `user_id` = %s""",
                (name, title, shift, status, validated_areas, user_id)
            )
        
        conn.commit()
        
        if cursor.rowcount == 0:
            raise HTTPException(
                status_code=400,
                detail="No changes made to the user. User may not exist or data is unchanged."
            )
        
        # Get the updated user
        cursor.execute(
            "SELECT user_id, email, name, title, shift, area_of_operation, status FROM users WHERE user_id = %s",
            (user_id,)
        )
        updated_user = cursor.fetchone()
        
        return JSONResponse({
            "success": True,
            "message": "User updated successfully",
            "user": {
                "user_id": updated_user['user_id'],
                "email": updated_user['email'],
                "name": updated_user['name'],
                "title": updated_user['title'],
                "shift": updated_user['shift'],
                "area_of_operation": updated_user.get('area_of_operation', ''),
                "status": updated_user.get('status', 1)
            },
            "password_changed": change_password and password is not None
        })
        
    except HTTPException:
        conn.rollback()
        raise
    except mysql.connector.Error as db_error:
        conn.rollback()
        import traceback
        error_detail = f"{str(db_error)}\n{traceback.format_exc()}"
        print(f"ERROR in update_user (Database): {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"Database error while updating user: {str(db_error)}"
        )
    except Exception as e:
        conn.rollback()
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in update_user: {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"An error occurred while updating user: {str(e)}"
        )
    finally:
        cursor.close()
        conn.close()

@router.get("/api/delete_user")
async def delete_user(request: Request, id: str):
    """Delete a user"""
    session = require_auth(request)
    
    conn = get_db_connection()
    cursor = conn.cursor()
    
    try:
        cursor.execute("DELETE FROM `users` WHERE `user_id` = %s AND `title` != 'admin'", (id,))
        conn.commit()
        
        if cursor.rowcount > 0:
            return JSONResponse({"status": "done"})
        else:
            return JSONResponse({"status": "failed"})
    except Exception as e:
        conn.rollback()
        raise HTTPException(status_code=500, detail=str(e))
    finally:
        cursor.close()
        conn.close()

# Legacy endpoints for backward compatibility (deprecated - use /api/users instead)
@router.get("/api/edit_users", operation_id="edit_users_legacy_redirect")
async def edit_users_legacy(request: Request, id: str):
    """Legacy endpoint - use GET /api/users?user_id={id} instead"""
    return await get_users(request, user_id=id)

@router.get("/api/current_user", operation_id="get_current_user_legacy_redirect")
async def get_current_user_legacy(request: Request):
    """Legacy endpoint - use GET /api/users?current=true instead"""
    return await get_users(request, current=True)

@router.get("/api/debug_user", operation_id="debug_user_legacy_redirect")
async def debug_user_legacy(request: Request, email: str):
    """Legacy endpoint - use GET /api/users?email={email}&debug=true instead"""
    return await get_users(request, email=email, debug=True)

# Unified users endpoint - consolidates edit_users, current_user, debug_user, and get_users
@router.get("/api/users")
async def get_users(
    request: Request,
    user_id: Optional[str] = Query(None, description="Get specific user by ID"),
    email: Optional[str] = Query(None, description="Get user by email (for lookup)"),
    current: bool = Query(False, description="Get current logged-in user"),
    debug: bool = Query(False, description="Debug mode - show password metadata (admin only)"),
    title: Optional[str] = Query(None, description="Filter by role/title"),
    status: Optional[int] = Query(None, description="Filter by status (1=active, 0=inactive)"),
    search: Optional[str] = Query(None, description="Search by name or email"),
    page: int = Query(1, ge=1, description="Page number (starts at 1)"),
    limit: int = Query(50, ge=1, le=100, description="Items per page (max 100)")
):
    """
    Unified endpoint to get users with various filters and options.
    
    Use cases:
    - Get current user: ?current=true
    - Get single user: ?user_id=123
    - Get user by email: ?email=user@example.com
    - Get all users: (no filters)
    - Filter by role: ?title=analyst
    - Filter by status: ?status=1
    - Search: ?search=john
    - Debug mode: ?email=user@example.com&debug=true (admin only)
    """
    # Require authentication
    current_user = require_auth(request)
    current_user_title = (current_user.get('title') or '').lower()
    current_user_id = current_user.get('id') or current_user.get('user_id')
    current_user_email = (current_user.get('email') or current_user.get('name') or '').lower().strip()
    
    # Debug mode requires admin access
    if debug and current_user_title not in ['admin', 'super admin']:
        raise HTTPException(
            status_code=403,
            detail="Unauthorized: Debug mode requires admin access"
        )
    
    # Input validation
    if user_id:
        user_id = user_id.strip()
        if not user_id:
            raise HTTPException(status_code=400, detail="user_id cannot be empty")
        if len(user_id) > 50:
            raise HTTPException(status_code=400, detail="user_id too long (max 50 characters)")
        # Validate user_id format (alphanumeric, hyphens, underscores)
        if not re.match(r'^[a-zA-Z0-9\-_]+$', user_id):
            raise HTTPException(status_code=400, detail="Invalid user_id format")
    
    if email:
        email = email.strip().lower()
        if not email:
            raise HTTPException(status_code=400, detail="Email cannot be empty")
        if len(email) > 100:
            raise HTTPException(status_code=400, detail="Email too long (max 100 characters)")
        # Validate email format
        email_pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
        if not re.match(email_pattern, email):
            raise HTTPException(status_code=400, detail="Invalid email format")
    
    if title:
        title = title.strip().lower()
        # Whitelist valid titles
        valid_titles = ['admin', 'super admin', 'user', 'analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead', 'operator']
        if title not in valid_titles:
            raise HTTPException(
                status_code=400,
                detail=f"Invalid title. Must be one of: {', '.join(valid_titles)}"
            )
    
    if search:
        search = search.strip()
        if len(search) > 100:
            raise HTTPException(status_code=400, detail="Search term too long (max 100 characters)")
        # Sanitize search term (remove potentially dangerous characters)
        if not re.match(r'^[a-zA-Z0-9\s._@-]+$', search):
            raise HTTPException(status_code=400, detail="Invalid search term format")
    
    if status is not None and status not in [0, 1]:
        raise HTTPException(status_code=400, detail="Status must be 0 (inactive) or 1 (active)")
    
    # Authorization: Non-admin users can only view their own data or limited public info
    is_admin = current_user_title in ['admin', 'super admin']
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        # Case 1: Get current logged-in user
        if current:
            return JSONResponse({
                "success": True,
                "user": {
                    "id": current_user_id,
                    "user_id": current_user_id,
                    "name": current_user.get('name') or current_user.get('email'),
                    "email": current_user.get('email') or current_user.get('name'),
                    "user_name": current_user.get('user_name') or current_user.get('name'),
                    "imglink": current_user.get('imglink', 'images/admin.png'),
                    "title": current_user.get('title'),
                    "shift": current_user.get('shift')
                },
                "source": "jwt_token"
            })
        
        # Case 2: Get single user by ID
        if user_id:
            if debug:
                cursor.execute(
                    """SELECT email, user_id, name, title, shift, area_of_operation, status,
                          LENGTH(password) as password_length, LEFT(password, 10) as password_preview 
                       FROM users WHERE user_id = %s""",
                    (user_id,)
                )
            else:
                cursor.execute(
                    "SELECT user_id, email, name, title, shift, area_of_operation, status, imglink FROM users WHERE user_id = %s",
                    (user_id,)
                )
            
            user = cursor.fetchone()
            
            if not user:
                raise HTTPException(status_code=404, detail=f"User with ID '{user_id}' not found")
            
            # Authorization: Non-admin users can only view their own data
            if not is_admin:
                user_email = (user.get('email') or '').lower().strip()
                if user_email != current_user_email and user_id != current_user_id:
                    raise HTTPException(
                        status_code=403,
                        detail="Unauthorized: You can only view your own user information"
                    )
            
            # Add debug info if requested
            if debug:
                user['is_hashed'] = user.get('password_preview', '').startswith('$2') if user.get('password_preview') else False
            
            return JSONResponse({
                "success": True,
                "user": user,
                "debug_mode": debug
            })
        
        # Case 3: Get user by email (for lookup/debug)
        if email:
            email = email.strip().lower()
            
            if debug:
                cursor.execute(
                    """SELECT email, user_id, name, title, shift, area_of_operation, status,
                          LENGTH(password) as password_length, LEFT(password, 10) as password_preview 
                       FROM users WHERE email = %s""",
                    (email,)
                )
            else:
                cursor.execute(
                    "SELECT user_id, email, name, title, shift, area_of_operation, status, imglink FROM users WHERE email = %s",
                    (email,)
                )
            
            user = cursor.fetchone()
            
            if not user:
                return JSONResponse({
                    "success": False,
                    "found": False,
                    "message": f"User with email '{email}' not found"
                })
            
            # Authorization: Non-admin users can only view their own data
            if not is_admin:
                user_email = (user.get('email') or '').lower().strip()
                if user_email != current_user_email:
                    raise HTTPException(
                        status_code=403,
                        detail="Unauthorized: You can only view your own user information"
                    )
            
            # Add debug info if requested
            if debug:
                user['is_hashed'] = user.get('password_preview', '').startswith('$2') if user.get('password_preview') else False
            
            return JSONResponse({
                "success": True,
                "found": True,
                "user": user,
                "debug_mode": debug
            })
        
        # Case 4: Get list of users with filters
        # Authorization: Only admins can view all users list
        if not is_admin:
            raise HTTPException(
                status_code=403,
                detail="Unauthorized: Only admins can view the users list. Use ?current=true to view your own information."
            )
        
        # Build query with filters
        query = "SELECT user_id, email, name, title, shift, area_of_operation, status, imglink FROM users WHERE 1=1"
        params = []
        
        if title:
            query += " AND title = %s"
            params.append(title)
        
        if status is not None:
            query += " AND status = %s"
            params.append(status)
        
        if search:
            search_term = f"%{search.strip()}%"
            query += " AND (name LIKE %s OR email LIKE %s)"
            params.extend([search_term, search_term])
        
        query += " ORDER BY name ASC"
        
        cursor.execute(query, tuple(params))
        all_users = cursor.fetchall()
        
        # Apply pagination
        paginated = paginate_results(all_users, page, limit)
        
        return JSONResponse({
            "success": True,
            "users": paginated["items"],
            "pagination": paginated["pagination"],
            "filters": {
                "title": title,
                "status": status,
                "search": search
            }
        })
        
    except HTTPException:
        raise
    except mysql.connector.Error as db_error:
        import traceback
        error_detail = f"{str(db_error)}\n{traceback.format_exc()}"
        print(f"ERROR in get_users (Database): {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"Database error while fetching users: {str(db_error)}"
        )
    except Exception as e:
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in get_users: {error_detail}")
        raise HTTPException(
            status_code=500,
            detail=f"An error occurred while fetching users: {str(e)}"
        )
    finally:
        cursor.close()
        conn.close()

# Duplicate legacy endpoints removed - using the ones defined earlier with unique operation_id

@router.get("/api/get_users")
async def get_users_legacy(
    request: Request, 
    title: Optional[str] = None,
    page: int = Query(1, ge=1),
    limit: int = Query(50, ge=1, le=100)
):
    """Legacy endpoint - use GET /api/users?title={title}&page={page}&limit={limit} instead"""
    return await get_users(request, title=title, page=page, limit=limit)

# Get categories by hub/section endpoint
@router.get("/api/categorys_hub")
async def get_categories_hub(request: Request, tracking_number: str):
    """Get categories for a given hub/section"""
    session = require_auth(request)
    
    conn = get_db_connection()
    cursor = conn.cursor()
    
    try:
        value = tracking_number.upper().strip()
        if not value:
            return HTMLResponse('<option>ERROR Wrong Input...</option>')
        
        cursor.execute("SELECT DISTINCT category FROM products WHERE section = %s AND category IS NOT NULL AND category != ''", (value,))
        categories = [row[0] for row in cursor.fetchall()]
        
        if not categories:
            return HTMLResponse('<option selected disabled>No categories found</option>')
        
        options = '<option selected disabled>---</option>'
        for cat in categories:
            options += f'<option value="{cat}">{cat}</option>'
        
        return HTMLResponse(options)
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))
    finally:
        cursor.close()
        conn.close()


# Get operator scans endpoint
@router.get("/api/operator_scans")
async def get_operator_scans(request: Request, operator: str):
    """Get scans for an operator"""
    session = require_auth(request)
    
    conn = get_db_connection()
    cursor = conn.cursor(dictionary=True)
    
    try:
        rows = []
        use_status_column = False
        
        # Match PHP query exactly: GROUP BY batch_no only
        # Note: MySQL in strict mode may require all selected columns in GROUP BY
        # But the PHP version works, so we'll try the same query
        query = """SELECT scan_date, scan_time, batch_no, operator, status,
                   COUNT(batch_no) as count 
                   FROM data 
                   WHERE operator = %s 
                   GROUP BY batch_no
                   ORDER BY scan_date DESC"""
        try:
            cursor.execute(query, (operator,))
            rows = cursor.fetchall()
            use_status_column = True
            print(f"DEBUG: operator_scans query successful, found {len(rows)} rows")
        except Exception as e:
            print(f"ERROR: Query with status column failed: {e}")
            print("Trying without status column...")
            # If status column doesn't exist or GROUP BY fails, try without status
            query = """SELECT scan_date, scan_time, batch_no, operator,
                       COUNT(batch_no) as count 
                       FROM data 
                       WHERE operator = %s 
                       GROUP BY batch_no
                       ORDER BY scan_date DESC"""
            try:
                cursor.execute(query, (operator,))
                rows = cursor.fetchall()
                use_status_column = False
                print(f"DEBUG: Query without status successful, found {len(rows)} rows")
            except Exception as e2:
                print(f"ERROR: Query without status also failed: {e2}")
                # Last resort: use ANY_VALUE() for MySQL 5.7+ compatibility
                try:
                    query = """SELECT ANY_VALUE(scan_date) as scan_date, 
                              ANY_VALUE(scan_time) as scan_time, 
                              batch_no, 
                              ANY_VALUE(operator) as operator,
                              ANY_VALUE(status) as status,
                              COUNT(batch_no) as count 
                              FROM data 
                              WHERE operator = %s 
                              GROUP BY batch_no
                              ORDER BY scan_date DESC"""
                    cursor.execute(query, (operator,))
                    rows = cursor.fetchall()
                    use_status_column = True
                    print(f"DEBUG: Query with ANY_VALUE successful, found {len(rows)} rows")
                except Exception as e3:
                    print(f"ERROR: All queries failed. Last error: {e3}")
                    raise Exception(f"All query attempts failed. Last error: {e3}")
        
        html = ''
        if rows:
            count = 0
            for row in rows:
                count += 1
                batch_no = str(row.get('batch_no') or '')
                operator_name = str(row.get('operator') or '')
                
                if use_status_column:
                    status = str(row.get('status', '0'))
                else:
                    # If no status column, default to '0' (pending)
                    status = '0'
                item_count = row.get('count', 0)
                scan_date = str(row.get('scan_date') or '')
                scan_time = str(row.get('scan_time') or '')
                date_time = f"{scan_date} {scan_time}".strip()
                
                # Escape for HTML
                batch_no_escaped = html_module.escape(batch_no, quote=True)
                
                html += f'<tr><th style="font-size:10px;">{count}</th>'
                html += f'<th style="font-size:10px;">{html_module.escape(batch_no, quote=True)}</th>'
                html += f'<th style="font-size:10px;">{html_module.escape(operator_name, quote=True)}</th>'
                html += f'<th style="font-size:10px;">{item_count}</th>'
                html += f'<th style="font-size:10px;">{html_module.escape(date_time, quote=True)}</th>'
                
                if status == "0":
                    html += f'<th><button class="send" style="color:color; font-size:9px;padding:4px;margin:0px;" onclick="send_data_processed(\'{batch_no_escaped}\')">Send</button></th>'
                else:
                    html += f'<th><button class="send" style="color:color; font-size:9px;padding:4px;margin:0px;" onclick="send_data_processed(\'{batch_no_escaped}\')">Re-send</button></th>'
                html += '</tr>'
        else:
            html = '<tr><th><b>No Data Found..!</b></th></tr>'
        
        return HTMLResponse(html)
    except Exception as e:
        import traceback
        error_detail = f"{str(e)}\n{traceback.format_exc()}"
        print(f"ERROR in get_operator_scans: {error_detail}")
        print(f"  Operator: {operator}")
        raise HTTPException(status_code=500, detail=f"Error fetching operator scans: {str(e)}")
    finally:
        cursor.close()
        conn.close()

# Email and Password Reset endpoints
from backend.password_reset import create_password_reset_token, verify_reset_token, reset_password_with_token
from backend.email_service import send_welcome_email
from pydantic import EmailStr

class ForgotPasswordRequest(BaseModel):
    email: EmailStr

class ResetPasswordRequest(BaseModel):
    token: str
    password: str
    confirm_password: str

@router.post("/api/forgot_password")
async def forgot_password(request: Request, forgot_data: ForgotPasswordRequest):
    """Handle password reset request"""
    try:
        print(f"\n{'='*60}")
        print(f"FORGOT PASSWORD REQUEST")
        print(f"{'='*60}")
        print(f"Email: {forgot_data.email}")
        
        result = create_password_reset_token(forgot_data.email, 1, request)
        
        print(f"Result success: {result.get('success')}")
        print(f"Email sent: {result.get('email_sent', 'unknown')}")
        if result.get('email_error'):
            print(f"Email error: {result.get('email_error')}")
        print(f"{'='*60}\n")
        
        if result['success']:
            # Check if email was actually sent (for debugging)
            email_sent = result.get('email_sent', True)
            if not email_sent:
                # Log the error but still return success to user (security)
                print(f"WARNING: Token created but email failed to send for: {forgot_data.email}")
                print(f"ERROR DETAILS: {result.get('email_error', 'Unknown error')}")
            
            return JSONResponse({
                'success': True,
                'message': 'A password reset link has been sent to your email address if it exists in our system. Please check your inbox and follow the instructions to reset your password.',
                'instructions': [
                    '1. Check your email inbox (and spam folder) for the password reset email',
                    '2. Click the "Reset Password" button in the email',
                    '3. Enter your new password (minimum 8 characters)',
                    '4. The reset link will expire in 1 hour for security reasons'
                ],
                'debug': {
                    'email_sent': email_sent,
                    'email_error': result.get('email_error') if not email_sent else None
                } if os.getenv('DEBUG', 'false').lower() == 'true' else None
            })
        else:
            raise HTTPException(status_code=500, detail=result['message'])
    except Exception as e:
        import traceback
        error_detail = traceback.format_exc()
        print(f"ERROR in forgot_password endpoint: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred: {str(e)}")

@router.get("/api/verify_reset_token")
async def verify_reset_token_endpoint(request: Request, token: str = Query(..., description="Password reset token")):
    """
    Verify a password reset token
    
    This endpoint checks if a password reset token is valid, not expired, and not already used.
    
    **Usage:**
    - GET /api/verify_reset_token?token=YOUR_TOKEN_HERE
    
    **Response:**
    - If valid: Returns email and success message
    - If invalid: Returns error message with 400 status
    
    **Token validation checks:**
    - Token exists in database
    - Token has not been used
    - Token has not expired (default: 1 hour)
    """
    try:
        print(f"\n{'='*60}")
        print(f"VERIFY RESET TOKEN REQUEST")
        print(f"{'='*60}")
        print(f"Token: {token[:20]}...{token[-10:] if len(token) > 30 else ''}")
        
        if not token or len(token) < 10:
            print(f"ERROR: Invalid token format (too short)")
            return JSONResponse({
                'valid': False,
                'message': 'Invalid token format'
            }, status_code=400)
        
        verification = verify_reset_token(token)
        
        print(f"Verification result: valid={verification.get('valid')}")
        if verification.get('email'):
            print(f"Email: {verification.get('email')}")
        print(f"Message: {verification.get('message')}")
        print(f"{'='*60}\n")
        
        if verification['valid']:
            return JSONResponse({
                'valid': True,
                'email': verification['email'],
                'user_id': verification.get('user_id'),
                'message': verification['message']
            })
        else:
            return JSONResponse({
                'valid': False,
                'message': verification['message']
            }, status_code=400)
            
    except Exception as e:
        import traceback
        error_detail = traceback.format_exc()
        print(f"ERROR in verify_reset_token endpoint: {error_detail}")
        return JSONResponse({
            'valid': False,
            'message': f'Error verifying token: {str(e)}'
        }, status_code=500)

@router.get("/api/verify_credential_token")
async def verify_credential_token_endpoint(request: Request, token: str = Query(..., description="Credential viewing token")):
    """
    Verify a credential viewing token and return credentials (one-time use)
    
    This endpoint checks if a credential token is valid, not expired, and not already viewed.
    After successful verification, the token is marked as viewed and cannot be used again.
    
    **Usage:**
    - GET /api/verify_credential_token?token=YOUR_TOKEN_HERE
    
    **Response:**
    - If valid: Returns email and password
    - If invalid: Returns error message with 400 status
    """
    try:
        from backend.password_reset import verify_credential_token
        
        print(f"\n{'='*60}")
        print(f"VERIFY CREDENTIAL TOKEN REQUEST")
        print(f"{'='*60}")
        print(f"Token: {token[:20]}...{token[-10:] if len(token) > 30 else ''}")
        
        if not token or len(token) < 10:
            print(f"ERROR: Invalid token format (too short)")
            return JSONResponse({
                'valid': False,
                'message': 'Invalid token format'
            }, status_code=400)
        
        verification = verify_credential_token(token)
        
        print(f"Verification result: valid={verification.get('valid')}")
        if verification.get('email'):
            print(f"Email: {verification.get('email')}")
        print(f"Message: {verification.get('message')}")
        print(f"{'='*60}\n")
        
        if verification['valid']:
            return JSONResponse({
                'valid': True,
                'email': verification['email'],
                'password': verification['password'],
                'user_id': verification.get('user_id'),
                'message': verification['message']
            })
        else:
            return JSONResponse({
                'valid': False,
                'message': verification['message']
            }, status_code=400)
            
    except Exception as e:
        import traceback
        error_detail = traceback.format_exc()
        print(f"ERROR in verify_credential_token endpoint: {error_detail}")
        return JSONResponse({
            'valid': False,
            'message': f'Error verifying token: {str(e)}'
        }, status_code=500)

@router.post("/api/reset_password")
async def reset_password_endpoint(request: Request, reset_data: ResetPasswordRequest):
    """Reset password using token"""
    # Validate passwords match
    if reset_data.password != reset_data.confirm_password:
        raise HTTPException(status_code=400, detail="Passwords do not match")
    
    result = reset_password_with_token(reset_data.token, reset_data.password)
    
    if result['success']:
        return JSONResponse({
            'success': True,
            'message': 'Your password has been successfully reset. You can now log in with your new password.',
            'instructions': [
                '1. Go to the login page',
                '2. Enter your email address and new password',
                '3. You should receive a confirmation email shortly'
            ]
        })
    else:
        raise HTTPException(
            status_code=400, 
            detail=result.get('message', 'Failed to reset password. Please ensure your reset token is valid and not expired.')
        )

class SendWelcomeEmailRequest(BaseModel):
    email: str
    name: str
    password: str
    title: str
    shift: str

@router.post("/api/send_welcome_email")
async def send_welcome_email_endpoint(request: Request, welcome_data: SendWelcomeEmailRequest):
    """
    Send welcome email to new user
    
    This endpoint sends a welcome email with account credentials to a new user.
    Requires authentication (admin/super admin).
    """
    try:
        # Require authentication
        session = require_auth(request)
        
        print(f"\n{'='*60}")
        print(f"SEND WELCOME EMAIL REQUEST")
        print(f"{'='*60}")
        print(f"Email: {welcome_data.email}")
        print(f"Name: {welcome_data.name}")
        print(f"Title: {welcome_data.title}")
        print(f"Shift: {welcome_data.shift}")
        
        # Generate login URL
        import os
        base_url = os.getenv('BASE_URL') or os.getenv('FRONTEND_URL')
        if not base_url:
            base_url = f"{request.url.scheme}://{request.url.netloc}"
            if ':8000' in base_url:
                base_url = base_url.replace(':8000', '')
            login_url = f"{base_url}/blue/index.php"
        else:
            login_url = f"{base_url}/index.php"
        
        print(f"Login URL: {login_url}")
        
        # Send welcome email
        result = send_welcome_email(
            welcome_data.email, 
            welcome_data.name, 
            welcome_data.password, 
            welcome_data.title, 
            welcome_data.shift, 
            login_url
        )
        
        print(f"Email result: {result.get('success')}")
        if not result.get('success'):
            print(f"Email error: {result.get('message')}")
        print(f"{'='*60}\n")
        
        if result['success']:
            return JSONResponse({
                'success': True,
                'message': f'Welcome email has been successfully sent to {welcome_data.email}. The user will receive their account credentials via email.',
                'details': {
                    'email': welcome_data.email,
                    'name': welcome_data.name,
                    'role': welcome_data.title,
                    'shift': welcome_data.shift
                }
            })
        else:
            return JSONResponse({
                'success': False,
                'message': f'Failed to send welcome email: {result.get("message", "Unknown error occurred")}. Please verify the email address and try again.',
                'error_details': result.get('message') if os.getenv('DEBUG', 'false').lower() == 'true' else None
            }, status_code=500)
            
    except Exception as e:
        import traceback
        error_detail = traceback.format_exc()
        print(f"ERROR in send_welcome_email endpoint: {error_detail}")
        raise HTTPException(status_code=500, detail=f"An error occurred: {str(e)}")

# Report Generation endpoints
from backend.report_generator import generate_pdf_report, generate_excel_report, get_batch_data
from fastapi.responses import StreamingResponse

class GenerateReportRequest(BaseModel):
    serial_no: str
    shift: str
    operator: str
    format: str = 'pdf'  # 'pdf' or 'xlsx'

@router.post("/api/generate_report")
async def generate_report_endpoint(request: Request, report_data: GenerateReportRequest):
    """Generate PDF or Excel report for batch data"""
    session = require_auth(request)
    
    # Validate format
    if report_data.format not in ['pdf', 'xlsx']:
        raise HTTPException(status_code=400, detail="Invalid format. Must be 'pdf' or 'xlsx'")
    
    # Fetch batch data
    items = get_batch_data(report_data.serial_no, report_data.shift, report_data.operator)
    
    if not items:
        raise HTTPException(status_code=404, detail="No data found for this batch")
    
    try:
        if report_data.format == 'pdf':
            # Generate PDF
            pdf_buffer = generate_pdf_report(
                items, 
                report_data.serial_no, 
                report_data.shift, 
                report_data.operator
            )
            
            filename = f'batch_report_{report_data.serial_no}.pdf'
            return StreamingResponse(
                pdf_buffer,
                media_type='application/pdf',
                headers={
                    'Content-Disposition': f'attachment; filename="{filename}"'
                }
            )
        
        else:  # xlsx
            # Generate Excel
            excel_buffer = generate_excel_report(
                items,
                report_data.serial_no,
                report_data.shift,
                report_data.operator
            )
            
            filename = f'batch_report_{report_data.serial_no}.xlsx'
            return StreamingResponse(
                excel_buffer,
                media_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                headers={
                    'Content-Disposition': f'attachment; filename="{filename}"'
                }
            )
    
    except ImportError as e:
        raise HTTPException(
            status_code=500, 
            detail=f"Report generation library not installed: {str(e)}"
        )
    except Exception as e:
        raise HTTPException(
            status_code=500,
            detail=f"Report generation failed: {str(e)}"
        )


class TestEmailRequest(BaseModel):
    email: str

@router.post("/api/test_email")
async def test_email(request: Request, test_data: Optional[TestEmailRequest] = None):
    """
    Test email sending functionality
    Sends a test email and returns debug information
    
    Usage:
    - POST /api/test_email with JSON body: {"email": "your@email.com"}
    - POST /api/test_email?email=your@email.com
    """
    """
    Test email sending functionality
    Sends a test email and returns debug information
    """
    from backend.email_service import send_email, load_email_template
    
    try:
        # Get test email from request body or query params
        test_email_address = None
        
        # Try to get from request body
        if test_data and isinstance(test_data, TestEmailRequest):
            test_email_address = test_data.email
        else:
            try:
                body = await request.json()
                if body and isinstance(body, dict):
                    test_email_address = body.get('email')
            except:
                pass
        
        # Try query params as fallback
        if not test_email_address:
            test_email_address = request.query_params.get('email')
        
        if not test_email_address:
            return JSONResponse({
                'success': False,
                'message': 'Please provide an email address',
                'usage': 'POST /api/test_email with {"email": "your@email.com"} or ?email=your@email.com'
            }, status_code=400)
        
        # Test 1: Load and render email template
        print(f"\n{'='*60}")
        print(f"EMAIL TEST - Sending to: {test_email_address}")
        print(f"{'='*60}\n")
        
        test_reset_url = "http://localhost/blue/reset_password.php?token=TEST_TOKEN_12345"
        template_context = {
            'name': 'Test User',
            'reset_url': test_reset_url,
            'expiry_hours': 1
        }
        
        print("Step 1: Loading email template...")
        rendered_template = load_email_template('password_reset', template_context)
        print(f"✓ Template loaded successfully ({len(rendered_template)} characters)")
        
        # Check if reset URL is in template
        import re
        href_matches = re.findall(r'href=["\']([^"\']+)["\']', rendered_template)
        url_in_text = test_reset_url in rendered_template
        
        print(f"✓ Reset URL found in template: {url_in_text}")
        if href_matches:
            print(f"✓ Found {len(href_matches)} href link(s): {href_matches[0][:50]}...")
        
        # Test 2: Send test email
        print("\nStep 2: Sending test email...")
        email_result = send_email(
            to=test_email_address,
            subject='Test Email - Krystal EA System',
            message=rendered_template,
            is_html=True
        )
        
        if email_result['success']:
            print(f"✓ Email sent successfully!")
            return JSONResponse({
                'success': True,
                'message': f'Test email sent successfully to {test_email_address}',
                'debug_info': {
                    'template_length': len(rendered_template),
                    'reset_url_in_template': url_in_text,
                    'href_links_found': len(href_matches),
                    'first_href': href_matches[0] if href_matches else None,
                    'email_sent': True
                },
                'instructions': [
                    '1. Check your email inbox for the test email',
                    '2. Verify the "Reset Password" button is clickable',
                    '3. Check that the reset URL is visible in the email',
                    '4. Click the link to verify it opens correctly'
                ]
            })
        else:
            return JSONResponse({
                'success': False,
                'message': f'Failed to send test email: {email_result.get("message", "Unknown error")}',
                'debug_info': {
                    'template_length': len(rendered_template),
                    'reset_url_in_template': url_in_text,
                    'href_links_found': len(href_matches),
                    'email_sent': False
                }
            }, status_code=500)
            
    except Exception as e:
        import traceback
        error_detail = traceback.format_exc()
        print(f"ERROR in test_email: {error_detail}")
        return JSONResponse({
            'success': False,
            'message': f'Error during email test: {str(e)}',
            'error_detail': error_detail
        }, status_code=500)


@router.get("/api/test_email_template")
async def test_email_template():
    """
    Test email template rendering without sending email
    Returns the rendered template HTML for inspection
    """
    from backend.email_service import load_email_template
    
    try:
        test_reset_url = "http://localhost/blue/reset_password.php?token=TEST_TOKEN_12345"
        template_context = {
            'name': 'Test User',
            'reset_url': test_reset_url,
            'expiry_hours': 1
        }
        
        rendered_template = load_email_template('password_reset', template_context)
        
        # Check for reset URL in template
        import re
        href_matches = re.findall(r'href=["\']([^"\']+)["\']', rendered_template)
        url_in_text = test_reset_url in rendered_template
        
        return JSONResponse({
            'success': True,
            'template_length': len(rendered_template),
            'reset_url_in_template': url_in_text,
            'reset_url': test_reset_url,
            'href_links_found': len(href_matches),
            'href_links': href_matches,
            'template_preview': rendered_template[:500] + '...' if len(rendered_template) > 500 else rendered_template,
            'full_template': rendered_template  # Include full template for inspection
        })
        
    except Exception as e:
        import traceback
        return JSONResponse({
            'success': False,
            'message': f'Error rendering template: {str(e)}',
            'error_detail': traceback.format_exc()
        }, status_code=500)

