"""
SAP Integration Usage Examples
Demonstrates how to use the SAP integration endpoints.
"""

import asyncio
import httpx
from datetime import date
from decimal import Decimal

# Configuration
API_BASE_URL = "http://localhost:8000"
JWT_TOKEN = "your-jwt-token-here"  # Get this from login endpoint

# Headers for authenticated requests
HEADERS = {
    "Authorization": f"Bearer {JWT_TOKEN}",
    "Content-Type": "application/json"
}


async def example_material_trigger():
    """
    Example: Trigger material request to SAP.
    This will cause SAP to send back process orders for the material.
    """
    print("=== Material Trigger Example ===")
    
    async with httpx.AsyncClient() as client:
        response = await client.post(
            f"{API_BASE_URL}/api/v1/sap/material-trigger",
            json={
                "material_number": "FG001234",
                "plant": "K013"
            },
            headers=HEADERS
        )
        
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Material trigger successful!")
            print(f"   Request ID: {data['request_id']}")
            print(f"   Material: {data['material_number']}")
            print(f"   Plant: {data['plant']}")
            print(f"   Message: {data['message']}")
        else:
            print(f"❌ Material trigger failed: {response.status_code}")
            print(f"   Error: {response.text}")


async def example_component_consumption():
    """
    Example: Send component consumption to SAP (MAK080).
    This creates goods issue transactions in SAP.
    """
    print("\n=== Component Consumption Example ===")
    
    consumption_data = {
        "document_date": date.today().strftime("%Y-%m-%d"),
        "posting_date": date.today().strftime("%Y-%m-%d"),
        "transaction_code": "MB1A",
        "description": "Component consumption from Krystal EA weighing",
        "components": [
            {
                "material_number": "RM001234",
                "plant": "K013",
                "storage_location": "0001",
                "batch_number": "BATCH001",
                "movement_type": "261",
                "quantity": 25.500,
                "unit_of_measure": "KG",
                "order_number": "000012345678"
            },
            {
                "material_number": "RM005678",
                "plant": "K013",
                "storage_location": "0001",
                "batch_number": "BATCH002",
                "movement_type": "261",
                "quantity": 15.200,
                "unit_of_measure": "KG",
                "order_number": "000012345678"
            }
        ]
    }
    
    async with httpx.AsyncClient() as client:
        response = await client.post(
            f"{API_BASE_URL}/api/v1/sap/component-consumption",
            json=consumption_data,
            headers=HEADERS
        )
        
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Component consumption successful!")
            print(f"   Request ID: {data['request_id']}")
            print(f"   IDOC Number: {data.get('idoc_number', 'N/A')}")
            print(f"   Components Count: {data['components_count']}")
            print(f"   Message: {data['message']}")
        else:
            print(f"❌ Component consumption failed: {response.status_code}")
            print(f"   Error: {response.text}")


async def example_get_plants():
    """
    Example: Get valid plants for SAP integration.
    """
    print("\n=== Get Valid Plants Example ===")
    
    async with httpx.AsyncClient() as client:
        response = await client.get(
            f"{API_BASE_URL}/api/v1/sap/plants",
            headers=HEADERS
        )
        
        if response.status_code == 200:
            data = response.json()
            print("✅ Valid plants retrieved:")
            for plant in data["plants"]:
                description = data["plant_descriptions"].get(plant, "Unknown")
                print(f"   {plant}: {description}")
        else:
            print(f"❌ Failed to get plants: {response.status_code}")


async def example_get_order_types():
    """
    Example: Get valid order types for a plant.
    """
    print("\n=== Get Order Types Example ===")
    
    plant = "K013"
    
    async with httpx.AsyncClient() as client:
        response = await client.get(
            f"{API_BASE_URL}/api/v1/sap/order-types/{plant}",
            headers=HEADERS
        )
        
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Order types for plant {plant}:")
            for order_type in data["order_types"]:
                description = data["descriptions"].get(order_type, "Unknown")
                print(f"   {order_type}: {description}")
        else:
            print(f"❌ Failed to get order types: {response.status_code}")


async def example_login():
    """
    Example: Login to get JWT token.
    """
    print("=== Login Example ===")
    
    async with httpx.AsyncClient() as client:
        response = await client.post(
            f"{API_BASE_URL}/api/v1/auth/login",
            json={
                "email": "user@example.com",
                "password": "password123"
            }
        )
        
        if response.status_code == 200:
            data = response.json()
            token = data["access_token"]
            print(f"✅ Login successful!")
            print(f"   Token: {token[:50]}...")
            return token
        else:
            print(f"❌ Login failed: {response.status_code}")
            print(f"   Error: {response.text}")
            return None


async def example_error_handling():
    """
    Example: Demonstrate error handling for SAP integration.
    """
    print("\n=== Error Handling Example ===")
    
    # Try with invalid plant
    async with httpx.AsyncClient() as client:
        response = await client.post(
            f"{API_BASE_URL}/api/v1/sap/material-trigger",
            json={
                "material_number": "FG001234",
                "plant": "INVALID"  # Invalid plant
            },
            headers=HEADERS
        )
        
        print(f"Invalid plant response: {response.status_code}")
        if response.status_code != 200:
            error_data = response.json()
            print(f"   Error details: {error_data}")
        
        # Try with invalid material number format
        response = await client.post(
            f"{API_BASE_URL}/api/v1/sap/material-trigger",
            json={
                "material_number": "",  # Empty material number
                "plant": "K013"
            },
            headers=HEADERS
        )
        
        print(f"Empty material response: {response.status_code}")
        if response.status_code != 200:
            error_data = response.json()
            print(f"   Error details: {error_data}")


async def complete_workflow_example():
    """
    Example: Complete SAP integration workflow.
    """
    print("\n=== Complete Workflow Example ===")
    
    # Step 1: Get valid plants
    print("Step 1: Getting valid plants...")
    await example_get_plants()
    
    # Step 2: Get order types for a plant
    print("\nStep 2: Getting order types...")
    await example_get_order_types()
    
    # Step 3: Trigger material request
    print("\nStep 3: Triggering material request...")
    await example_material_trigger()
    
    # Step 4: Send component consumption
    print("\nStep 4: Sending component consumption...")
    await example_component_consumption()
    
    print("\n✅ Complete workflow finished!")


async def main():
    """
    Main function to run all examples.
    """
    print("🚀 SAP Integration Examples")
    print("=" * 50)
    
    # Note: In a real scenario, you would first login to get a valid token
    print("⚠️  Note: Make sure to set a valid JWT_TOKEN in the script")
    print("   You can get this by calling the login endpoint first")
    
    try:
        # Run all examples
        await example_get_plants()
        await example_get_order_types()
        await example_material_trigger()
        await example_component_consumption()
        await example_error_handling()
        await complete_workflow_example()
        
    except Exception as e:
        print(f"❌ Error running examples: {e}")
        print("   Make sure the API server is running and you have a valid JWT token")


if __name__ == "__main__":
    # Run the examples
    asyncio.run(main())