"""
Email Service for FastAPI
Handles email sending functionality including welcome emails and password reset emails
Uses PHP endpoint (api_send_email.php) which uses PHPMailer via Composer
"""

import requests
from typing import Dict, Optional
import os
from pathlib import Path
from datetime import datetime
import logging

from app.core.config import get_settings
from app.core.exceptions import EmailServiceError

logger = logging.getLogger(__name__)
settings = get_settings()

# Email templates directory
TEMPLATES_DIR = Path(__file__).parent.parent.parent / 'email_templates'


class EmailService:
    """Enhanced email service with better error handling and integration."""
    
    def __init__(self):
        self.php_endpoint = settings.php_email_endpoint
        self.from_email = settings.smtp_from_email
        self.from_name = settings.smtp_from_name
        self.mail_enabled = settings.mail_enabled
        self.debug = settings.mail_debug
    
    def send_email(self, to: str, subject: str, message: str, is_html: bool = True) -> Dict[str, any]:
        """
        Send an email using PHP endpoint (api_send_email.php) which uses PHPMailer
        
        Args:
            to: Recipient email address
            subject: Email subject
            message: Email body (HTML or plain text)
            is_html: Whether message is HTML (default: True)
        
        Returns:
            Dict with 'success' and 'message' keys
        """
        if not self.mail_enabled:
            logger.info(f"Email sending disabled - would send to {to}: {subject}")
            return {'success': True, 'message': 'Email sending disabled (development mode)'}
        
        try:
            if self.debug:
                logger.info(f"Sending email to {to}: {subject}")
            
            # Prepare email data
            email_data = {
                'to': to,
                'subject': subject,
                'message': message,
                'is_html': is_html
            }
            
            # Send request to PHP endpoint
            response = requests.post(
                self.php_endpoint,
                json=email_data,
                headers={'Content-Type': 'application/json'},
                timeout=30
            )
            
            if self.debug:
                logger.info(f"PHP endpoint response status: {response.status_code}")
            
            # Parse response
            try:
                result = response.json()
            except Exception as json_error:
                logger.error(f"Failed to parse JSON response: {json_error}")
                raise EmailServiceError(f"Invalid response from email service: {response.text[:100]}")
            
            if response.status_code == 200 and result.get('success'):
                logger.info(f"Email sent successfully to {to}")
                return {'success': True, 'message': result.get('message', 'Email sent successfully')}
            else:
                error_msg = result.get('message', f'Email service returned status {response.status_code}')
                logger.error(f"Email sending failed: {error_msg}")
                raise EmailServiceError(error_msg)
        
        except requests.exceptions.ConnectionError as e:
            error_msg = f'Failed to connect to email service: {str(e)}'
            logger.error(error_msg)
            raise EmailServiceError(error_msg)
        except requests.exceptions.Timeout as e:
            error_msg = f'Email service request timeout: {str(e)}'
            logger.error(error_msg)
            raise EmailServiceError(error_msg)
        except EmailServiceError:
            raise
        except Exception as e:
            error_msg = f'Failed to send email: {str(e)}'
            logger.error(error_msg)
            raise EmailServiceError(error_msg)
    
    def send_welcome_email(
        self, 
        email: str, 
        name: str, 
        password: str = None, 
        title: str = None, 
        shift: str = None, 
        login_url: str = None, 
        credential_view_url: str = None
    ) -> Dict[str, any]:
        """
        Send welcome email to new user
        
        Args:
            email: User's email address
            name: User's full name
            password: Plain text password (optional - use credential_view_url instead for security)
            title: User's role/title
            shift: User's shift
            login_url: Login page URL
            credential_view_url: Secure one-time link to view credentials (preferred over plain text password)
        
        Returns:
            Dict with 'success' and 'message' keys
        """
        if not login_url:
            login_url = settings.login_url
        
        context = {
            'name': name,
            'email': email,
            'password': password or '',
            'title': title or '',
            'shift': shift or '',
            'login_url': login_url,
            'credential_view_url': credential_view_url or ''
        }
        
        message = self.load_email_template('welcome', context)
        subject = 'Welcome to Krystal EA System - Account Created'
        
        return self.send_email(email, subject, message)
    
    def send_password_reset_email(
        self, 
        email: str, 
        name: str, 
        reset_url: str, 
        expiry_hours: int = 1
    ) -> Dict[str, any]:
        """
        Send password reset email
        
        Args:
            email: Recipient email
            name: User's name
            reset_url: Password reset URL with token
            expiry_hours: Hours until token expires
        
        Returns:
            Dict with 'success' and 'message' keys
        """
        context = {
            'name': name,
            'reset_url': reset_url,
            'expiry_hours': expiry_hours
        }
        
        message = self.load_email_template('password_reset', context)
        subject = 'Password Reset Request - Krystal EA System'
        
        return self.send_email(email, subject, message)
    
    def send_password_reset_confirmation_email(self, email: str, name: str) -> Dict[str, any]:
        """
        Send password reset confirmation email
        
        Args:
            email: User's email address
            name: User's name
        
        Returns:
            Dict with 'success' and 'message' keys
        """
        context = {
            'name': name,
            'email': email
        }
        
        # Create simple confirmation message
        message = f"""
        <html>
        <head>
            <style>
                body {{ font-family: Arial, sans-serif; line-height: 1.6; color: #333; }}
                .container {{ max-width: 600px; margin: 0 auto; padding: 20px; }}
                .header {{ background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }}
                .content {{ background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }}
                .success {{ background: #d4edda; border-left: 4px solid #28a745; padding: 15px; margin: 20px 0; border-radius: 5px; }}
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1>Password Reset Successful</h1>
                </div>
                <div class="content">
                    <p>Hello {context['name']},</p>
                    
                    <div class="success">
                        <strong>✓ Password Successfully Reset</strong>
                        <p style="margin: 10px 0 0 0;">Your password has been changed successfully. You can now log in with your new password.</p>
                    </div>
                    
                    <p><strong>What to do next:</strong></p>
                    <ol style="padding-left: 20px;">
                        <li>Go to the login page and enter your email address</li>
                        <li>Enter your new password that you just set</li>
                        <li>You should now be able to access your account</li>
                    </ol>
                    
                    <div class="warning" style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin: 20px 0; border-radius: 5px;">
                        <strong>⚠️ Security Alert:</strong> If you did not request this password reset, please contact your system administrator immediately. Your account may have been compromised.
                    </div>
                    
                    <p><strong>Need Help?</strong> If you have any questions or encounter any issues logging in, please contact your system administrator or IT support team.</p>
                    
                    <p>Best regards,<br><strong>Krystal EA System Team</strong></p>
                </div>
            </div>
        </body>
        </html>
        """
        
        subject = 'Password Reset Confirmation - Krystal EA System'
        
        return self.send_email(email, subject, message)
    
    def load_email_template(self, template_name: str, context: Dict) -> str:
        """
        Load and render an email template
        
        Args:
            template_name: Name of template file (without .php extension)
            context: Dictionary of variables to use in template
        
        Returns:
            Rendered HTML email content
        """
        template_path = TEMPLATES_DIR / f"{template_name}.php"
        
        if not template_path.exists():
            return self.get_fallback_template(template_name, context)
        
        try:
            # Read template file
            with open(template_path, 'r', encoding='utf-8') as f:
                template_content = f.read()
            
            # Simple template variable replacement
            import re
            
            # Replace PHP echo statements with Python format placeholders
            def replace_php_var(match):
                var_name = match.group(1)
                return f"{{{var_name}}}"
            
            # Pattern to match: <?php echo htmlspecialchars($var ?? ''); ?>
            pattern = r'\<\?php\s+echo\s+htmlspecialchars\(\$(\w+)\s+\?\?\s+\'\'\);\s+\?\>'
            template_content = re.sub(pattern, replace_php_var, template_content)
            
            # Also handle simple <?php echo $var; ?> patterns
            pattern2 = r'\<\?php\s+echo\s+\$(\w+);\s+\?\>'
            template_content = re.sub(pattern2, replace_php_var, template_content)
            
            # Replace date('Y') with current year
            template_content = re.sub(r'\<\?php\s+echo\s+date\(\'Y\'\);\s+\?\>', str(datetime.now().year), template_content)
            
            # Format with context - ensure all None values are converted to empty strings
            formatted_context = {k: (v if v is not None else '') for k, v in context.items()}
            
            return template_content.format(**formatted_context)
            
        except Exception as e:
            logger.error(f"Error loading email template {template_name}: {e}")
            return self.get_fallback_template(template_name, context)
    
    def get_fallback_template(self, template_name: str, context: Dict) -> str:
        """Generate fallback email template if file doesn't exist"""
        if template_name == 'welcome':
            return f"""
            <h2>Welcome to Krystal EA System!</h2>
            <p>Hello {context.get('name', 'User')},</p>
            <p>Your account has been successfully created. Here are your login credentials:</p>
            <p><strong>Email:</strong> {context.get('email', '')}<br>
            <strong>Password:</strong> {context.get('password', '')}<br>
            <strong>Role:</strong> {context.get('title', '')}<br>
            <strong>Shift:</strong> {context.get('shift', '')}</p>
            <p>Please change your password after your first login.</p>
            <p>Login at: <a href="{context.get('login_url', '')}">{context.get('login_url', '')}</a></p>
            """
        elif template_name == 'password_reset':
            return f"""
            <h2>Password Reset Request</h2>
            <p>Hello {context.get('name', 'User')},</p>
            <p>We received a request to reset your password. Click the link below:</p>
            <p><a href="{context.get('reset_url', '')}">Reset Password</a></p>
            <p>This link will expire in {context.get('expiry_hours', 1)} hour(s).</p>
            <p>If you did not request this, please ignore this email.</p>
            """
        else:
            return f"<p>{context.get('message', 'Email from Krystal EA System')}</p>"


# Global email service instance
email_service = EmailService()

