"""
Create database schema and populate with dummy data for SAP Simulator.
This allows the simulator to use a database instead of hardcoded dictionaries.
"""

import mysql.connector
from mysql.connector import Error
from datetime import date, timedelta
from decimal import Decimal
import os
from pathlib import Path
from dotenv import load_dotenv

# Load environment variables
env_path = Path(__file__).parent / '.env'
load_dotenv(dotenv_path=env_path)

# Database configuration
DB_CONFIG = {
    'host': os.getenv('DB_HOST', 'localhost'),
    'user': os.getenv('DB_USER', 'root'),
    'password': os.getenv('DB_PASSWORD', ''),
    'database': os.getenv('SAP_SIMULATOR_DB_NAME', 'sap_simulator_db'),
    'port': int(os.getenv('DB_PORT', 3306))
}


def create_database():
    """Create the SAP simulator database if it doesn't exist."""
    try:
        # Connect without database
        config = DB_CONFIG.copy()
        db_name = config.pop('database')
        
        conn = mysql.connector.connect(**config)
        cursor = conn.cursor()
        
        # Create database
        cursor.execute(f"CREATE DATABASE IF NOT EXISTS `{db_name}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci")
        print(f"[OK] Database '{db_name}' created or already exists")
        
        cursor.close()
        conn.close()
        
        return True
    except Error as e:
        print(f"[ERROR] Error creating database: {e}")
        return False


def create_tables():
    """Create tables for SAP simulator data."""
    try:
        conn = mysql.connector.connect(**DB_CONFIG)
        cursor = conn.cursor()
        
        # Check if tables already exist
        cursor.execute("SHOW TABLES")
        existing_tables = [table[0] for table in cursor.fetchall()]
        
        if 'sap_material_plants' in existing_tables and 'sap_process_orders' in existing_tables and 'sap_order_components' in existing_tables:
            print("[INFO] Tables already exist, skipping creation")
            cursor.close()
            conn.close()
            return True
        
        # Drop existing tables if they exist (to avoid read-only issues)
        if existing_tables:
            try:
                for table in ['sap_order_components', 'sap_process_orders', 'sap_material_plants']:
                    if table in existing_tables:
                        cursor.execute(f"DROP TABLE IF EXISTS `{table}`")
                print("[INFO] Dropped existing tables (if any)")
                conn.commit()
            except Error as e:
                print(f"[WARN] Could not drop existing tables: {e}")
                # Continue anyway
        
        # Material-Plant Combinations Table
        cursor.execute("""
            CREATE TABLE `sap_material_plants` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `material_number` VARCHAR(50) NOT NULL,
                `plant` VARCHAR(10) NOT NULL,
                `material_description` VARCHAR(255),
                `material_type` VARCHAR(10) DEFAULT 'FERT',
                `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY `unique_material_plant` (`material_number`, `plant`),
                INDEX `idx_material` (`material_number`),
                INDEX `idx_plant` (`plant`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        """)
        print("[OK] Table 'sap_material_plants' created")
        
        # Process Orders Table
        cursor.execute("""
            CREATE TABLE `sap_process_orders` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `order_number` VARCHAR(20) NOT NULL UNIQUE,
                `order_type` VARCHAR(10) DEFAULT 'KE33',
                `material_number` VARCHAR(50) NOT NULL,
                `plant` VARCHAR(10) NOT NULL,
                `total_order_quantity` DECIMAL(15, 3) NOT NULL,
                `base_unit_of_measure` VARCHAR(10) DEFAULT 'KG',
                `basic_start_date` DATE,
                `basic_finish_date` DATE,
                `material_description` VARCHAR(255),
                `material_type` VARCHAR(10) DEFAULT 'FERT',
                `status` JSON,
                `is_closed` BOOLEAN DEFAULT FALSE,
                `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX `idx_material_plant` (`material_number`, `plant`),
                INDEX `idx_order_number` (`order_number`),
                INDEX `idx_status` (`is_closed`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        """)
        print("[OK] Table 'sap_process_orders' created")
        
        # Process Order Components Table
        cursor.execute("""
            CREATE TABLE `sap_order_components` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `order_id` INT NOT NULL,
                `order_number` VARCHAR(20) NOT NULL,
                `item_number` VARCHAR(10) NOT NULL,
                `material_number` VARCHAR(50) NOT NULL,
                `material_description` VARCHAR(255),
                `requirements_quantity` DECIMAL(15, 3) NOT NULL,
                `base_unit_of_measure` VARCHAR(10) DEFAULT 'KG',
                `requirements_date` DATE,
                `plant` VARCHAR(10),
                `storage_location` VARCHAR(10) DEFAULT '0001',
                `batch_required` BOOLEAN DEFAULT TRUE,
                `needs_weighing` BOOLEAN DEFAULT TRUE,
                `variance_tolerance_upper` DECIMAL(15, 3),
                `variance_tolerance_lower` DECIMAL(15, 3),
                `variance_tolerance_unit` VARCHAR(10) DEFAULT 'KG',
                `variance_tolerance_percentage` DECIMAL(5, 2),
                `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (`order_id`) REFERENCES `sap_process_orders`(`id`) ON DELETE CASCADE,
                INDEX `idx_order` (`order_number`),
                INDEX `idx_order_id` (`order_id`),
                UNIQUE KEY `unique_order_item` (`order_number`, `item_number`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        """)
        print("[OK] Table 'sap_order_components' created")
        
        conn.commit()
        cursor.close()
        conn.close()
        
        return True
    except Error as e:
        print(f"[ERROR] Error creating tables: {e}")
        return False


def populate_dummy_data():
    """Populate database with dummy test data."""
    try:
        conn = mysql.connector.connect(**DB_CONFIG)
        cursor = conn.cursor()
        
        # Clear existing data
        cursor.execute("DELETE FROM `sap_order_components`")
        cursor.execute("DELETE FROM `sap_process_orders`")
        cursor.execute("DELETE FROM `sap_material_plants`")
        print("[INFO] Cleared existing test data")
        
        # Insert Material-Plant Combinations
        materials_data = [
            ("FG001234", "K013", "Vaseline Extra Strength", "FERT"),
            ("FG001234", "K012", "Vaseline Extra Strength", "FERT"),
            ("FG005678", "K002", "Product B", "FERT"),
            ("FG005678", "K013", "Product B", "FERT"),
            ("RM001234", "K013", "Sodium Carbonate", "ROH"),
            ("RM001234", "K012", "Sodium Carbonate", "ROH"),
            ("RM001234", "K002", "Sodium Carbonate", "ROH"),
            ("RM005678", "K013", "Linear Alkylbenzene", "ROH"),
            ("RM005678", "K002", "Linear Alkylbenzene", "ROH"),
            ("RM009999", "K013", "Water", "ROH"),
            ("RM009999", "K012", "Water", "ROH"),
        ]
        
        cursor.executemany("""
            INSERT INTO `sap_material_plants` 
            (material_number, plant, material_description, material_type)
            VALUES (%s, %s, %s, %s)
            ON DUPLICATE KEY UPDATE 
            material_description = VALUES(material_description),
            material_type = VALUES(material_type)
        """, materials_data)
        print(f"[OK] Inserted {len(materials_data)} material-plant combinations")
        
        # Insert Process Orders
        today = date.today()
        
        orders_data = [
            {
                "order_number": "000012345678",
                "order_type": "KE33",
                "material_number": "FG001234",
                "plant": "K013",
                "total_order_quantity": Decimal("1000.0"),
                "base_unit_of_measure": "KG",
                "basic_start_date": today,
                "basic_finish_date": today + timedelta(days=4),
                "material_description": "Vaseline Extra Strength",
                "material_type": "FERT",
                "status": '["REL"]',
                "is_closed": False
            },
            {
                "order_number": "000012345679",
                "order_type": "KE33",
                "material_number": "FG001234",
                "plant": "K013",
                "total_order_quantity": Decimal("500.0"),
                "base_unit_of_measure": "KG",
                "basic_start_date": today,
                "basic_finish_date": today + timedelta(days=3),
                "material_description": "Vaseline Extra Strength",
                "material_type": "FERT",
                "status": '["REL"]',
                "is_closed": False
            },
            {
                "order_number": "000012345680",
                "order_type": "KE33",
                "material_number": "FG005678",
                "plant": "K002",
                "total_order_quantity": Decimal("750.0"),
                "base_unit_of_measure": "KG",
                "basic_start_date": today,
                "basic_finish_date": today + timedelta(days=5),
                "material_description": "Product B",
                "material_type": "FERT",
                "status": '["REL"]',
                "is_closed": False
            },
        ]
        
        order_ids = []
        for order in orders_data:
            cursor.execute("""
                INSERT INTO `sap_process_orders`
                (order_number, order_type, material_number, plant, total_order_quantity,
                 base_unit_of_measure, basic_start_date, basic_finish_date,
                 material_description, material_type, status, is_closed)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)
            """, (
                order["order_number"], order["order_type"], order["material_number"],
                order["plant"], order["total_order_quantity"], order["base_unit_of_measure"],
                order["basic_start_date"], order["basic_finish_date"],
                order["material_description"], order["material_type"],
                order["status"], order["is_closed"]
            ))
            order_ids.append(cursor.lastrowid)
        
        print(f"[OK] Inserted {len(orders_data)} process orders")
        
        # Insert Order Components
        components_data = [
            # Order 000012345678 - FG001234 in K013
            {
                "order_id": order_ids[0],
                "order_number": "000012345678",
                "item_number": "0010",
                "material_number": "RM001234",
                "material_description": "Sodium Carbonate",
                "requirements_quantity": Decimal("25.5"),
                "base_unit_of_measure": "KG",
                "requirements_date": today,
                "plant": "K013",
                "storage_location": "0001",
                "batch_required": True,
                "needs_weighing": True,
                "variance_tolerance_upper": Decimal("26.0"),
                "variance_tolerance_lower": Decimal("25.0"),
                "variance_tolerance_unit": "KG",
                "variance_tolerance_percentage": Decimal("2.0")
            },
            {
                "order_id": order_ids[0],
                "order_number": "000012345678",
                "item_number": "0020",
                "material_number": "RM005678",
                "material_description": "Linear Alkylbenzene",
                "requirements_quantity": Decimal("15.2"),
                "base_unit_of_measure": "KG",
                "requirements_date": today,
                "plant": "K013",
                "storage_location": "0001",
                "batch_required": True,
                "needs_weighing": True,
                "variance_tolerance_upper": Decimal("15.5"),
                "variance_tolerance_lower": Decimal("15.0"),
                "variance_tolerance_unit": "KG",
                "variance_tolerance_percentage": Decimal("2.0")
            },
            {
                "order_id": order_ids[0],
                "order_number": "000012345678",
                "item_number": "0030",
                "material_number": "RM009999",
                "material_description": "Water",
                "requirements_quantity": Decimal("10.0"),
                "base_unit_of_measure": "KG",
                "requirements_date": today,
                "plant": "K013",
                "storage_location": "0001",
                "batch_required": False,
                "needs_weighing": False,  # Manual entry
                "variance_tolerance_upper": Decimal("10.5"),
                "variance_tolerance_lower": Decimal("9.5"),
                "variance_tolerance_unit": "KG",
                "variance_tolerance_percentage": Decimal("5.0")
            },
            # Order 000012345679 - FG001234 in K013 (smaller order)
            {
                "order_id": order_ids[1],
                "order_number": "000012345679",
                "item_number": "0010",
                "material_number": "RM001234",
                "material_description": "Sodium Carbonate",
                "requirements_quantity": Decimal("12.75"),
                "base_unit_of_measure": "KG",
                "requirements_date": today,
                "plant": "K013",
                "storage_location": "0001",
                "batch_required": True,
                "needs_weighing": True,
                "variance_tolerance_upper": Decimal("13.0"),
                "variance_tolerance_lower": Decimal("12.5"),
                "variance_tolerance_unit": "KG",
                "variance_tolerance_percentage": Decimal("2.0")
            },
            {
                "order_id": order_ids[1],
                "order_number": "000012345679",
                "item_number": "0020",
                "material_number": "RM005678",
                "material_description": "Linear Alkylbenzene",
                "requirements_quantity": Decimal("7.6"),
                "base_unit_of_measure": "KG",
                "requirements_date": today,
                "plant": "K013",
                "storage_location": "0001",
                "batch_required": True,
                "needs_weighing": True,
                "variance_tolerance_upper": Decimal("7.8"),
                "variance_tolerance_lower": Decimal("7.4"),
                "variance_tolerance_unit": "KG",
                "variance_tolerance_percentage": Decimal("2.0")
            },
            # Order 000012345680 - FG005678 in K002
            {
                "order_id": order_ids[2],
                "order_number": "000012345680",
                "item_number": "0010",
                "material_number": "RM001234",
                "material_description": "Sodium Carbonate",
                "requirements_quantity": Decimal("30.0"),
                "base_unit_of_measure": "KG",
                "requirements_date": today,
                "plant": "K002",
                "storage_location": "0001",
                "batch_required": True,
                "needs_weighing": True,
                "variance_tolerance_upper": Decimal("31.0"),
                "variance_tolerance_lower": Decimal("29.0"),
                "variance_tolerance_unit": "KG",
                "variance_tolerance_percentage": Decimal("3.3")
            },
        ]
        
        cursor.executemany("""
            INSERT INTO `sap_order_components`
            (order_id, order_number, item_number, material_number, material_description,
             requirements_quantity, base_unit_of_measure, requirements_date, plant,
             storage_location, batch_required, needs_weighing,
             variance_tolerance_upper, variance_tolerance_lower,
             variance_tolerance_unit, variance_tolerance_percentage)
            VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)
        """, [
            (
                comp["order_id"], comp["order_number"], comp["item_number"],
                comp["material_number"], comp["material_description"],
                comp["requirements_quantity"], comp["base_unit_of_measure"],
                comp["requirements_date"], comp["plant"], comp["storage_location"],
                comp["batch_required"], comp["needs_weighing"],
                comp["variance_tolerance_upper"], comp["variance_tolerance_lower"],
                comp["variance_tolerance_unit"], comp["variance_tolerance_percentage"]
            )
            for comp in components_data
        ])
        
        print(f"[OK] Inserted {len(components_data)} order components")
        
        conn.commit()
        cursor.close()
        conn.close()
        
        return True
    except Error as e:
        print(f"[ERROR] Error populating data: {e}")
        return False


def main():
    """Main function to set up SAP simulator database."""
    print("=" * 60)
    print("SAP Simulator Database Setup")
    print("=" * 60)
    print()
    
    print("Step 1: Creating database...")
    if not create_database():
        print("[ERROR] Failed to create database")
        return
    
    print()
    print("Step 2: Creating tables...")
    if not create_tables():
        print("[ERROR] Failed to create tables")
        return
    
    print()
    print("Step 3: Populating dummy data...")
    if not populate_dummy_data():
        print("[ERROR] Failed to populate data")
        return
    
    print()
    print("=" * 60)
    print("[OK] SAP Simulator database setup completed successfully!")
    print("=" * 60)
    print()
    print("Database Configuration:")
    print(f"  Host: {DB_CONFIG['host']}")
    print(f"  Database: {DB_CONFIG['database']}")
    print(f"  Port: {DB_CONFIG['port']}")
    print()
    print("Test Data Summary:")
    print("  - Materials: FG001234, FG005678, RM001234, RM005678, RM009999")
    print("  - Plants: K013, K012, K002")
    print("  - Process Orders: 3 orders with multiple components")
    print()
    print("Next Steps:")
    print("  1. Update sap_simulator.py to use database")
    print("  2. Restart SAP simulator")
    print("  3. Test with Android app")


if __name__ == "__main__":
    main()

