"""
Shared logic for building one batch insert row from raw request payload.
Used by both the test app (test_submit_app.py) and the main app (batch_service).
Single source of truth so expected_weight and comment are never dropped.
"""
from typing import Dict, Any, Optional


def build_row_from_payload(
    raw_row: Dict[str, Any],
    payload: Dict[str, Any],
    batch_metadata: Dict[str, Any],
    expected_weights_fallback: Optional[Dict[str, str]] = None,
) -> Optional[Dict[str, Any]]:
    """
    Build one insert row from raw request row. Same logic as test app.
    expected_weight and comment are taken only from raw_row (snake or camel).
    Returns None if item/actual_weight missing; otherwise dict ready for INSERT.
    """
    item_name = (raw_row.get("item") or "").strip()
    actual_weight = str(raw_row.get("actual_weight") or "").strip()
    if not item_name or not actual_weight:
        return None

    # Same as test app: expected_weight from request row only
    expected_weight = raw_row.get("expected_weight") or raw_row.get("expectedWeight")
    if expected_weight is not None:
        expected_weight = str(expected_weight).strip()
    else:
        fallback = expected_weights_fallback or {}
        expected_weight = (fallback.get(item_name) or "") if item_name else ""

    # Same as test app: comment from request row only
    comment = raw_row.get("comment")
    if comment is None or (isinstance(comment, str) and not comment.strip()):
        comment = None
    else:
        comment = str(comment).strip() or None

    batch_no = f"{batch_metadata['scan_date']}_{batch_metadata['scan_time'].replace(':', '')}"
    product = (payload.get("product") or "").strip()
    operator = (payload.get("operator") or "").strip()
    analyst = (payload.get("analyst") or "").strip()
    team_lead = (payload.get("teamLead") or "").strip()
    factory = (payload.get("factory") or "").strip()

    row = {
        "product": product,
        "items": item_name,
        "batch_no": batch_no,
        "expected_weight": expected_weight,
        "actual_weight": actual_weight,
        "scan_time": batch_metadata["scan_time"],
        "operator": operator,
        "analyst": analyst,
        "scan_date": batch_metadata["scan_date"],
        "teamLead": team_lead,
        "mdn": str(raw_row.get("mdn") or "").strip(),
        "date_manufacturer": raw_row.get("date_manufacturer") or None,
        "expiry_date": raw_row.get("expiry_date") or None,
        "factory": factory,
        "comment": comment,
    }
    return row


def main_app_extra_fields(batch_metadata: Dict[str, Any]) -> Dict[str, Any]:
    """Extra keys required for main app `data` table (week, units, serial_no, status, shift, analyst_status, tl_status)."""
    return {
        "week": batch_metadata.get("week", ""),
        "units": "kg",
        "serial_no": batch_metadata.get("serial_no", ""),
        "status": "1",
        "shift": batch_metadata.get("shift", ""),
        "analyst_status": "0",
        "tl_status": "0",
    }
