"""
SAP Integration Service for handling inbound and outbound interfaces.
Manages communication with SAP ECC via BTP middleware.
"""

import logging
import uuid
from datetime import datetime, date
from typing import Dict, List, Optional, Any
import json
import httpx
from decimal import Decimal

from app.core.config import get_settings
from app.core.exceptions import (
    BusinessLogicError,
    ValidationError,
    ExternalServiceError,
    ResourceNotFoundError
)
from app.models.sap_schemas import (
    MaterialTriggerRequest,
    MaterialTriggerResponse,
    MAK080ConsumptionRequest,
    MAK080ConsumptionResponse,
    ProcessOrderResponse,
    ALEAUDAcknowledgment,
    SAPErrorResponse
)

logger = logging.getLogger(__name__)
settings = get_settings()


class SAPIntegrationService:
    """Service for SAP ECC integration via BTP."""
    
    def __init__(self):
        self.btp_base_url = getattr(settings, 'sap_btp_base_url', 'https://your-btp-endpoint.com')
        self.btp_client_id = getattr(settings, 'sap_btp_client_id', '')
        self.btp_client_secret = getattr(settings, 'sap_btp_client_secret', '')
        # Reduced timeout for faster failure detection (5 seconds for simulator, 30 for production)
        # Simulator should respond quickly, production SAP might need longer
        self.timeout = 5 if 'localhost' in str(self.btp_base_url) or '127.0.0.1' in str(self.btp_base_url) else 30
        
        # Valid plants for Kenya
        self.valid_plants = ['K002', 'K013', 'K012']
        
        # Valid order types per plant (from Z2XPR1 table)
        self.valid_order_types = {
            'K013': ['KE33'],
            'K012': ['KE23', 'KE24'],
            'K002': ['KE71', 'KE72']
        }
    
    async def trigger_material_request(self, request: MaterialTriggerRequest) -> MaterialTriggerResponse:
        """
        Send material trigger request to SAP to fetch process orders.
        
        This corresponds to the 1st Inbound Interface: Material Trigger.
        
        Args:
            request: Material trigger request data
            
        Returns:
            MaterialTriggerResponse with request status
            
        Raises:
            ValidationError: If material/plant combination is invalid
            ExternalServiceError: If SAP communication fails
        """
        try:
            # Generate unique request ID
            request_id = f"REQ-{datetime.now().strftime('%Y-%m-%d')}-{str(uuid.uuid4())[:8]}"
            
            logger.info(f"Triggering material request for {request.material_number} in plant {request.plant}")
            
            # Validate plant
            if request.plant not in self.valid_plants:
                raise ValidationError(f"Invalid plant: {request.plant}. Valid plants: {self.valid_plants}")
            
            # Prepare payload for BTP
            payload = {
                "message_type": "MATERIAL_TRIGGER",
                "request_id": request_id,
                "timestamp": datetime.utcnow().isoformat(),
                "data": {
                    "material_number": request.material_number,
                    "plant": request.plant
                }
            }
            
            # Send to BTP
            response_data = await self._send_to_btp("/inbound/material-trigger", payload)
            
            # Check if SAP validation passed
            if not response_data.get("success", False):
                error_code = response_data.get("error_code", "UNKNOWN_ERROR")
                error_message = response_data.get("error_message", "Unknown error occurred")
                
                logger.warning(f"SAP validation failed: {error_code} - {error_message}")
                
                return MaterialTriggerResponse(
                    success=False,
                    message=error_message,
                    material_number=request.material_number,
                    plant=request.plant,
                    request_id=request_id,
                    error_code=error_code
                )
            
            logger.info(f"Material trigger successful for request {request_id}")
            
            return MaterialTriggerResponse(
                success=True,
                message="Material trigger sent to SAP successfully. Process orders will be sent shortly.",
                material_number=request.material_number,
                plant=request.plant,
                request_id=request_id
            )
            
        except ValidationError:
            raise
        except Exception as e:
            logger.error(f"Error in material trigger request: {e}")
            raise ExternalServiceError(f"Failed to send material trigger to SAP: {str(e)}")
    
    async def send_component_consumption(self, request: MAK080ConsumptionRequest) -> MAK080ConsumptionResponse:
        """
        Send component consumption data to SAP (MAK080 interface).
        
        This corresponds to the 2nd Inbound Interface: MAK080 Process order component consumption.
        
        Args:
            request: Component consumption request data
            
        Returns:
            MAK080ConsumptionResponse with processing status
            
        Raises:
            ValidationError: If request data is invalid
            ExternalServiceError: If SAP communication fails
        """
        try:
            # Generate unique request ID
            request_id = f"MAK080-{datetime.now().strftime('%Y-%m-%d')}-{str(uuid.uuid4())[:8]}"
            
            logger.info(f"Sending component consumption with {len(request.components)} components")
            
            # Validate all plants in components
            for component in request.components:
                if component.plant not in self.valid_plants:
                    raise ValidationError(f"Invalid plant in component: {component.plant}")
            
            # Prepare MAK080 payload for BTP
            payload = {
                "message_type": "MAK080",
                "request_id": request_id,
                "timestamp": datetime.utcnow().isoformat(),
                "data": {
                    "document_date": request.document_date.strftime("%Y%m%d"),
                    "posting_date": request.posting_date.strftime("%Y%m%d"),
                    "transaction_code": request.transaction_code,
                    "description": request.description or "Component consumption from Krystal EA",
                    "components": [
                        {
                            "material_number": comp.material_number,
                            "plant": comp.plant,
                            "storage_location": comp.storage_location,
                            "batch_number": comp.batch_number,
                            "movement_type": comp.movement_type,
                            "quantity": float(comp.quantity),
                            "unit_of_measure": comp.unit_of_measure,
                            "order_number": comp.order_number
                        }
                        for comp in request.components
                    ]
                }
            }
            
            # Send to BTP
            response_data = await self._send_to_btp("/inbound/mak080-consumption", payload)
            
            # Extract IDOC number if available
            idoc_number = response_data.get("idoc_number")
            
            if not response_data.get("success", False):
                error_code = response_data.get("error_code", "PROCESSING_ERROR")
                error_message = response_data.get("error_message", "Failed to process component consumption")
                
                logger.warning(f"MAK080 processing failed: {error_code} - {error_message}")
                
                return MAK080ConsumptionResponse(
                    success=False,
                    message=error_message,
                    idoc_number=idoc_number,
                    components_count=len(request.components),
                    request_id=request_id,
                    error_code=error_code
                )
            
            logger.info(f"MAK080 consumption successful for request {request_id}, IDOC: {idoc_number}")
            
            return MAK080ConsumptionResponse(
                success=True,
                message="Component consumption sent to SAP successfully",
                idoc_number=idoc_number,
                components_count=len(request.components),
                request_id=request_id
            )
            
        except ValidationError:
            raise
        except Exception as e:
            logger.error(f"Error in MAK080 consumption: {e}")
            raise ExternalServiceError(f"Failed to send component consumption to SAP: {str(e)}")
    
    async def receive_process_orders(self, payload: Dict[str, Any]) -> ProcessOrderResponse:
        """
        Receive process orders from SAP (1st Outbound Interface).
        
        This handles the Process Order Interface from SAP ECC to Krystal via BTP.
        
        Args:
            payload: Process order data from SAP via BTP
            
        Returns:
            ProcessOrderResponse with received data
        """
        try:
            logger.info("Receiving process orders from SAP")
            
            # Extract basic information
            material_number = payload.get("material_number", "")
            plant = payload.get("plant", "")
            process_orders_data = payload.get("process_orders", [])
            
            # Validate the response structure
            if not material_number or not plant:
                raise ValidationError("Missing material_number or plant in SAP response")
            
            # Process the orders data
            process_orders = []
            for order_data in process_orders_data:
                # Here you would parse the complex IDOC structure
                # For now, we'll create a simplified structure
                
                header_data = order_data.get("header", {})
                material_data = order_data.get("material_data", {})
                components_data = order_data.get("components", [])
                
                # You would implement full IDOC parsing here
                # This is a simplified version
                process_orders.append({
                    "header": header_data,
                    "material_data": material_data,
                    "components": components_data,
                    "status": order_data.get("status", [])
                })
            
            logger.info(f"Received {len(process_orders)} process orders for {material_number} in {plant}")
            
            return ProcessOrderResponse(
                success=True,
                message=f"Received {len(process_orders)} process orders successfully",
                material_number=material_number,
                plant=plant,
                process_orders=process_orders
            )
            
        except Exception as e:
            logger.error(f"Error processing received process orders: {e}")
            raise BusinessLogicError(f"Failed to process process orders from SAP: {str(e)}")
    
    async def receive_aleaud_acknowledgment(self, payload: Dict[str, Any]) -> ALEAUDAcknowledgment:
        """
        Receive ALEAUD acknowledgment from SAP (2nd Outbound Interface).
        
        This handles the Aleaud Interface – Acknowledgement response from SAP ECC to Krystal via BTP.
        
        Args:
            payload: ALEAUD acknowledgment data from SAP via BTP
            
        Returns:
            ALEAUDAcknowledgment with status information
        """
        try:
            logger.info("Receiving ALEAUD acknowledgment from SAP")
            
            # Extract ALEAUD data
            message_type = payload.get("message_type", "")
            idoc_number = payload.get("idoc_number", "")
            status = payload.get("status", "")
            status_code = payload.get("status_code", "")
            status_text = payload.get("status_text", "")
            status_type = payload.get("status_type", "")
            
            # Determine success based on status
            success = status_type in ["S", "I"]  # Success or Information
            error_message = None if success else status_text
            
            logger.info(f"ALEAUD acknowledgment: IDOC {idoc_number}, Status {status}, Success: {success}")
            
            return ALEAUDAcknowledgment(
                message_type=message_type,
                message_code=payload.get("message_code"),
                message_function=payload.get("message_function"),
                idoc_number=idoc_number,
                status=status,
                status_code=status_code,
                status_text=status_text,
                status_type=status_type,
                status_message_qualifier=payload.get("status_message_qualifier", "SAP"),
                status_message_id=payload.get("status_message_id", ""),
                status_message_number=payload.get("status_message_number", ""),
                parameters=payload.get("parameters", {}),
                plant=payload.get("plant"),
                pallet_number=payload.get("pallet_number"),
                success=success,
                error_message=error_message
            )
            
        except Exception as e:
            logger.error(f"Error processing ALEAUD acknowledgment: {e}")
            raise BusinessLogicError(f"Failed to process ALEAUD acknowledgment: {str(e)}")
    
    async def _send_to_btp(self, endpoint: str, payload: Dict[str, Any]) -> Dict[str, Any]:
        """
        Send data to SAP BTP middleware.
        
        Args:
            endpoint: BTP endpoint path
            payload: Data to send
            
        Returns:
            Response data from BTP
            
        Raises:
            ExternalServiceError: If BTP communication fails
        """
        try:
            url = f"{self.btp_base_url}{endpoint}"
            
            headers = {
                "Content-Type": "application/json"
            }
            
            # Only add Authorization header if BTP client credentials are configured
            # SAP simulator doesn't require authentication
            if self.btp_client_id and self.btp_client_secret:
                headers["Authorization"] = f"Bearer {await self._get_btp_token()}"
            
            # Use shorter timeout for connection and read operations
            # Connection timeout: 2 seconds (how long to wait to establish connection)
            # Read timeout: self.timeout (how long to wait for response)
            timeout_config = httpx.Timeout(
                connect=2.0,  # Fast connection timeout
                read=self.timeout,  # Read timeout
                write=5.0,  # Write timeout
                pool=5.0  # Pool timeout
            )
            
            async with httpx.AsyncClient(timeout=timeout_config) as client:
                response = await client.post(url, json=payload, headers=headers)
                response.raise_for_status()
                
                return response.json()
                
        except httpx.HTTPStatusError as e:
            logger.error(f"BTP HTTP error: {e.response.status_code} - {e.response.text}")
            raise ExternalServiceError(f"BTP communication failed: {e.response.status_code}")
        except httpx.TimeoutException as e:
            logger.error(f"BTP timeout: {e} (url={url})")
            raise ExternalServiceError(f"BTP request timed out: {str(e)}")
        except httpx.RequestError as e:
            logger.error(f"BTP request error: {e} (url={url})")
            raise ExternalServiceError(f"Failed to connect to BTP: {str(e)}")
        except Exception as e:
            logger.error(f"Unexpected BTP error: {e}")
            raise ExternalServiceError(f"Unexpected BTP error: {str(e)}")
    
    async def _get_btp_token(self) -> str:
        """
        Get authentication token for BTP.
        
        Returns:
            Bearer token for BTP authentication
        """
        # This would implement OAuth2 token retrieval from BTP
        # For now, return a placeholder
        return "your-btp-oauth-token"
    
    def validate_material_plant_combination(self, material_number: str, plant: str) -> bool:
        """
        Validate material and plant combination.
        
        In a real implementation, this would check against MARC table in SAP.
        For now, we'll do basic validation.
        
        Args:
            material_number: Material number to validate
            plant: Plant to validate
            
        Returns:
            True if combination is valid
        """
        # Basic validation
        if not material_number or not plant:
            return False
        
        if plant not in self.valid_plants:
            return False
        
        # In real implementation, you would:
        # 1. Query SAP MARC table via BTP
        # 2. Check if material exists in the plant
        # 3. Validate material type and status
        
        return True
    
    def get_valid_order_types(self, plant: str) -> List[str]:
        """
        Get valid order types for a plant based on Z2XPR1 configuration.
        
        Args:
            plant: Plant code
            
        Returns:
            List of valid order types for the plant
        """
        return self.valid_order_types.get(plant, [])


# Create service instance
sap_service = SAPIntegrationService()