"""
Login logs service for business logic related to authentication logging.
Handles login event logging, reporting, and security monitoring.
"""

from typing import Dict, List, Optional, Any
import logging
from datetime import datetime

from app.repositories.login_logs_repository import login_logs_repository
from app.repositories.user_repository import user_repository
from app.core.exceptions import (
    ValidationError,
    BusinessLogicError,
    AuthorizationError
)

logger = logging.getLogger(__name__)


class LoginLogsService:
    """Service class for login logs business logic."""
    
    def __init__(self):
        self.login_logs_repo = login_logs_repository
        self.user_repo = user_repository
    
    def log_login_success(
        self,
        user_id: str,
        email: str,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Log a successful login attempt.
        
        Args:
            user_id: User's unique identifier
            email: User's email address
            ip_address: Client IP address
            user_agent: Client user agent string
            
        Returns:
            Created log entry
        """
        try:
            return self.login_logs_repo.log_login_attempt(
                user_id=user_id,
                email=email,
                status='success',
                message='Login successful',
                ip_address=ip_address,
                user_agent=user_agent
            )
        except Exception as e:
            logger.error(f"Error logging successful login for {email}: {e}")
            raise BusinessLogicError(f"Failed to log login success: {str(e)}")
    
    def log_login_failure(
        self,
        email: str,
        reason: str = 'Invalid credentials',
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Log a failed login attempt.
        
        Args:
            email: User's email address
            reason: Reason for failure
            ip_address: Client IP address
            user_agent: Client user agent string
            
        Returns:
            Created log entry
        """
        try:
            return self.login_logs_repo.log_login_attempt(
                user_id=None,  # No user_id for failed logins
                email=email,
                status='failure',
                message=reason,
                ip_address=ip_address,
                user_agent=user_agent
            )
        except Exception as e:
            logger.error(f"Error logging failed login for {email}: {e}")
            raise BusinessLogicError(f"Failed to log login failure: {str(e)}")
    
    def log_logout(
        self,
        user_id: str,
        email: str,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Log a user logout.
        
        Args:
            user_id: User's unique identifier
            email: User's email address
            ip_address: Client IP address
            user_agent: Client user agent string
            
        Returns:
            Created log entry
        """
        try:
            return self.login_logs_repo.log_login_attempt(
                user_id=user_id,
                email=email,
                status='logout',
                message='User logged out',
                ip_address=ip_address,
                user_agent=user_agent
            )
        except Exception as e:
            logger.error(f"Error logging logout for {email}: {e}")
            raise BusinessLogicError(f"Failed to log logout: {str(e)}")
    
    def get_login_logs(
        self,
        requesting_user: Dict[str, Any],
        user_id: Optional[str] = None,
        email: Optional[str] = None,
        status: Optional[str] = None,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None,
        page: int = 1,
        limit: int = 50
    ) -> Dict[str, Any]:
        """
        Get login logs with role-based access control and pagination.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            user_id: Filter by user ID
            email: Filter by email
            status: Filter by status
            date_from: Start date filter
            date_to: End date filter
            page: Page number
            limit: Items per page
            
        Returns:
            Paginated login logs with metadata
        """
        try:
            # Access is already enforced at API level (admin_only_access dependency)
            
            # Get paginated logs
            result = self.login_logs_repo.get_login_logs_paginated(
                user_id=user_id,
                email=email,
                status=status,
                date_from=date_from,
                date_to=date_to,
                page=page,
                limit=limit
            )
            
            # Admin gets full access, no data sanitization needed
            items = result['items']
            
            # Calculate pagination metadata
            total = result['total']
            pages = (total + limit - 1) // limit
            has_next = page < pages
            has_prev = page > 1
            
            # Build user permissions for frontend
            user_permissions = self._get_user_permissions(requesting_user)
            
            return {
                "items": items,
                "pagination": {
                    "page": page,
                    "limit": limit,
                    "total": total,
                    "pages": pages,
                    "has_next": has_next,
                    "has_prev": has_prev
                },
                "user_permissions": user_permissions,
                "summary": {
                    "total_logs": total,
                    "filters_applied": {
                        "user_id": user_id,
                        "email": email,
                        "status": status,
                        "date_from": date_from,
                        "date_to": date_to
                    }
                }
            }
            
        except Exception as e:
            logger.error(f"Error getting login logs: {e}")
            raise BusinessLogicError(f"Failed to get login logs: {str(e)}")
    
    def get_login_statistics(
        self,
        requesting_user: Dict[str, Any],
        date_from: Optional[str] = None,
        date_to: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Get login statistics with role-based access control.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            date_from: Start date filter
            date_to: End date filter
            
        Returns:
            Login statistics
        """
        try:
            # Access is already enforced at API level (admin_only_access dependency)
            
            # Get statistics
            stats = self.login_logs_repo.get_login_statistics(date_from, date_to)
            
            return stats
            
        except Exception as e:
            logger.error(f"Error getting login statistics: {e}")
            raise BusinessLogicError(f"Failed to get login statistics: {str(e)}")
    
    def get_user_login_history(
        self,
        requesting_user: Dict[str, Any],
        target_user_id: Optional[str] = None,
        limit: int = 20
    ) -> List[Dict[str, Any]]:
        """
        Get login history for a specific user.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            target_user_id: Target user ID (None for own history)
            limit: Maximum number of records
            
        Returns:
            List of login history records
        """
        try:
            # Manual admin check
            user_role = requesting_user.get('title', '').lower()
            if user_role not in ['admin', 'super_admin']:
                from app.core.exceptions import AuthorizationError
                raise AuthorizationError("Login history access requires admin or super_admin role")
            
            requesting_user_id = requesting_user.get('id')
            
            # Determine target user
            if target_user_id is None:
                target_user_id = requesting_user_id
            
            # Ensure user_id is a string (database expects varchar)
            if target_user_id is not None:
                target_user_id = str(target_user_id)
            
            # Get login history from repository
            history = self.login_logs_repo.get_user_login_history(target_user_id, limit)
            
            return history
            
        except Exception as e:
            logger.error(f"Error getting user login history: {e}")
            raise BusinessLogicError(f"Failed to get user login history: {str(e)}")
    
    def _get_user_permissions(self, user: Dict[str, Any]) -> Dict[str, Any]:
        """Get user permissions for frontend UI control."""
        user_role = user.get('title', '').lower()
        
        # Since login logs are admin-only, all users here are admin/super_admin
        return {
            "can_view_all_logs": True,
            "can_view_statistics": True,
            "can_export_logs": True,
            "role": user_role,
            "is_admin": True,
            "is_management": True
        }
    
    def create_login_log(
        self,
        requesting_user: Dict[str, Any],
        user_id: Optional[str],
        email: str,
        status: str,
        message: str,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Create a new login log record.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            user_id: User's unique identifier (None for failed logins)
            email: User's email address
            status: Login status (success, failure, logout)
            message: Log message
            ip_address: Client IP address (optional)
            user_agent: Client user agent (optional)
            
        Returns:
            Created log record details
        """
        try:
            # Admin-only access control
            user_role = requesting_user.get('title', '').lower()
            if user_role not in ['admin', 'super_admin']:
                raise AuthorizationError("Creating login logs requires admin or super_admin role")
            
            # Validate status
            valid_statuses = ['success', 'failure', 'logout']
            if status not in valid_statuses:
                raise ValidationError(f"Invalid status. Must be one of: {', '.join(valid_statuses)}")
            
            # Validate user_id if provided (for successful logins and logouts)
            if user_id is not None:
                if not self.user_repo.user_exists(user_id):
                    raise ValidationError(f"User with ID '{user_id}' does not exist")
                
                # For successful logins and logouts, user_id should be provided
                if status in ['success', 'logout'] and not user_id:
                    raise ValidationError(f"user_id is required for status '{status}'")
            
            # For failed logins, user_id should typically be None
            if status == 'failure' and user_id is not None:
                # Still validate the user exists if provided
                if not self.user_repo.user_exists(user_id):
                    raise ValidationError(f"User with ID '{user_id}' does not exist")
            
            # Create the log record
            result = self.login_logs_repo.create_login_log(
                user_id=user_id,
                email=email,
                status=status,
                message=message,
                ip_address=ip_address,
                user_agent=user_agent
            )
            
            return {
                'success': True,
                'message': 'Login log created successfully',
                'record_id': result.get('id'),
                'created_at': result.get('created_at'),
                'record': result.get('record')
            }
            
        except (AuthorizationError, ValidationError):
            raise
        except Exception as e:
            logger.error(f"Error creating login log: {e}")
            raise BusinessLogicError(f"Failed to create login log: {str(e)}")


# Global login logs service instance
login_logs_service = LoginLogsService()