"""
Authentication service for handling user authentication and authorization.
Provides JWT token management, password validation, and session handling.
"""

import secrets
import bcrypt
import jwt
from datetime import datetime, timezone, timedelta
from typing import Dict, Optional, Tuple
import logging

from app.core.config import get_settings
from app.core.exceptions import (
    InvalidCredentialsError, 
    TokenExpiredError, 
    InvalidTokenError,
    UserNotFoundError
)
from app.utils.database import db_manager
from app.models.schemas import LoginRequest, LoginResponse

logger = logging.getLogger(__name__)
settings = get_settings()


class AuthService:
    """Authentication service for user login, token management, and authorization."""
    
    def __init__(self):
        self.jwt_secret = settings.jwt_secret_key
        self.jwt_algorithm = settings.jwt_algorithm
        self.jwt_expiration_hours = settings.jwt_expiration_hours
    
    def authenticate_user(self, login_data: LoginRequest, ip_address: Optional[str] = None, user_agent: Optional[str] = None) -> LoginResponse:
        """
        Authenticate user with email and password.
        
        Args:
            login_data: Login request data
            ip_address: Client IP address for logging
            user_agent: Client user agent for logging
            
        Returns:
            LoginResponse with tokens and redirect URL
            
        Raises:
            InvalidCredentialsError: If credentials are invalid
            UserNotFoundError: If user doesn't exist
        """
        # Normalize email for case-insensitive matching
        normalized_email = login_data.email.strip().lower()
        
        try:
            # Get user from database
            user = self._get_user_by_email(normalized_email)
            if not user:
                logger.warning(f"Login attempt for non-existent user: {login_data.email}")
                # Log failed login attempt
                self._log_login_failure(login_data.email, "User not found", ip_address, user_agent)
                raise InvalidCredentialsError()
            
            # Validate password
            if not self._verify_password(login_data.password, user['password'], user['user_id']):
                logger.warning(f"Invalid password for user: {login_data.email}")
                # Log failed login attempt
                self._log_login_failure(login_data.email, "Invalid password", ip_address, user_agent)
                raise InvalidCredentialsError()
            
            # Create JWT token
            token_data = {
                'id': user['user_id'],
                'name': user['email'],
                'email': user['email'],
                'imglink': user.get('imglink', ''),
                'shift': user.get('shift', ''),
                'title': user.get('title', ''),
                'user_name': user.get('name', ''),
            }
            
            access_token = self._create_access_token(token_data)
            
            # Determine redirect URL based on user role
            redirect_url = self._get_redirect_url(user.get('title', ''))
            
            logger.info(f"Successful login for user: {user['email']} (ID: {user['user_id']})")
            
            # Log successful login
            self._log_login_success(user['user_id'], user['email'], ip_address, user_agent)
            
            return LoginResponse(
                success=True,
                redirect=redirect_url,
                message="Login successful",
                access_token=access_token,
                token_type="bearer"
            )
            
        except (InvalidCredentialsError, UserNotFoundError):
            # Re-raise authentication errors
            raise
        except Exception as e:
            logger.error(f"Unexpected error during authentication for {login_data.email}: {e}")
            # Log failed login attempt for unexpected errors
            self._log_login_failure(login_data.email, "System error", ip_address, user_agent)
            raise InvalidCredentialsError()
    
    def verify_token(self, token: str) -> Dict:
        """
        Verify and decode JWT token.
        
        Args:
            token: JWT token string
            
        Returns:
            Decoded token payload
            
        Raises:
            TokenExpiredError: If token has expired
            InvalidTokenError: If token is invalid
        """
        try:
            payload = jwt.decode(token, self.jwt_secret, algorithms=[self.jwt_algorithm])
            return payload
        except jwt.ExpiredSignatureError:
            raise TokenExpiredError()
        except jwt.JWTError as e:
            logger.warning(f"Invalid JWT token: {e}")
            raise InvalidTokenError()
    
    def get_current_user(self, authorization_header: Optional[str] = None) -> Dict:
        """
        Get current user from JWT token.
        
        Args:
            authorization_header: Authorization header with Bearer token
            
        Returns:
            User data dictionary
            
        Raises:
            InvalidTokenError: If no valid authentication found
        """
        if authorization_header and authorization_header.startswith("Bearer "):
            token = authorization_header.split(" ")[1]
            return self.verify_token(token)
        
        raise InvalidTokenError()
    
    def logout_user(self) -> bool:
        """
        Logout user (JWT tokens are stateless, so this is mainly for logging).
        
        Returns:
            True (always successful for JWT)
        """
        # With JWT tokens, logout is handled client-side by removing the token
        # Server-side logout would require a token blacklist (optional enhancement)
        logger.info("User logout requested (JWT token should be removed client-side)")
        return True
        
        return success
    
    def check_user_permissions(self, user: Dict, required_role: Optional[str] = None, required_area: Optional[str] = None) -> bool:
        """
        Check if user has required permissions.
        
        Args:
            user: User data dictionary
            required_role: Required role (optional)
            required_area: Required area access (optional)
            
        Returns:
            True if user has permissions
        """
        # Check role-based permissions
        if required_role:
            user_role = user.get('title', '').lower()
            
            # Admin and super_admin have access to everything
            if user_role in ['admin', 'super_admin']:
                return True
            
            # Check specific role requirements
            if required_role.lower() != user_role:
                return False
        
        # Check area-based permissions
        if required_area:
            user_areas = user.get('areas', '').split(',')
            user_areas = [area.strip() for area in user_areas if area.strip()]
            
            if required_area not in user_areas:
                return False
        
        return True
    
    def _get_user_by_email(self, email: str) -> Optional[Dict]:
        """Get user by email from database."""
        query = "SELECT * FROM users WHERE LOWER(TRIM(email)) = %s"
        return db_manager.execute_query(query, (email,), fetch_one=True)
    
    def _verify_password(self, input_password: str, stored_password: str, user_id: str) -> bool:
        """
        Verify password against stored hash.
        Supports both bcrypt hashes and plain text (for migration).
        """
        input_password = input_password.strip()
        stored_password = (stored_password or '').strip()
        
        # Check if password is hashed (bcrypt)
        if stored_password.startswith(('$2y$', '$2a$', '$2b$')):
            try:
                return bcrypt.checkpw(input_password.encode('utf-8'), stored_password.encode('utf-8'))
            except Exception as e:
                logger.error(f"Error verifying bcrypt password: {e}")
                return False
        else:
            # Legacy plain text password
            is_valid = (input_password == stored_password)
            
            # Migrate to hashed password if valid
            if is_valid:
                self._migrate_password_to_hash(user_id, input_password)
            
            return is_valid
    
    def _migrate_password_to_hash(self, user_id: str, plain_password: str) -> None:
        """Migrate plain text password to bcrypt hash."""
        try:
            hashed_password = bcrypt.hashpw(plain_password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
            
            query = "UPDATE users SET password = %s WHERE user_id = %s"
            db_manager.execute_update(query, (hashed_password, user_id))
            
            logger.info(f"Migrated password to hash for user ID: {user_id}")
        except Exception as e:
            logger.error(f"Failed to migrate password for user {user_id}: {e}")
    
    def _create_access_token(self, data: Dict, expires_delta: Optional[timedelta] = None) -> str:
        """Create JWT access token."""
        to_encode = data.copy()
        
        if expires_delta:
            expire = datetime.now(timezone.utc) + expires_delta
        else:
            expire = datetime.now(timezone.utc) + timedelta(hours=self.jwt_expiration_hours)
        
        to_encode.update({"exp": expire})
        encoded_jwt = jwt.encode(to_encode, self.jwt_secret, algorithm=self.jwt_algorithm)
        
        return encoded_jwt
    
    def _get_redirect_url(self, user_title: str) -> str:
        """Get redirect URL based on user role."""
        if user_title.lower() in ['team_lead', 'analyst']:
            return "analyst-tl.html"
        else:
            return "blue.html"
    
    def _log_login_success(self, user_id: str, email: str, ip_address: Optional[str], user_agent: Optional[str]) -> None:
        """Log successful login attempt."""
        try:
            # Import here to avoid circular imports
            from app.services.login_logs_service import login_logs_service
            login_logs_service.log_login_success(user_id, email, ip_address, user_agent)
        except Exception as e:
            logger.error(f"Failed to log successful login for {email}: {e}")
    
    def _log_login_failure(self, email: str, reason: str, ip_address: Optional[str], user_agent: Optional[str]) -> None:
        """Log failed login attempt."""
        try:
            # Import here to avoid circular imports
            from app.services.login_logs_service import login_logs_service
            login_logs_service.log_login_failure(email, reason, ip_address, user_agent)
        except Exception as e:
            logger.error(f"Failed to log failed login for {email}: {e}")
    
    def log_logout(self, user_id: str, email: str, ip_address: Optional[str], user_agent: Optional[str]) -> None:
        """Log user logout."""
        try:
            # Import here to avoid circular imports
            from app.services.login_logs_service import login_logs_service
            login_logs_service.log_logout(user_id, email, ip_address, user_agent)
        except Exception as e:
            logger.error(f"Failed to log logout for {email}: {e}")


# Global auth service instance
auth_service = AuthService()