"""
User repository for database operations related to users.
Implements user-specific database queries and operations.
"""

from typing import Dict, List, Optional, Any
import logging
import bcrypt

from app.repositories.base import BaseRepository, TimestampMixin
from app.core.exceptions import (
    UserNotFoundError, 
    EmailAlreadyExistsError, 
    ValidationError,
    DatabaseError
)

logger = logging.getLogger(__name__)


class UserRepository(BaseRepository):
    """Repository for user database operations."""
    
    def __init__(self):
        super().__init__("users")
        self.primary_key = "user_id"
    
    def find_by_email(self, email: str) -> Optional[Dict[str, Any]]:
        """
        Find user by email address (case-insensitive).
        
        Args:
            email: User's email address
            
        Returns:
            User record or None if not found
        """
        try:
            # Normalize email for case-insensitive search
            normalized_email = email.strip().lower()
            
            query = "SELECT * FROM users WHERE LOWER(TRIM(email)) = %s"
            result = self.db.execute_query(query, (normalized_email,), fetch_one=True)
            
            return result
            
        except Exception as e:
            logger.error(f"Error finding user by email {email}: {e}")
            raise DatabaseError(f"Failed to find user by email: {str(e)}")
    
    def user_exists(self, user_id: str) -> bool:
        """
        Check if a user exists by user_id.
        
        Args:
            user_id: User's unique identifier
            
        Returns:
            True if user exists, False otherwise
        """
        try:
            query = "SELECT COUNT(*) as count FROM users WHERE user_id = %s"
            result = self.db.execute_query(query, (user_id,), fetch_one=True)
            
            return result.get('count', 0) > 0 if result else False
            
        except Exception as e:
            logger.error(f"Error checking if user exists {user_id}: {e}")
            return False
    
    def get_user_by_id(self, user_id: str) -> Optional[Dict[str, Any]]:
        """
        Get user by user_id with basic validation.
        
        Args:
            user_id: User's unique identifier
            
        Returns:
            User data if found, None otherwise
        """
        try:
            query = "SELECT user_id, name, email, title, status FROM users WHERE user_id = %s"
            result = self.db.execute_query(query, (user_id,), fetch_one=True)
            
            return result
            
        except Exception as e:
            logger.error(f"Error getting user by ID {user_id}: {e}")
            return None
    
    def find_by_user_id(self, user_id: str) -> Optional[Dict[str, Any]]:
        """
        Find user by user_id.
        
        Args:
            user_id: User's unique identifier
            
        Returns:
            User record or None if not found
        """
        return self.find_by_id(user_id, "user_id")
    
    def email_exists(self, email: str, exclude_user_id: Optional[str] = None) -> bool:
        """
        Check if email already exists.
        
        Args:
            email: Email to check
            exclude_user_id: User ID to exclude from check (for updates)
            
        Returns:
            True if email exists
        """
        try:
            normalized_email = email.strip().lower()
            
            if exclude_user_id:
                query = "SELECT COUNT(*) as count FROM users WHERE LOWER(TRIM(email)) = %s AND user_id != %s"
                params = (normalized_email, exclude_user_id)
            else:
                query = "SELECT COUNT(*) as count FROM users WHERE LOWER(TRIM(email)) = %s"
                params = (normalized_email,)
            
            result = self.db.execute_query(query, params, fetch_one=True)
            return result['count'] > 0 if result else False
            
        except Exception as e:
            logger.error(f"Error checking email existence {email}: {e}")
            raise DatabaseError(f"Failed to check email existence: {str(e)}")
    
    def create_user(self, user_data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Create a new user with validation and password hashing.
        
        Args:
            user_data: User data dictionary
            
        Returns:
            Created user record
            
        Raises:
            EmailAlreadyExistsError: If email already exists
            ValidationError: If required fields are missing
        """
        try:
            # Validate required fields
            required_fields = ['user_id', 'name', 'email', 'password', 'title']
            for field in required_fields:
                if not user_data.get(field):
                    raise ValidationError(f"Field '{field}' is required", field)
            
            # Check if email already exists
            if self.email_exists(user_data['email']):
                raise EmailAlreadyExistsError()
            
            # Hash password if it's not already hashed
            password = user_data['password']
            if not password.startswith(('$2y$', '$2a$', '$2b$')):
                hashed_password = bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
                user_data['password'] = hashed_password
            
            # Set default values
            user_data.setdefault('status', 1)  # Active by default
            user_data.setdefault('area_of_operation', user_data.pop('areas', ''))  # Map areas to area_of_operation
            user_data.setdefault('shift', 'Day Shift')
            user_data.setdefault('imglink', '')
            
            # Create user
            created_user = self.create(user_data)
            
            logger.info(f"Created user: {user_data['email']} (ID: {user_data['user_id']})")
            return created_user
            
        except (EmailAlreadyExistsError, ValidationError):
            raise
        except Exception as e:
            logger.error(f"Error creating user: {e}")
            raise DatabaseError(f"Failed to create user: {str(e)}")
    
    def update_user(
        self, 
        user_id: str, 
        user_data: Dict[str, Any],
        hash_password: bool = True
    ) -> Dict[str, Any]:
        """
        Update user information.
        
        Args:
            user_id: User ID to update
            user_data: Updated user data
            hash_password: Whether to hash the password if provided
            
        Returns:
            Updated user record
            
        Raises:
            UserNotFoundError: If user doesn't exist
            EmailAlreadyExistsError: If email already exists for another user
        """
        try:
            # Check if user exists
            existing_user = self.find_by_user_id(user_id)
            if not existing_user:
                raise UserNotFoundError(user_id)
            
            # Check email uniqueness if email is being updated
            if 'email' in user_data and user_data['email'] != existing_user['email']:
                if self.email_exists(user_data['email'], exclude_user_id=user_id):
                    raise EmailAlreadyExistsError()
            
            # Hash password if provided and not already hashed
            if 'password' in user_data and user_data['password'] and hash_password:
                password = user_data['password']
                if not password.startswith(('$2y$', '$2a$', '$2b$')):
                    hashed_password = bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
                    user_data['password'] = hashed_password
            
            # Map areas to area_of_operation if present
            if 'areas' in user_data:
                user_data['area_of_operation'] = user_data.pop('areas')
            
            # Remove password from update if it's empty
            if 'password' in user_data and not user_data['password']:
                del user_data['password']
            
            # Update user
            updated_user = self.update(user_data, {'user_id': user_id})
            
            logger.info(f"Updated user: {user_id}")
            return updated_user
            
        except (UserNotFoundError, EmailAlreadyExistsError):
            raise
        except Exception as e:
            logger.error(f"Error updating user {user_id}: {e}")
            raise DatabaseError(f"Failed to update user: {str(e)}")
    
    def delete_user(self, user_id: str) -> bool:
        """
        Delete a user (soft delete by setting status to 0).
        
        Args:
            user_id: User ID to delete
            
        Returns:
            True if user was deleted
            
        Raises:
            UserNotFoundError: If user doesn't exist
        """
        try:
            # Check if user exists
            existing_user = self.find_by_user_id(user_id)
            if not existing_user:
                raise UserNotFoundError(user_id)
            
            # Soft delete by setting status to 0
            self.update({'status': 0}, {'user_id': user_id})
            
            logger.info(f"Deleted user: {user_id}")
            return True
            
        except UserNotFoundError:
            raise
        except Exception as e:
            logger.error(f"Error deleting user {user_id}: {e}")
            raise DatabaseError(f"Failed to delete user: {str(e)}")
    
    def find_by_role(self, role: str, active_only: bool = True) -> List[Dict[str, Any]]:
        """
        Find users by role/title.
        
        Args:
            role: User role to search for
            active_only: Whether to include only active users
            
        Returns:
            List of user records
        """
        try:
            where_conditions = {'title': role}
            if active_only:
                where_conditions['status'] = 1
            
            return self.find_all(where=where_conditions, order_by="name ASC")
            
        except Exception as e:
            logger.error(f"Error finding users by role {role}: {e}")
            raise DatabaseError(f"Failed to find users by role: {str(e)}")
    
    def find_by_area(self, area: str, active_only: bool = True) -> List[Dict[str, Any]]:
        """
        Find users by area of operation.
        
        Args:
            area: Area name to search for
            active_only: Whether to include only active users
            
        Returns:
            List of user records
        """
        try:
            # Use LIKE query to search in comma-separated areas
            where_clause = "area_of_operation LIKE %s"
            params = [f"%{area}%"]
            
            if active_only:
                where_clause += " AND status = %s"
                params.append(1)
            
            query = f"SELECT * FROM users WHERE {where_clause} ORDER BY name ASC"
            
            result = self.db.execute_query(query, tuple(params), fetch_all=True)
            return result or []
            
        except Exception as e:
            logger.error(f"Error finding users by area {area}: {e}")
            raise DatabaseError(f"Failed to find users by area: {str(e)}")
    
    def get_user_statistics(self) -> Dict[str, Any]:
        """
        Get user statistics.
        
        Returns:
            Dictionary with user statistics
        """
        try:
            stats = {}
            
            # Total users
            stats['total_users'] = self.count()
            
            # Active users
            stats['active_users'] = self.count({'status': 1})
            
            # Users by role
            role_query = """
                SELECT title, COUNT(*) as count 
                FROM users 
                WHERE status = 1 
                GROUP BY title 
                ORDER BY count DESC
            """
            role_stats = self.db.execute_query(role_query, fetch_all=True) or []
            stats['users_by_role'] = {row['title']: row['count'] for row in role_stats}
            
            # Users by shift
            shift_query = """
                SELECT shift, COUNT(*) as count 
                FROM users 
                WHERE status = 1 
                GROUP BY shift 
                ORDER BY count DESC
            """
            shift_stats = self.db.execute_query(shift_query, fetch_all=True) or []
            stats['users_by_shift'] = {row['shift']: row['count'] for row in shift_stats}
            
            return stats
            
        except Exception as e:
            logger.error(f"Error getting user statistics: {e}")
            raise DatabaseError(f"Failed to get user statistics: {str(e)}")
    
    def search_users(
        self, 
        search_term: str, 
        active_only: bool = True,
        limit: int = 50
    ) -> List[Dict[str, Any]]:
        """
        Search users by name or email.
        
        Args:
            search_term: Search term
            active_only: Whether to include only active users
            limit: Maximum number of results
            
        Returns:
            List of matching user records
        """
        try:
            search_pattern = f"%{search_term}%"
            
            where_clause = "(name LIKE %s OR email LIKE %s)"
            params = [search_pattern, search_pattern]
            
            if active_only:
                where_clause += " AND status = %s"
                params.append(1)
            
            query = f"SELECT * FROM users WHERE {where_clause} ORDER BY name ASC LIMIT {limit}"
            
            result = self.db.execute_query(query, tuple(params), fetch_all=True)
            return result or []
            
        except Exception as e:
            logger.error(f"Error searching users with term '{search_term}': {e}")
            raise DatabaseError(f"Failed to search users: {str(e)}")


# Global user repository instance
user_repository = UserRepository()