"""
Login logs repository for database operations related to authentication logging.
Handles login, logout, and failure event logging and reporting.
"""

from typing import Dict, List, Optional, Any
import logging
from datetime import datetime

from app.repositories.base import BaseRepository
from app.core.exceptions import DatabaseError

logger = logging.getLogger(__name__)


class LoginLogsRepository(BaseRepository):
    """Repository for login logs database operations."""
    
    def __init__(self):
        super().__init__("login_logs")
        self.primary_key = "id"
    
    def log_login_attempt(
        self,
        user_id: Optional[str],
        email: str,
        status: str,
        message: str,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Log a login attempt (success, failure, or logout).
        
        Args:
            user_id: User's unique identifier (None for failed logins)
            email: User's email address
            status: Login status (success, failure, logout)
            message: Descriptive message
            ip_address: Client IP address
            user_agent: Client user agent string
            
        Returns:
            Created log entry
        """
        try:
            log_data = {
                'user_id': user_id,
                'email': email.lower().strip(),
                'status': status,
                'message': message,
                'ip_address': ip_address,
                'user_agent': user_agent,
                'created_at': datetime.now()
            }
            
            created_log = self.create(log_data)
            
            logger.info(f"Login attempt logged: {email} - {status}")
            return created_log
            
        except Exception as e:
            logger.error(f"Error logging login attempt for {email}: {e}")
            raise DatabaseError(f"Failed to log login attempt: {str(e)}")
    
    def get_login_logs_paginated(
        self,
        user_id: Optional[str] = None,
        email: Optional[str] = None,
        status: Optional[str] = None,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None,
        page: int = 1,
        limit: int = 50
    ) -> Dict[str, Any]:
        """
        Get login logs with filtering and pagination.
        
        Args:
            user_id: Filter by user ID
            email: Filter by email
            status: Filter by status (success, failure, logout)
            date_from: Start date filter (YYYY-MM-DD)
            date_to: End date filter (YYYY-MM-DD)
            page: Page number
            limit: Items per page
            
        Returns:
            Dict with items and pagination info
        """
        try:
            conditions = []
            params = []
            
            # Apply filters
            if user_id:
                conditions.append("user_id = %s")
                params.append(user_id)
            if email:
                conditions.append("LOWER(email) LIKE %s")
                params.append(f"%{email.lower()}%")
            if status:
                conditions.append("status = %s")
                params.append(status)
            if date_from:
                conditions.append("DATE(created_at) >= %s")
                params.append(date_from)
            if date_to:
                conditions.append("DATE(created_at) <= %s")
                params.append(date_to)
            
            where_clause = " AND ".join(conditions) if conditions else "1 = 1"
            
            # Get total count
            count_query = f"SELECT COUNT(*) as total FROM `{self.table_name}` WHERE {where_clause}"
            count_result = self.db.execute_query(count_query, tuple(params), fetch_one=True)
            total = count_result['total'] if count_result else 0
            
            # Get paginated items
            offset = (page - 1) * limit
            items_query = f"""
                SELECT * FROM `{self.table_name}` 
                WHERE {where_clause} 
                ORDER BY created_at DESC
                LIMIT %s OFFSET %s
            """
            
            items_params = tuple(params) + (limit, offset)
            items = self.db.execute_query(items_query, items_params, fetch_all=True)
            
            # Convert datetime objects to strings for API response
            if items:
                for record in items:
                    if record.get('created_at') and hasattr(record['created_at'], 'strftime'):
                        record['created_at'] = record['created_at'].strftime('%Y-%m-%dT%H:%M:%S')
            
            return {
                "items": items or [],
                "total": total
            }
            
        except Exception as e:
            logger.error(f"Error getting login logs: {e}")
            raise DatabaseError(f"Failed to get login logs: {str(e)}")
    
    def get_login_statistics(
        self,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Get login statistics for reporting.
        
        Args:
            date_from: Start date filter (YYYY-MM-DD)
            date_to: End date filter (YYYY-MM-DD)
            
        Returns:
            Statistics dictionary
        """
        try:
            stats = {}
            
            # Build date filter
            date_conditions = []
            date_params = []
            
            if date_from:
                date_conditions.append("DATE(created_at) >= %s")
                date_params.append(date_from)
            if date_to:
                date_conditions.append("DATE(created_at) <= %s")
                date_params.append(date_to)
            
            date_where = " AND ".join(date_conditions) if date_conditions else "1 = 1"
            
            # Total login attempts
            total_query = f"SELECT COUNT(*) as count FROM `{self.table_name}` WHERE {date_where}"
            total_result = self.db.execute_query(total_query, tuple(date_params), fetch_one=True)
            stats['total_attempts'] = total_result['count'] if total_result else 0
            
            # Login attempts by status
            status_query = f"""
                SELECT status, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY status 
                ORDER BY count DESC
            """
            status_results = self.db.execute_query(status_query, tuple(date_params), fetch_all=True) or []
            stats['attempts_by_status'] = {row['status']: row['count'] for row in status_results}
            
            # Top users by login attempts
            users_query = f"""
                SELECT email, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY email 
                ORDER BY count DESC 
                LIMIT 10
            """
            users_results = self.db.execute_query(users_query, tuple(date_params), fetch_all=True) or []
            stats['top_users'] = {row['email']: row['count'] for row in users_results}
            
            # Login attempts by hour (for activity patterns)
            hourly_query = f"""
                SELECT HOUR(created_at) as hour, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY HOUR(created_at) 
                ORDER BY hour
            """
            hourly_results = self.db.execute_query(hourly_query, tuple(date_params), fetch_all=True) or []
            stats['attempts_by_hour'] = {f"{row['hour']:02d}:00": row['count'] for row in hourly_results}
            
            # Failed login attempts (security monitoring)
            failed_query = f"""
                SELECT email, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE status = 'failure' AND {date_where}
                GROUP BY email 
                HAVING count >= 3
                ORDER BY count DESC 
                LIMIT 10
            """
            failed_results = self.db.execute_query(failed_query, tuple(date_params), fetch_all=True) or []
            stats['high_failure_users'] = {row['email']: row['count'] for row in failed_results}
            
            # Recent activity (last 24 hours)
            recent_query = f"""
                SELECT status, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                GROUP BY status
            """
            recent_results = self.db.execute_query(recent_query, fetch_all=True) or []
            stats['recent_24h'] = {row['status']: row['count'] for row in recent_results}
            
            return stats
            
        except Exception as e:
            logger.error(f"Error getting login statistics: {e}")
            raise DatabaseError(f"Failed to get login statistics: {str(e)}")
    
    def create_login_log(
        self,
        user_id: Optional[str],
        email: str,
        status: str,
        message: str,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Create a new login log record.
        
        Args:
            user_id: User's unique identifier (None for failed logins)
            email: User's email address
            status: Login status (success, failure, logout)
            message: Log message
            ip_address: Client IP address (optional)
            user_agent: Client user agent (optional)
            
        Returns:
            Created log record with ID and timestamp
        """
        try:
            query = f"""
                INSERT INTO `{self.table_name}` 
                (user_id, email, status, message, ip_address, user_agent, created_at)
                VALUES (%s, %s, %s, %s, %s, %s, NOW())
            """
            
            params = (user_id, email, status, message, ip_address, user_agent)
            
            record_id = self.db.execute_insert(query, params)
            
            # Get the created record
            created_record = self.db.execute_query(
                f"SELECT * FROM `{self.table_name}` WHERE id = %s",
                (record_id,),
                fetch_one=True
            )
            
            # Convert datetime to string
            if created_record and created_record.get('created_at'):
                if hasattr(created_record['created_at'], 'strftime'):
                    created_record['created_at'] = created_record['created_at'].strftime('%Y-%m-%dT%H:%M:%S')
            
            return {
                'id': record_id,
                'created_at': created_record.get('created_at') if created_record else None,
                'record': created_record
            }
            
        except Exception as e:
            logger.error(f"Error creating login log: {e}")
            raise DatabaseError(f"Failed to create login log: {str(e)}")
    
    def get_user_login_history(
        self,
        user_id: str,
        limit: int = 20
    ) -> List[Dict[str, Any]]:
        """
        Get recent login history for a specific user.
        
        Args:
            user_id: User's unique identifier
            limit: Maximum number of records
            
        Returns:
            List of login history records with datetime converted to string
        """
        try:
            query = f"""
                SELECT * FROM `{self.table_name}` 
                WHERE user_id = %s 
                ORDER BY created_at DESC 
                LIMIT %s
            """
            
            result = self.db.execute_query(query, (user_id, limit), fetch_all=True)
            
            # Convert datetime objects to strings for API response
            if result:
                for record in result:
                    if record.get('created_at') and hasattr(record['created_at'], 'strftime'):
                        record['created_at'] = record['created_at'].strftime('%Y-%m-%dT%H:%M:%S')
            
            return result or []
            
        except Exception as e:
            logger.error(f"Error getting user login history for {user_id}: {e}")
            raise DatabaseError(f"Failed to get user login history: {str(e)}")


# Global login logs repository instance
login_logs_repository = LoginLogsRepository()