"""
Base repository class providing common database operations.
Implements the Repository pattern for data access abstraction.
"""

from abc import ABC, abstractmethod
from typing import Dict, List, Optional, Any, Tuple, Union
import logging

from app.utils.database import db_manager, QueryBuilder
from app.core.exceptions import DatabaseError, ResourceNotFoundError

logger = logging.getLogger(__name__)


class BaseRepository(ABC):
    """
    Abstract base repository class.
    Provides common CRUD operations and database utilities.
    """
    
    def __init__(self, table_name: str):
        self.table_name = table_name
        self.db = db_manager
        self.query_builder = QueryBuilder()
    
    def find_by_id(self, id_value: Any, id_column: str = "id") -> Optional[Dict[str, Any]]:
        """
        Find a record by ID.
        
        Args:
            id_value: The ID value to search for
            id_column: The ID column name (default: "id")
            
        Returns:
            Record dictionary or None if not found
        """
        try:
            query, params = self.query_builder.select(
                table=self.table_name,
                where={id_column: id_value}
            )
            
            result = self.db.execute_query(query, params, fetch_one=True)
            return result
            
        except Exception as e:
            logger.error(f"Error finding {self.table_name} by {id_column}={id_value}: {e}")
            raise DatabaseError(f"Failed to find record: {str(e)}")
    
    def find_all(
        self, 
        where: Optional[Dict[str, Any]] = None,
        order_by: Optional[str] = None,
        limit: Optional[int] = None
    ) -> List[Dict[str, Any]]:
        """
        Find all records matching criteria.
        
        Args:
            where: Filter conditions
            order_by: Order by clause
            limit: Maximum number of records
            
        Returns:
            List of record dictionaries
        """
        try:
            query, params = self.query_builder.select(
                table=self.table_name,
                where=where,
                order_by=order_by,
                limit=limit
            )
            
            result = self.db.execute_query(query, params, fetch_all=True)
            return result or []
            
        except Exception as e:
            logger.error(f"Error finding all {self.table_name}: {e}")
            raise DatabaseError(f"Failed to find records: {str(e)}")
    
    def find_one(self, where: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """
        Find one record matching criteria.
        
        Args:
            where: Filter conditions
            
        Returns:
            Record dictionary or None if not found
        """
        try:
            query, params = self.query_builder.select(
                table=self.table_name,
                where=where,
                limit=1
            )
            
            result = self.db.execute_query(query, params, fetch_one=True)
            return result
            
        except Exception as e:
            logger.error(f"Error finding one {self.table_name}: {e}")
            raise DatabaseError(f"Failed to find record: {str(e)}")
    
    def create(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Create a new record.
        
        Args:
            data: Record data
            
        Returns:
            Created record dictionary
        """
        try:
            query, params = self.query_builder.insert(self.table_name, data)
            
            rows_affected = self.db.execute_update(query, params)
            
            if rows_affected == 0:
                raise DatabaseError("No rows were inserted")
            
            # Try to return the created record if possible
            if 'id' in data:
                return self.find_by_id(data['id'])
            elif hasattr(self, 'primary_key'):
                return self.find_by_id(data[self.primary_key], self.primary_key)
            else:
                return data
                
        except Exception as e:
            logger.error(f"Error creating {self.table_name}: {e}")
            raise DatabaseError(f"Failed to create record: {str(e)}")
    
    def update(
        self, 
        data: Dict[str, Any], 
        where: Dict[str, Any]
    ) -> Dict[str, Any]:
        """
        Update records matching criteria.
        
        Args:
            data: Update data
            where: Filter conditions
            
        Returns:
            Updated record dictionary
        """
        try:
            query, params = self.query_builder.update(self.table_name, data, where)
            
            rows_affected = self.db.execute_update(query, params)
            
            if rows_affected == 0:
                raise ResourceNotFoundError(self.table_name, str(where))
            
            # Try to return the updated record
            updated_record = self.find_one(where)
            return updated_record or data
            
        except ResourceNotFoundError:
            raise
        except Exception as e:
            logger.error(f"Error updating {self.table_name}: {e}")
            raise DatabaseError(f"Failed to update record: {str(e)}")
    
    def delete(self, where: Dict[str, Any]) -> bool:
        """
        Delete records matching criteria.
        
        Args:
            where: Filter conditions
            
        Returns:
            True if records were deleted
        """
        try:
            query, params = self.query_builder.delete(self.table_name, where)
            
            rows_affected = self.db.execute_update(query, params)
            
            return rows_affected > 0
            
        except Exception as e:
            logger.error(f"Error deleting {self.table_name}: {e}")
            raise DatabaseError(f"Failed to delete record: {str(e)}")
    
    def count(self, where: Optional[Dict[str, Any]] = None) -> int:
        """
        Count records matching criteria.
        
        Args:
            where: Filter conditions
            
        Returns:
            Number of matching records
        """
        try:
            conditions = []
            params = []
            
            if where:
                for key, value in where.items():
                    conditions.append(f"`{key}` = %s")
                    params.append(value)
            
            where_clause = " WHERE " + " AND ".join(conditions) if conditions else ""
            query = f"SELECT COUNT(*) as count FROM `{self.table_name}`{where_clause}"
            
            result = self.db.execute_query(query, tuple(params), fetch_one=True)
            return result['count'] if result else 0
            
        except Exception as e:
            logger.error(f"Error counting {self.table_name}: {e}")
            raise DatabaseError(f"Failed to count records: {str(e)}")
    
    def exists(self, where: Dict[str, Any]) -> bool:
        """
        Check if a record exists.
        
        Args:
            where: Filter conditions
            
        Returns:
            True if record exists
        """
        return self.count(where) > 0
    
    def paginate(
        self, 
        page: int = 1, 
        per_page: int = 50,
        where: Optional[Dict[str, Any]] = None,
        order_by: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Paginate records.
        
        Args:
            page: Page number (1-based)
            per_page: Records per page
            where: Filter conditions
            order_by: Order by clause
            
        Returns:
            Pagination result with items and metadata
        """
        try:
            # Validate pagination parameters
            page = max(1, page)
            per_page = max(1, min(100, per_page))
            
            # Get total count
            total_count = self.count(where)
            
            # Calculate pagination
            total_pages = (total_count + per_page - 1) // per_page
            offset = (page - 1) * per_page
            
            # Build query with LIMIT and OFFSET
            conditions = []
            params = []
            
            if where:
                for key, value in where.items():
                    conditions.append(f"`{key}` = %s")
                    params.append(value)
            
            where_clause = " WHERE " + " AND ".join(conditions) if conditions else ""
            order_clause = f" ORDER BY {order_by}" if order_by else ""
            limit_clause = f" LIMIT {per_page} OFFSET {offset}"
            
            query = f"SELECT * FROM `{self.table_name}`{where_clause}{order_clause}{limit_clause}"
            
            items = self.db.execute_query(query, tuple(params), fetch_all=True) or []
            
            return {
                "items": items,
                "pagination": {
                    "current_page": page,
                    "per_page": per_page,
                    "total_items": total_count,
                    "total_pages": total_pages,
                    "has_next": page < total_pages,
                    "has_previous": page > 1
                }
            }
            
        except Exception as e:
            logger.error(f"Error paginating {self.table_name}: {e}")
            raise DatabaseError(f"Failed to paginate records: {str(e)}")
    
    def execute_raw_query(
        self, 
        query: str, 
        params: Optional[Tuple] = None,
        fetch_one: bool = False
    ) -> Union[Dict[str, Any], List[Dict[str, Any]], None]:
        """
        Execute a raw SQL query.
        
        Args:
            query: SQL query string
            params: Query parameters
            fetch_one: Whether to fetch only one result
            
        Returns:
            Query result
        """
        try:
            return self.db.execute_query(query, params, fetch_one=fetch_one)
        except Exception as e:
            logger.error(f"Error executing raw query: {e}")
            raise DatabaseError(f"Failed to execute query: {str(e)}")
    
    def execute_raw_update(
        self, 
        query: str, 
        params: Optional[Tuple] = None
    ) -> int:
        """
        Execute a raw SQL update/insert/delete query.
        
        Args:
            query: SQL query string
            params: Query parameters
            
        Returns:
            Number of affected rows
        """
        try:
            return self.db.execute_update(query, params)
        except Exception as e:
            logger.error(f"Error executing raw update: {e}")
            raise DatabaseError(f"Failed to execute update: {str(e)}")


class TimestampMixin:
    """Mixin for repositories with timestamp fields."""
    
    def create_with_timestamps(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Create record with automatic timestamps."""
        from datetime import datetime
        
        now = datetime.utcnow()
        data['created_at'] = now
        data['updated_at'] = now
        
        return self.create(data)
    
    def update_with_timestamps(
        self, 
        data: Dict[str, Any], 
        where: Dict[str, Any]
    ) -> Dict[str, Any]:
        """Update record with automatic updated_at timestamp."""
        from datetime import datetime
        
        data['updated_at'] = datetime.utcnow()
        
        return self.update(data, where)