"""
Active Sessions Repository
Handles database operations for active session tracking.
"""

from typing import Dict, List, Optional, Any
import logging
from datetime import datetime, timedelta

from app.repositories.base import BaseRepository
from app.core.exceptions import DatabaseError

logger = logging.getLogger(__name__)


class ActiveSessionsRepository(BaseRepository):
    """Repository for active sessions data access."""
    
    def __init__(self):
        super().__init__("active_sessions")
        self.primary_key = "session_id"
    
    def get_active_sessions_paginated(
        self,
        user_id: Optional[str] = None,
        email: Optional[str] = None,
        status: Optional[str] = None,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None,
        page: int = 1,
        limit: int = 50
    ) -> Dict[str, Any]:
        """
        Get active sessions with filtering and pagination.
        
        Args:
            user_id: Filter by user ID
            email: Filter by email
            status: Filter by status (active, expired, destroyed)
            date_from: Start date filter (YYYY-MM-DD)
            date_to: End date filter (YYYY-MM-DD)
            page: Page number
            limit: Items per page
            
        Returns:
            Dict with items and pagination info
        """
        try:
            conditions = []
            params = []
            
            # Apply filters
            if user_id:
                conditions.append("user_id = %s")
                params.append(user_id)
            if email:
                conditions.append("LOWER(email) LIKE %s")
                params.append(f"%{email.lower()}%")
            if status:
                conditions.append("status = %s")
                params.append(status)
            if date_from:
                conditions.append("DATE(login_time) >= %s")
                params.append(date_from)
            if date_to:
                conditions.append("DATE(login_time) <= %s")
                params.append(date_to)
            
            where_clause = " AND ".join(conditions) if conditions else "1 = 1"
            
            # Get total count
            count_query = f"SELECT COUNT(*) as total FROM `{self.table_name}` WHERE {where_clause}"
            count_result = self.db.execute_query(count_query, tuple(params), fetch_one=True)
            total = count_result['total'] if count_result else 0
            
            # Get paginated items
            offset = (page - 1) * limit
            items_query = f"""
                SELECT * FROM `{self.table_name}` 
                WHERE {where_clause} 
                ORDER BY login_time DESC
                LIMIT %s OFFSET %s
            """
            
            items_params = tuple(params) + (limit, offset)
            items = self.db.execute_query(items_query, items_params, fetch_all=True)
            
            # Convert datetime objects to strings for API response
            if items:
                for record in items:
                    for date_field in ['login_time', 'last_activity', 'expires_at']:
                        if record.get(date_field) and hasattr(record[date_field], 'strftime'):
                            record[date_field] = record[date_field].strftime('%Y-%m-%dT%H:%M:%S')
            
            return {
                "items": items or [],
                "total": total
            }
            
        except Exception as e:
            logger.error(f"Error getting active sessions: {e}")
            raise DatabaseError(f"Failed to get active sessions: {str(e)}")
    
    def get_active_sessions_statistics(
        self,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Get active sessions statistics for reporting.
        
        Args:
            date_from: Start date filter (YYYY-MM-DD)
            date_to: End date filter (YYYY-MM-DD)
            
        Returns:
            Statistics dictionary
        """
        try:
            stats = {}
            
            # Build date filter
            date_conditions = []
            date_params = []
            
            if date_from:
                date_conditions.append("DATE(login_time) >= %s")
                date_params.append(date_from)
            if date_to:
                date_conditions.append("DATE(login_time) <= %s")
                date_params.append(date_to)
            
            date_where = " AND ".join(date_conditions) if date_conditions else "1 = 1"
            
            # Total sessions
            total_query = f"SELECT COUNT(*) as count FROM `{self.table_name}` WHERE {date_where}"
            total_result = self.db.execute_query(total_query, tuple(date_params), fetch_one=True)
            stats['total_sessions'] = total_result['count'] if total_result else 0
            
            # Sessions by status
            status_query = f"""
                SELECT status, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY status 
                ORDER BY count DESC
            """
            status_result = self.db.execute_query(status_query, tuple(date_params), fetch_all=True)
            stats['sessions_by_status'] = {row['status']: row['count'] for row in status_result} if status_result else {}
            
            # Most active users by session count
            user_query = f"""
                SELECT email, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY email 
                ORDER BY count DESC 
                LIMIT 10
            """
            user_result = self.db.execute_query(user_query, tuple(date_params), fetch_all=True)
            stats['most_active_users'] = {row['email']: row['count'] for row in user_result} if user_result else {}
            
            # Sessions by hour of day
            hour_query = f"""
                SELECT HOUR(login_time) as hour, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY HOUR(login_time) 
                ORDER BY hour
            """
            hour_result = self.db.execute_query(hour_query, tuple(date_params), fetch_all=True)
            stats['sessions_by_hour'] = {f"{row['hour']:02d}:00": row['count'] for row in hour_result} if hour_result else {}
            
            # Currently active sessions
            active_query = f"""
                SELECT COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE status = 'active' AND expires_at > NOW()
            """
            active_result = self.db.execute_query(active_query, (), fetch_one=True)
            stats['currently_active'] = active_result['count'] if active_result else 0
            
            # Average session duration (for completed sessions)
            duration_query = f"""
                SELECT AVG(TIMESTAMPDIFF(MINUTE, login_time, last_activity)) as avg_duration_minutes
                FROM `{self.table_name}` 
                WHERE {date_where} AND status IN ('expired', 'destroyed')
                AND last_activity > login_time
            """
            duration_result = self.db.execute_query(duration_query, tuple(date_params), fetch_one=True)
            avg_duration = duration_result['avg_duration_minutes'] if duration_result and duration_result['avg_duration_minutes'] else 0
            stats['avg_session_duration_minutes'] = round(float(avg_duration), 2) if avg_duration else 0
            
            # Recent 24h sessions
            recent_query = f"""
                SELECT status, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE login_time >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                GROUP BY status
            """
            recent_result = self.db.execute_query(recent_query, (), fetch_all=True)
            stats['recent_24h'] = {row['status']: row['count'] for row in recent_result} if recent_result else {}
            
            return stats
            
        except Exception as e:
            logger.error(f"Error getting active sessions statistics: {e}")
            raise DatabaseError(f"Failed to get active sessions statistics: {str(e)}")
    
    def create_active_session(
        self,
        session_id: str,
        user_id: str,
        email: str,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None,
        login_time: Optional[str] = None,
        expires_at: Optional[str] = None,
        status: str = "active"
    ) -> Dict[str, Any]:
        """
        Create a new active session record.
        
        Args:
            session_id: Session identifier
            user_id: User's unique identifier
            email: User's email address
            ip_address: Client IP address (optional)
            user_agent: Client user agent (optional)
            login_time: Login timestamp (optional, defaults to NOW())
            expires_at: Session expiration time (optional)
            status: Session status (default: active)
            
        Returns:
            Created session record with timestamp
        """
        try:
            # Build query - always use provided values or NOW()
            query = f"""
                INSERT INTO `{self.table_name}` 
                (session_id, user_id, email, ip_address, user_agent, login_time, last_activity, expires_at, status)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)
            """
            
            # Use provided login_time or current timestamp
            login_time_value = login_time if login_time else datetime.now().strftime('%Y-%m-%d %H:%M:%S')
            last_activity_value = login_time_value  # Same as login time initially
            
            # If expires_at is not provided, set it to 4 hours from login time
            if not expires_at:
                login_dt = datetime.strptime(login_time_value, '%Y-%m-%d %H:%M:%S')
                expires_dt = login_dt + timedelta(hours=4)
                expires_at = expires_dt.strftime('%Y-%m-%d %H:%M:%S')
            
            params = (
                session_id, user_id, email, ip_address, user_agent,
                login_time_value, last_activity_value, expires_at, status
            )
            
            logger.info(f"Creating active session with query: {query}")
            logger.info(f"Parameters: {params}")
            
            self.db.execute_insert(query, params)
            
            # Get the created record
            created_record = self.db.execute_query(
                f"SELECT * FROM `{self.table_name}` WHERE session_id = %s",
                (session_id,),
                fetch_one=True
            )
            
            # Convert datetime objects to strings
            if created_record:
                for date_field in ['login_time', 'last_activity', 'expires_at']:
                    if created_record.get(date_field) and hasattr(created_record[date_field], 'strftime'):
                        created_record[date_field] = created_record[date_field].strftime('%Y-%m-%dT%H:%M:%S')
            
            return {
                'session_id': session_id,
                'created_at': created_record.get('login_time') if created_record else None,
                'record': created_record
            }
            
        except Exception as e:
            logger.error(f"Error creating active session: {e}")
            raise DatabaseError(f"Failed to create active session: {str(e)}")
    
# Global active sessions repository instance
active_sessions_repository = ActiveSessionsRepository()