"""
Pydantic schemas for request/response models.
Defines data validation and serialization models.
"""

from pydantic import BaseModel, Field, EmailStr, validator
from typing import Optional, List, Dict, Any
from datetime import datetime
from enum import Enum


# Base schemas
class BaseResponse(BaseModel):
    """Base response model."""
    success: bool = Field(True, description="Indicates if the operation was successful")
    message: Optional[str] = Field(None, description="Response message")


class ErrorResponse(BaseModel):
    """Error response model."""
    error: Dict[str, Any] = Field(..., description="Error details")


class PaginationMeta(BaseModel):
    """Pagination metadata."""
    current_page: int = Field(..., description="Current page number")
    per_page: int = Field(..., description="Items per page")
    total_items: int = Field(..., description="Total number of items")
    total_pages: int = Field(..., description="Total number of pages")
    has_next: bool = Field(..., description="Whether there is a next page")
    has_previous: bool = Field(..., description="Whether there is a previous page")


class PaginatedResponse(BaseModel):
    """Paginated response model."""
    items: List[Dict[str, Any]] = Field(..., description="List of items")
    pagination: PaginationMeta = Field(..., description="Pagination metadata")


# Enums
class UserRole(str, Enum):
    """User roles enumeration."""
    ADMIN = "admin"
    SUPER_ADMIN = "super_admin"
    ANALYST = "analyst"
    HEAD_OF_ANALYST = "head_of_analyst"
    TEAM_LEAD = "team_lead"
    HEAD_OF_TEAM_LEAD = "head_of_team_lead"
    OPERATOR = "operator"
    USER = "user"


class ApprovalType(str, Enum):
    """Approval types enumeration."""
    ANALYST = "analyst"
    TEAM_LEAD = "tl"


class ApprovalAction(str, Enum):
    """Approval actions enumeration."""
    APPROVE = "approve"
    REJECT = "reject"


class ReportFormat(str, Enum):
    """Report format enumeration."""
    PDF = "pdf"
    EXCEL = "excel"


# Authentication schemas
class LoginRequest(BaseModel):
    """Login request model."""
    email: EmailStr = Field(..., description="User's email address", example="user@example.com")
    password: str = Field(..., description="User's password", min_length=1, example="password123")


class LoginResponse(BaseModel):
    """Login response model."""
    success: bool = Field(True, description="Login successful")
    redirect: str = Field(..., description="Suggested redirect URL based on user role")
    message: str = Field(..., description="Success message")
    access_token: str = Field(..., description="JWT token for Authorization header")
    token_type: str = Field("bearer", description="Token type")


class TokenResponse(BaseModel):
    """Token response model."""
    access_token: str = Field(..., description="JWT access token")
    token_type: str = Field("bearer", description="Token type")
    expires_in: int = Field(..., description="Token expiration time in seconds")


# User schemas
class UserBase(BaseModel):
    """Base user model."""
    name: str = Field(..., description="User's full name", min_length=1, max_length=255)
    email: EmailStr = Field(..., description="User's email address")
    title: UserRole = Field(..., description="User's role/title")
    shift: str = Field(..., description="User's shift assignment")
    areas: Optional[str] = Field("", description="Comma-separated list of area names")


class UserCreate(UserBase):
    """User creation model."""
    password: str = Field(..., description="User's password", min_length=6)


class UserUpdate(BaseModel):
    """User update model."""
    name: str = Field(..., description="User's full name", min_length=1, max_length=255)
    title: UserRole = Field(..., description="User's role/title")
    shift: str = Field(..., description="User's shift assignment")
    status: Optional[int] = Field(1, description="User status: 1 = active, 0 = inactive", ge=0, le=1)
    areas: Optional[str] = Field("", description="Comma-separated list of area names")
    password: Optional[str] = Field(None, description="New password", min_length=6)
    change_password: Optional[bool] = Field(False, description="Set to true to change password")


class UserResponse(UserBase):
    """User response model."""
    user_id: str = Field(..., description="User's unique identifier")
    status: int = Field(..., description="User status")
    created_at: Optional[datetime] = Field(None, description="User creation timestamp")
    updated_at: Optional[datetime] = Field(None, description="User last update timestamp")


# Batch operation schemas
class BatchItemData(BaseModel):
    """Batch item data model. Batch number is auto-generated by backend; bn from frontend is optional and ignored for storage."""
    item: str = Field(..., description="Item name", min_length=1)
    actual_weight: str = Field(..., description="Actual weight measured")
    expected_weight: Optional[str] = Field(None, description="Expected/target weight (from frontend or product table)")
    bn: Optional[str] = Field(default='', description="Scanned batch number (optional; omit or send empty; backend ignores it and uses auto-generated batch_no)")
    date_manufacturer: Optional[str] = Field(None, description="Manufacturing date from scan (YYYY-MM-DD)")
    expiry_date: Optional[str] = Field(None, description="Expiry date from scan (YYYY-MM-DD)")
    mdn: Optional[str] = Field("", description="Material document number")
    comment: Optional[str] = Field(None, description="Operator comment for this item", max_length=1000)

    @validator('bn', pre=True)
    def bn_optional(cls, v):
        """Accept missing bn; backend ignores it and uses auto-generated batch_no."""
        if v is None or (isinstance(v, str) and v.strip() == ''):
            return ''
        return v


class BackendRequest(BaseModel):
    """Backend data submission model."""
    tableData: List[BatchItemData] = Field(..., description="List of batch items", min_items=1)
    operator: EmailStr = Field(..., description="Operator's email address")
    analyst: EmailStr = Field(..., description="Analyst's email address")
    teamLead: EmailStr = Field(..., description="Team lead's email address")
    product: str = Field(..., description="Product name/code", min_length=1)
    factory: str = Field(..., description="Factory identifier", min_length=1)


# Explicit submit API contract: request and response
class BatchSubmitRow(BaseModel):
    """One row in a batch submit. All fields are taken from the client and stored as-is where applicable."""
    item: str = Field(..., description="Item/material name", min_length=1)
    actual_weight: str = Field(..., description="Actual weight measured")
    expected_weight: Optional[str] = Field("", description="Expected weight (stored in DB)")
    bn: Optional[str] = Field("", description="Scanned batch number (ignored; backend generates batch_no)")
    date_manufacturer: Optional[str] = Field(None, description="Manufacturing date YYYY-MM-DD")
    expiry_date: Optional[str] = Field(None, description="Expiry date YYYY-MM-DD")
    mdn: Optional[str] = Field("", description="Material document number")
    comment: Optional[str] = Field(None, description="Operator comment (stored in DB)")

    @validator("bn", "expected_weight", "mdn", pre=True)
    def empty_str_default(cls, v):
        """Accept None or missing; coerce to '' for optional strings."""
        if v is None:
            return ""
        return v


class BatchSubmitRequest(BaseModel):
    """Request body for POST /batches/submit. Exactly what the API accepts."""
    tableData: List[BatchSubmitRow] = Field(..., description="Rows to submit", min_items=1)
    operator: EmailStr = Field(..., description="Operator email")
    analyst: EmailStr = Field(..., description="Analyst email")
    teamLead: EmailStr = Field(..., description="Team lead email")
    product: str = Field(..., description="Product code", min_length=1)
    factory: str = Field(..., description="Factory code", min_length=1)


class BatchSubmitResponse(BaseModel):
    """Response for POST /batches/submit."""
    success: bool = Field(True, description="Operation succeeded")
    message: str = Field(..., description="Human-readable message")
    items_created: int = Field(..., description="Number of rows inserted")
    serial_numbers: List[str] = Field(..., description="Serial numbers for this batch")
    batch_metadata: Dict[str, Any] = Field(..., description="Scan date, time, shift, week")
    # Optional debug: what the server received for first row (for troubleshooting expected_weight/comment)
    debug_first_row_expected_weight: Optional[str] = Field(None, description="[Debug] First row expected_weight as received")
    debug_first_row_comment: Optional[str] = Field(None, description="[Debug] First row comment as received")
    debug_first_row_keys: Optional[List[str]] = Field(None, description="[Debug] First row dict keys as received")
    debug_raw_body_snippet: Optional[str] = Field(None, description="[Debug] First 600 chars of raw request body")
    debug_parsed_first_row: Optional[Dict[str, Any]] = Field(None, description="[Debug] Parsed body tableData[0] as received")


class BatchItemsRequest(BaseModel):
    """Batch items query model."""
    serial_no: Optional[str] = Field(None, description="Serial number filter")
    shift: Optional[str] = Field(None, description="Shift filter")
    operator: Optional[str] = Field(None, description="Operator email filter")


class BatchItemsBulkRequest(BaseModel):
    """Bulk batch items query model."""
    items: List[BatchItemsRequest] = Field(..., description="List of batch queries", min_items=1, max_items=100)


# Approval schemas
class ApproveItemRequest(BaseModel):
    """Item approval request model."""
    serial_no: str = Field(..., description="Serial number of the batch item")
    operator: EmailStr = Field(..., description="Operator's email address")
    shift: str = Field(..., description="Shift name")
    analyst: EmailStr = Field(..., description="Analyst's email address")
    type: ApprovalType = Field(..., description="Approval type")
    action: ApprovalAction = Field(..., description="Approval action")
    batch_no: Optional[str] = Field(None, description="Batch number (e.g. scan_date_scan_time). When provided, only this submission is approved; otherwise all rows with same serial_no/operator/shift are updated (legacy).")
    
    # Optional rejection reason (required when action is "reject")
    rejection_reason: Optional[str] = Field(
        None, 
        description="Reason for rejection (required when action is 'reject')",
        max_length=1000
    )
    
    @validator('rejection_reason')
    def validate_rejection_reason(cls, v, values):
        """Validate rejection_reason based on action."""
        action = values.get('action')
        
        if action == ApprovalAction.REJECT and not v:
            raise ValueError('rejection_reason is required when action is "reject"')
        
        if action == ApprovalAction.APPROVE and v:
            raise ValueError('rejection_reason should not be provided when action is "approve"')
        
        return v


# Product and area schemas
class AreaRequest(BaseModel):
    """Area creation/update model."""
    name: str = Field(..., description="Area name", min_length=1, max_length=100)
    code: str = Field(..., description="Area code", min_length=1, max_length=50)


class AreaResponse(BaseModel):
    """Area response model."""
    id: int = Field(..., description="Area ID")
    name: str = Field(..., description="Area name")
    code: str = Field(..., description="Area code")
    status: int = Field(..., description="Area status")
    created_at: datetime = Field(..., description="Creation timestamp")
    updated_at: datetime = Field(..., description="Last update timestamp")


class ProductResponse(BaseModel):
    """Product response model."""
    id: int = Field(..., description="Product ID")
    product_name: str = Field(..., description="Product name")
    table_name: str = Field(..., description="Associated table name")


# Report schemas
class GenerateReportRequest(BaseModel):
    """Report generation request model."""
    serial_no: str = Field(..., description="Batch serial number")
    shift: str = Field(..., description="Shift information")
    operator: str = Field(..., description="Operator email/name")
    format: ReportFormat = Field(ReportFormat.PDF, description="Report format")


# Batch statistics schemas
class BatchStatisticsResponse(BaseModel):
    """Batch statistics response model."""
    total_items: int = Field(..., description="Total number of batch items")
    items_by_status: Dict[str, int] = Field(..., description="Items grouped by status")
    items_by_product: Dict[str, int] = Field(..., description="Items grouped by product")
    items_by_operator: Dict[str, int] = Field(..., description="Items grouped by operator")


class BatchItemResponse(BaseModel):
    """Batch item response model."""
    id: int = Field(..., description="Batch item ID")
    product: str = Field(..., description="Product name")
    items: str = Field(..., description="Item/ingredient name")
    batch_no: str = Field(..., description="Batch number")
    expected_weight: str = Field(..., description="Expected weight")
    actual_weight: str = Field(..., description="Actual weight measured")
    operator: str = Field(..., description="Operator email")
    analyst: str = Field(..., description="Analyst email")
    teamLead: str = Field(..., description="Team lead email")
    serial_no: str = Field(..., description="Serial number")
    shift: str = Field(..., description="Shift name")
    scan_date: str = Field(..., description="Scan date")
    scan_time: str = Field(..., description="Scan time")
    status: str = Field(..., description="Item status")
    analyst_status: str = Field(..., description="Analyst approval status")
    tl_status: str = Field(..., description="Team lead approval status")
    factory: str = Field(..., description="Factory identifier")


class LoginLogRequest(BaseModel):
    """Login log request model."""
    user_id: Optional[str] = Field(None, description="Filter by user ID")
    email: Optional[str] = Field(None, description="Filter by email (partial match)")
    status: Optional[str] = Field(None, description="Filter by status", regex="^(success|failure|logout)$")
    date_from: Optional[str] = Field(None, description="Start date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    date_to: Optional[str] = Field(None, description="End date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    page: int = Field(1, ge=1, description="Page number (starts at 1)")
    limit: int = Field(50, ge=1, le=100, description="Items per page (max 100)")


class LoginLogResponse(BaseModel):
    """Login log response model."""
    id: int = Field(..., description="Log entry ID")
    user_id: Optional[str] = Field(None, description="User ID")
    email: str = Field(..., description="User email")
    status: str = Field(..., description="Login status")
    message: str = Field(..., description="Log message")
    ip_address: Optional[str] = Field(None, description="Client IP address")
    user_agent: Optional[str] = Field(None, description="Client user agent")
    created_at: str = Field(..., description="Timestamp")


class LoginStatisticsRequest(BaseModel):
    """Login statistics request model."""
    date_from: Optional[str] = Field(None, description="Start date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    date_to: Optional[str] = Field(None, description="End date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")


class LoginStatisticsResponse(BaseModel):
    """Login statistics response model."""
    total_attempts: int = Field(..., description="Total login attempts")
    attempts_by_status: Dict[str, int] = Field(..., description="Attempts grouped by status")
    top_users: Dict[str, int] = Field(..., description="Top users by login attempts")
    attempts_by_hour: Dict[str, int] = Field(..., description="Attempts by hour of day")
    high_failure_users: Dict[str, int] = Field(..., description="Users with high failure rates")


class LoginHistoryRequest(BaseModel):
    """Login history request model."""
    user_id: Optional[str] = Field(None, description="User ID to get history for (None for current user)")
    limit: int = Field(20, description="Maximum number of records", ge=1, le=50)


# Activity Logs Schemas
class UserActivityRequest(BaseModel):
    """User activity logs request model."""
    user_id: Optional[str] = Field(None, description="Filter by user ID")
    email: Optional[str] = Field(None, description="Filter by email")
    activity_type: Optional[str] = Field(None, description="Filter by activity type (view, click, etc.)")
    action: Optional[str] = Field(None, description="Filter by action")
    page_url: Optional[str] = Field(None, description="Filter by page URL")
    date_from: Optional[str] = Field(None, description="Start date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    date_to: Optional[str] = Field(None, description="End date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    page: Optional[int] = Field(1, description="Page number", ge=1)
    limit: Optional[int] = Field(50, description="Items per page", ge=1, le=100)


class UserActivityResponse(BaseModel):
    """User activity response model."""
    id: int = Field(..., description="Activity ID")
    user_id: Optional[str] = Field(None, description="User ID")
    email: Optional[str] = Field(None, description="User email")
    activity_type: str = Field(..., description="Activity type")
    action: str = Field(..., description="Action performed")
    page_url: Optional[str] = Field(None, description="Page URL")
    element_id: Optional[str] = Field(None, description="Element ID")
    element_type: Optional[str] = Field(None, description="Element type")
    data: Optional[str] = Field(None, description="Additional data (JSON)")
    ip_address: Optional[str] = Field(None, description="Client IP address")
    user_agent: Optional[str] = Field(None, description="Client user agent")
    created_at: str = Field(..., description="Timestamp")


class ActiveSessionRequest(BaseModel):
    """Active sessions request model."""
    user_id: Optional[str] = Field(None, description="Filter by user ID")
    email: Optional[str] = Field(None, description="Filter by email")
    status: Optional[str] = Field(None, description="Filter by status (active, expired, destroyed)")
    date_from: Optional[str] = Field(None, description="Start date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    date_to: Optional[str] = Field(None, description="End date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    page: Optional[int] = Field(1, description="Page number", ge=1)
    limit: Optional[int] = Field(50, description="Items per page", ge=1, le=100)


class ActiveSessionResponse(BaseModel):
    """Active session response model."""
    session_id: str = Field(..., description="Session ID")
    user_id: str = Field(..., description="User ID")
    email: str = Field(..., description="User email")
    ip_address: Optional[str] = Field(None, description="Client IP address")
    user_agent: Optional[str] = Field(None, description="Client user agent")
    login_time: str = Field(..., description="Login timestamp")
    last_activity: str = Field(..., description="Last activity timestamp")
    expires_at: str = Field(..., description="Session expiration timestamp")
    status: str = Field(..., description="Session status")


class ActivityStatisticsRequest(BaseModel):
    """Activity statistics request model."""
    date_from: Optional[str] = Field(None, description="Start date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    date_to: Optional[str] = Field(None, description="End date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")


class ActivityStatisticsResponse(BaseModel):
    """Activity statistics response model."""
    user_activities: Dict[str, Any] = Field(..., description="User activities statistics")
    active_sessions: Dict[str, Any] = Field(..., description="Active sessions statistics")
    summary: Dict[str, Any] = Field(..., description="Overall activity summary")


# Create/Insert Schemas
class CreateUserActivityRequest(BaseModel):
    """Create user activity request model."""
    user_id: str = Field(..., description="User ID (must exist in users table)")
    email: str = Field(..., description="User email")
    activity_type: str = Field(..., description="Activity type (view, click, form, navigation)")
    action: str = Field(..., description="Action performed")
    page_url: Optional[str] = Field(None, description="Page URL")
    element_id: Optional[str] = Field(None, description="Element ID")
    element_type: Optional[str] = Field(None, description="Element type")
    data: Optional[str] = Field(None, description="Additional data (JSON string)")
    ip_address: Optional[str] = Field(None, description="Client IP address")
    user_agent: Optional[str] = Field(None, description="Client user agent")


class CreateActiveSessionRequest(BaseModel):
    """Create active session request model."""
    session_id: str = Field(..., description="Session ID")
    user_id: str = Field(..., description="User ID (must exist in users table)")
    email: str = Field(..., description="User email")
    ip_address: Optional[str] = Field(None, description="Client IP address")
    user_agent: Optional[str] = Field(None, description="Client user agent")
    login_time: Optional[str] = Field(None, description="Login time (YYYY-MM-DD HH:MM:SS)")
    expires_at: Optional[str] = Field(None, description="Session expiration time (YYYY-MM-DD HH:MM:SS)")
    status: Optional[str] = Field("active", description="Session status (active, expired, destroyed)")


class CreateLoginLogRequest(BaseModel):
    """Create login log request model."""
    user_id: Optional[str] = Field(None, description="User ID (must exist in users table if provided, null for failed logins)")
    email: str = Field(..., description="User email")
    status: str = Field(..., description="Login status (success, failure, logout)")
    message: str = Field(..., description="Log message")
    ip_address: Optional[str] = Field(None, description="Client IP address")
    user_agent: Optional[str] = Field(None, description="Client user agent")


class CreateRecordResponse(BaseModel):
    """Create record response model."""
    success: bool = Field(True, description="Operation success")
    message: str = Field(..., description="Success message")
    record_id: Optional[int] = Field(None, description="Created record ID")
    created_at: str = Field(..., description="Creation timestamp")


class BatchStatisticsRequest(BaseModel):
    """Batch statistics request model."""
    date_from: Optional[str] = Field(None, description="Start date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    date_to: Optional[str] = Field(None, description="End date (YYYY-MM-DD)", regex=r"^\d{4}-\d{2}-\d{2}$")
    product: Optional[str] = Field(None, description="Product filter")


class ProcessedDataRequest(BaseModel):
    """Processed data request model."""
    serial_no: Optional[str] = Field(None, description="Serial number filter")
    shift: Optional[str] = Field(None, description="Shift filter")
    operator: Optional[str] = Field(None, description="Operator email filter")
    approval_status: Optional[str] = Field(
        "all", 
        description="Approval status filter",
        regex="^(approved|not_approved|rejected|pending|all)$"
    )
    page: int = Field(1, ge=1, description="Page number (starts at 1)")
    limit: int = Field(20, ge=1, le=100, description="Items per page (max 100)")


class ProcessedDataResponse(BaseModel):
    """Processed data response model."""
    items: List[BatchItemResponse] = Field(..., description="List of processed batch items")
    summary: Dict[str, Any] = Field(..., description="Processing summary")


# Product ingredient schemas
class ProductIngredientResponse(BaseModel):
    """Product ingredient response model."""
    id: int = Field(..., description="Ingredient ID")
    ingredient: str = Field(..., description="Ingredient name")
    expected_weight: str = Field(..., description="Expected weight")
    weight_variance: Optional[str] = Field(None, description="Weight variance")
    to_be_measured: Optional[str] = Field(None, description="To be measured flag")
    mdn: Optional[str] = Field(None, description="Material document number")
    batch_no: Optional[str] = Field(None, description="Batch number")


class UpdateIngredientRequest(BaseModel):
    """Update ingredient request model."""
    value: str = Field(..., description="New value for the field")
    column: str = Field(..., description="Column to update")
    
    @validator('column')
    def validate_column(cls, v):
        allowed_columns = ['ingredient', 'expected_weight', 'weight_variance', 'to_be_measured', 'mdn', 'batch_no']
        if v not in allowed_columns:
            raise ValueError(f'Column must be one of: {", ".join(allowed_columns)}')
        return v


# Password reset schemas
class ForgotPasswordRequest(BaseModel):
    """Forgot password request model."""
    email: EmailStr = Field(..., description="User's email address")


class ResetPasswordRequest(BaseModel):
    """Reset password request model."""
    token: str = Field(..., description="Password reset token")
    new_password: str = Field(..., description="New password", min_length=8)
    confirm_password: str = Field(..., description="Password confirmation", min_length=8)
    
    @validator('confirm_password')
    def passwords_match(cls, v, values, **kwargs):
        if 'new_password' in values and v != values['new_password']:
            raise ValueError('Passwords do not match')
        return v


class SendWelcomeEmailRequest(BaseModel):
    """Send welcome email request model."""
    user_id: str = Field(..., description="User ID")
    send_credentials: bool = Field(True, description="Whether to include credentials in email")


# Utility schemas
class ListItemsRequest(BaseModel):
    """List items request model."""
    data: str = Field(..., description="Batch number filter", min_length=1)


class UpdateRowRequest(BaseModel):
    """Update row request model."""
    value: str = Field(..., description="New value for the column")
    column: str = Field(..., description="Column name to update")
    table: str = Field(..., description="Table name")
    id: int = Field(..., description="Row ID to update", gt=0)
    
    @validator('column')
    def validate_column(cls, v):
        allowed_columns = ['ingredient', 'expected_weight', 'weight_variance', 'to_be_measured', 'mdn', 'batch_no']
        if v not in allowed_columns:
            raise ValueError(f'Column must be one of: {", ".join(allowed_columns)}')
        return v


class TestEmailRequest(BaseModel):
    """Test email request model."""
    to: EmailStr = Field(..., description="Recipient email address")
    subject: str = Field("Test Email", description="Email subject")
    message: str = Field("This is a test email.", description="Email message")
    is_html: bool = Field(False, description="Whether message is HTML")


# Health check schemas
class HealthCheckResponse(BaseModel):
    """Health check response model."""
    status: str = Field(..., description="Service status")
    timestamp: datetime = Field(..., description="Check timestamp")
    version: str = Field(..., description="Application version")
    database: bool = Field(..., description="Database connectivity status")
    services: Dict[str, bool] = Field(..., description="External services status")