"""
SAP Integration Pydantic Schemas
Defines data models for SAP ECC integration via BTP middleware.
"""

from pydantic import BaseModel, Field, validator
from typing import List, Optional, Dict, Any
from datetime import datetime, date
from decimal import Decimal


# Valid plants for Kenya
VALID_PLANTS = ['K002', 'K013', 'K012']

# Valid movement types
VALID_MOVEMENT_TYPES = ['261']  # Goods issue for production


class MaterialTriggerRequest(BaseModel):
    """Request schema for triggering material request to SAP."""
    material_number: str = Field(..., description="Material number (e.g., FG001234)")
    plant: str = Field(..., description="Plant code (K002, K013, or K012)")
    
    @validator('material_number')
    def validate_material_number(cls, v):
        if not v or not v.strip():
            raise ValueError("Material number cannot be empty")
        return v.strip().upper()
    
    @validator('plant')
    def validate_plant(cls, v):
        if v not in VALID_PLANTS:
            raise ValueError(f"Invalid plant: {v}. Valid plants: {VALID_PLANTS}")
        return v


class MaterialTriggerResponse(BaseModel):
    """Response schema for material trigger request."""
    success: bool = Field(..., description="Whether the request was successful")
    message: str = Field(..., description="Response message")
    material_number: str = Field(..., description="Material number")
    plant: str = Field(..., description="Plant code")
    request_id: str = Field(..., description="Unique request identifier")
    error_code: Optional[str] = Field(None, description="Error code if failed")
    timestamp: Optional[datetime] = Field(default_factory=datetime.utcnow, description="Response timestamp")


class MAK080ComponentItem(BaseModel):
    """Component item for MAK080 consumption request."""
    material_number: str = Field(..., description="Material number")
    plant: str = Field(..., description="Plant code")
    storage_location: str = Field(..., description="Storage location (e.g., 0001)")
    batch_number: str = Field(..., description="Batch number")
    movement_type: str = Field(..., description="Movement type (must be 261)")
    quantity: Decimal = Field(..., description="Quantity to consume")
    unit_of_measure: str = Field(..., description="Unit of measure (e.g., KG)")
    order_number: str = Field(..., description="Production order number")
    
    @validator('plant')
    def validate_plant(cls, v):
        if v not in VALID_PLANTS:
            raise ValueError(f"Invalid plant: {v}. Valid plants: {VALID_PLANTS}")
        return v
    
    @validator('movement_type')
    def validate_movement_type(cls, v):
        if v not in VALID_MOVEMENT_TYPES:
            raise ValueError(f"Invalid movement type: {v}. Valid types: {VALID_MOVEMENT_TYPES}")
        return v
    
    @validator('quantity')
    def validate_quantity(cls, v):
        if v <= 0:
            raise ValueError("Quantity must be greater than zero")
        return v


class MAK080ConsumptionRequest(BaseModel):
    """Request schema for MAK080 component consumption."""
    document_date: date = Field(..., description="Document date")
    posting_date: date = Field(..., description="Posting date")
    transaction_code: Optional[str] = Field(None, description="Transaction code")
    description: Optional[str] = Field(None, description="Description")
    components: List[MAK080ComponentItem] = Field(..., description="List of components to consume")
    
    @validator('document_date', 'posting_date')
    def validate_dates(cls, v):
        if v > date.today():
            raise ValueError("Date cannot be in the future")
        return v
    
    @validator('components')
    def validate_components(cls, v):
        if not v or len(v) == 0:
            raise ValueError("Components list cannot be empty")
        return v


class MAK080ConsumptionResponse(BaseModel):
    """Response schema for MAK080 consumption request."""
    success: bool = Field(..., description="Whether the request was successful")
    message: str = Field(..., description="Response message")
    idoc_number: Optional[str] = Field(None, description="IDOC number from SAP")
    components_count: int = Field(..., description="Number of components processed")
    request_id: str = Field(..., description="Unique request identifier")
    error_code: Optional[str] = Field(None, description="Error code if failed")
    timestamp: Optional[datetime] = Field(default_factory=datetime.utcnow, description="Response timestamp")


class ProcessOrderHeader(BaseModel):
    """Process order header information."""
    order_number: str = Field(..., description="Production order number")
    order_type: str = Field(..., description="Order type (e.g., KE33)")
    material_number: str = Field(..., description="Material number")
    plant: str = Field(..., description="Plant code")
    total_order_quantity: Decimal = Field(..., description="Total order quantity")
    base_unit_of_measure: str = Field(..., description="Base unit of measure")
    basic_start_date: Optional[date] = Field(None, description="Basic start date")
    basic_finish_date: Optional[date] = Field(None, description="Basic finish date")


class ProcessOrderComponent(BaseModel):
    """Process order component information."""
    material_number: str = Field(..., description="Component material number")
    requirements_quantity: Decimal = Field(..., description="Required quantity")
    base_unit_of_measure: str = Field(..., description="Base unit of measure")
    requirements_date: Optional[date] = Field(None, description="Requirements date")
    plant: str = Field(..., description="Plant code")
    variance_tolerance: Optional[Dict[str, Any]] = Field(None, description="Variance tolerance limits")


class ProcessOrderData(BaseModel):
    """Complete process order data."""
    header: ProcessOrderHeader = Field(..., description="Order header")
    material_data: Optional[Dict[str, Any]] = Field(None, description="Material data")
    components: List[ProcessOrderComponent] = Field(..., description="Order components")
    status: Optional[List[str]] = Field(None, description="Order status codes")


class ProcessOrderResponse(BaseModel):
    """Response schema for receiving process orders from SAP."""
    success: bool = Field(..., description="Whether the request was successful")
    message: str = Field(..., description="Response message")
    material_number: str = Field(..., description="Material number")
    plant: str = Field(..., description="Plant code")
    process_orders: List[ProcessOrderData] = Field(..., description="List of process orders")
    timestamp: Optional[datetime] = Field(default_factory=datetime.utcnow, description="Response timestamp")


class ALEAUDAcknowledgment(BaseModel):
    """ALEAUD acknowledgment schema from SAP."""
    message_type: str = Field(..., description="Message type (e.g., Z2WMPOISSFX080)")
    message_code: Optional[str] = Field(None, description="Message code")
    message_function: Optional[str] = Field(None, description="Message function")
    idoc_number: str = Field(..., description="IDOC number")
    status: str = Field(..., description="Status code")
    status_code: str = Field(..., description="Status code")
    status_text: str = Field(..., description="Status text")
    status_type: str = Field(..., description="Status type (S=Success, E=Error, W=Warning, I=Information)")
    status_message_qualifier: Optional[str] = Field(None, description="Status message qualifier")
    status_message_id: Optional[str] = Field(None, description="Status message ID")
    status_message_number: Optional[str] = Field(None, description="Status message number")
    parameters: Optional[Dict[str, Any]] = Field(None, description="Additional parameters")
    plant: Optional[str] = Field(None, description="Plant code")
    pallet_number: Optional[str] = Field(None, description="Pallet number")
    success: bool = Field(..., description="Whether the acknowledgment indicates success")
    error_message: Optional[str] = Field(None, description="Error message if failed")
    timestamp: Optional[datetime] = Field(default_factory=datetime.utcnow, description="Acknowledgment timestamp")


class SAPErrorResponse(BaseModel):
    """Standard SAP error response schema."""
    success: bool = Field(False, description="Always false for errors")
    error_code: str = Field(..., description="Error code")
    error_message: str = Field(..., description="Error message")
    timestamp: Optional[datetime] = Field(default_factory=datetime.utcnow, description="Error timestamp")
    details: Optional[Dict[str, Any]] = Field(None, description="Additional error details")

