"""
FastAPI dependencies for authentication, authorization, and common functionality.
"""

from fastapi import Depends, HTTPException, Request, status
from typing import Dict, Optional
import logging

from app.services.auth_service import auth_service
from app.core.config import get_settings
from app.core.exceptions import (
    InvalidTokenError,
    TokenExpiredError,
    InsufficientPermissionsError
)

logger = logging.getLogger(__name__)
settings = get_settings()


def get_current_user(request: Request) -> Dict:
    """
    Dependency to get current authenticated user from JWT token.
    
    Args:
        request: FastAPI request object
        
    Returns:
        User data dictionary
        
    Raises:
        HTTPException: If user is not authenticated
    """
    try:
        auth_header = request.headers.get("Authorization")
        
        user_data = auth_service.get_current_user(
            authorization_header=auth_header
        )
        
        return user_data
        
    except (InvalidTokenError, TokenExpiredError) as e:
        logger.warning(f"Authentication failed: {e.message}")
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Not authenticated",
            headers={"WWW-Authenticate": "Bearer"}
        )
    except Exception as e:
        logger.error(f"Unexpected authentication error: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="Authentication service error"
        )


def get_current_active_user(current_user: Dict = Depends(get_current_user)) -> Dict:
    """
    Dependency to get current active user.
    
    Args:
        current_user: Current user from get_current_user dependency
        
    Returns:
        Active user data dictionary
        
    Raises:
        HTTPException: If user is inactive
    """
    # Check if user is active (if status field exists)
    user_status = current_user.get('status')
    if user_status is not None and user_status == 0:
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="User account is inactive"
        )
    
    return current_user


def require_role(required_role: str):
    """
    Dependency factory to require specific user role.
    
    Args:
        required_role: Required user role
        
    Returns:
        Dependency function
    """
    def role_checker(current_user: Dict = Depends(get_current_active_user)) -> Dict:
        user_role = current_user.get('title', '').lower()
        
        # Admin and super_admin have access to everything
        if user_role in ['admin', 'super_admin']:
            return current_user
        
        # Check specific role requirement
        if user_role != required_role.lower():
            logger.warning(f"Access denied for user {current_user.get('email')} - required role: {required_role}, user role: {user_role}")
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=f"Access denied. Required role: {required_role}"
            )
        
        return current_user
    
    return role_checker


def require_admin(current_user: Dict = Depends(get_current_active_user)) -> Dict:
    """
    Dependency to require admin or super_admin role.
    
    Args:
        current_user: Current user from get_current_active_user dependency
        
    Returns:
        Admin user data dictionary
        
    Raises:
        HTTPException: If user is not admin
    """
    user_role = current_user.get('title', '').lower()
    
    if user_role not in ['admin', 'super_admin']:
        logger.warning(f"Admin access denied for user {current_user.get('email')} - role: {user_role}")
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Admin access required"
        )
    
    return current_user


def require_area_access(required_area: str):
    """
    Dependency factory to require access to specific area.
    
    Args:
        required_area: Required area name
        
    Returns:
        Dependency function
    """
    def area_checker(current_user: Dict = Depends(get_current_active_user)) -> Dict:
        user_role = current_user.get('title', '').lower()
        
        # Admin and super_admin have access to all areas
        if user_role in ['admin', 'super_admin']:
            return current_user
        
        # Check area access
        user_areas = current_user.get('areas', '').split(',')
        user_areas = [area.strip() for area in user_areas if area.strip()]
        
        if required_area not in user_areas:
            logger.warning(f"Area access denied for user {current_user.get('email')} - required: {required_area}, user areas: {user_areas}")
            raise HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=f"Access denied. Required area: {required_area}"
            )
        
        return current_user
    
    return area_checker



def get_optional_user(request: Request) -> Optional[Dict]:
    """
    Dependency to get current user if authenticated (optional).
    
    Args:
        request: FastAPI request object
        
    Returns:
        User data dictionary or None if not authenticated
    """
    try:
        return get_current_user(request)
    except HTTPException:
        return None


class RoleBasedAccess:
    """Class-based dependency for complex role-based access control."""
    
    def __init__(self, allowed_roles: list = None, require_area: str = None):
        self.allowed_roles = allowed_roles or []
        self.require_area = require_area
    
    def __call__(self, current_user: Dict = Depends(get_current_active_user)) -> Dict:
        user_role = current_user.get('title', '').lower()
        user_email = current_user.get('email', 'unknown')
        
        # Admin and super_admin always have access
        if user_role in ['admin', 'super_admin']:
            return current_user
        
        # Check role requirements
        if self.allowed_roles:
            allowed_roles_lower = [role.lower() for role in self.allowed_roles]
            if user_role not in allowed_roles_lower:
                logger.warning(f"Role access denied for {user_email} - required: {self.allowed_roles}, user: {user_role}")
                raise HTTPException(
                    status_code=status.HTTP_403_FORBIDDEN,
                    detail=f"Access denied. Required roles: {', '.join(self.allowed_roles)}"
                )
        
        # Check area requirements
        if self.require_area:
            user_areas = current_user.get('areas', '').split(',')
            user_areas = [area.strip() for area in user_areas if area.strip()]
            
            if self.require_area not in user_areas:
                logger.warning(f"Area access denied for {user_email} - required: {self.require_area}, user areas: {user_areas}")
                raise HTTPException(
                    status_code=status.HTTP_403_FORBIDDEN,
                    detail=f"Access denied. Required area: {self.require_area}"
                )
        
        return current_user


# Convenience instances for common access patterns
analyst_or_team_lead = RoleBasedAccess(allowed_roles=['analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead'])
operator_access = RoleBasedAccess(allowed_roles=['operator', 'analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead'])
management_access = RoleBasedAccess(allowed_roles=['head_of_analyst', 'head_of_team_lead', 'admin', 'super_admin'])


def verify_api_key(request: Request) -> bool:
    """
    Dependency to verify API key for external system access.
    
    Args:
        request: FastAPI request object
        
    Returns:
        True if API key is valid
        
    Raises:
        HTTPException: If API key is invalid or missing
    """
    api_key_header = settings.api_key_header
    provided_key = request.headers.get(api_key_header)
    
    if not provided_key:
        logger.warning(f"Missing API key in header {api_key_header}")
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="API key required",
            headers={"WWW-Authenticate": "ApiKey"}
        )
    
    expected_key = settings.sap_api_key
    if not expected_key:
        logger.error("SAP API key not configured in settings")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail="API key authentication not configured"
        )
    
    if provided_key != expected_key:
        logger.warning(f"Invalid API key provided: {provided_key[:8]}...")
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid API key",
            headers={"WWW-Authenticate": "ApiKey"}
        )
    
    logger.info("Valid API key provided for external system access")
    return True


def verify_sap_api_key(api_key_valid: bool = Depends(verify_api_key)) -> bool:
    """
    Dependency specifically for SAP/BTP system-to-system authentication.
    
    Args:
        api_key_valid: Result from verify_api_key dependency
        
    Returns:
        True if SAP API key is valid
    """
    return api_key_valid