"""
Configuration management for the FastAPI application.
Centralized configuration using Pydantic BaseSettings.

Secrets (DB password, JWT secret, SMTP password, etc.) must be set via
environment variables or .env only. Do not hardcode passwords here.
"""

from pydantic import BaseSettings, Field, validator
from typing import Optional, List
import os
from pathlib import Path


class Settings(BaseSettings):
    """Application settings with validation and environment variable support."""

    # Application
    app_name: str = "Krystal EA API"
    app_version: str = "1.0.0"
    debug: bool = False

    # Database Configuration (set DB_* in .env; no passwords in code)
    db_host: str = "localhost"
    db_user: str = "root"
    db_password: str = ""
    db_name: str = "bt"
    db_port: int = 3306
    # Batch submit table name (use "data" for production; "data_test" only for testing)
    batch_data_table: str = "data"
    
    @property
    def database_url(self) -> str:
        """Generate database URL from components."""
        return f"mysql+pymysql://{self.db_user}:{self.db_password}@{self.db_host}:{self.db_port}/{self.db_name}"
    
    # JWT Configuration (required: set JWT_SECRET_KEY in .env, min 32 chars)
    jwt_secret_key: str
    jwt_algorithm: str = "HS256"
    jwt_expiration_hours: int = 24
    
    @validator('jwt_secret_key')
    def validate_jwt_secret(cls, v):
        if not v or len(v) < 32:
            raise ValueError('JWT secret key must be at least 32 characters long')
        return v
    
    # CORS Configuration
    # Set CORS_ORIGINS in .env as comma-separated URLs (not JSON), e.g. http://example.com,https://admin.example.com
    # Stored as str so Pydantic does not try to JSON-parse the env var (which would fail).
    cors_origins_raw: str = Field(default="*", env="CORS_ORIGINS")

    @property
    def cors_origins(self) -> List[str]:
        s = (self.cors_origins_raw or "").strip()
        if not s:
            return ["*"]
        parts = [x.strip() for x in s.split(",") if x.strip()]
        return parts if parts else ["*"]
    
    # Email Configuration (set SMTP_* in .env; no passwords in code)
    smtp_host: str = "smtp.mailsafi.com"
    smtp_port: int = 587
    smtp_secure: str = "tls"  # 'tls' or 'ssl'
    smtp_username: str = "systems@krystal-ea.com"
    smtp_password: str = ""
    smtp_from_email: str = "systems@krystal-ea.com"
    smtp_from_name: str = "Krystal EA System"
    mail_enabled: bool = True
    mail_debug: bool = False
    smtp_debug: bool = False
    
    # Application URLs
    base_url: str = "http://localhost"
    frontend_url: str = "http://localhost"
    login_url: str = "http://localhost/blue/index.php"
    
    # File Paths
    api_logo_path: Optional[str] = None
    
    # Timezone
    timezone: str = "Africa/Nairobi"
    
    # PHP Integration (for email service)
    php_email_endpoint: str = "http://localhost/blue/api_send_email.php"
    
    # Logging
    log_level: str = "INFO"
    
    # SAP Integration Configuration
    sap_btp_base_url: str = "http://localhost:9000"  # Default to SAP simulator
    sap_btp_client_id: str = ""
    sap_btp_client_secret: str = ""
    sap_integration_enabled: bool = True  # Enable by default for testing
    sap_timeout: int = 30  # seconds
    
    # API Key Configuration for external systems
    api_key_header: str = "X-API-Key"
    sap_api_key: str = ""  # API key for SAP/BTP system-to-system calls
    
    @validator('sap_api_key')
    def validate_sap_api_key(cls, v):
        if v and len(v) < 32:
            raise ValueError('SAP API key must be at least 32 characters long')
        return v
    
    class Config:
        env_file = ".env"
        case_sensitive = False
        
        # Allow extra fields for backward compatibility
        extra = "allow"


# Global settings instance
settings = Settings()


def get_settings() -> Settings:
    """Get application settings instance."""
    return settings


# Project root detection (for backward compatibility)
def get_project_root() -> Path:
    """Get the project root directory."""
    backend_dir = Path(__file__).resolve().parent.parent.parent
    project_root = backend_dir.parent
    
    # Verify project root by checking for common directories/files
    if not (project_root / "backend").exists() and not (project_root / "images").exists():
        # Try parent directory if current doesn't look right
        potential_root = project_root.parent
        if (potential_root / "backend").exists() or (potential_root / "images").exists():
            project_root = potential_root
    
    return project_root


# Export commonly used values
PROJECT_ROOT = get_project_root()