"""
SAP Integration API endpoints for inbound and outbound interfaces.
Handles communication with SAP ECC via BTP middleware.
"""

from fastapi import APIRouter, HTTPException, Request, Depends, status, BackgroundTasks, Query
from fastapi.responses import JSONResponse
from typing import Dict, Any, Optional
from datetime import datetime
import logging

from app.models.sap_schemas import (
    MaterialTriggerRequest,
    MaterialTriggerResponse,
    MAK080ConsumptionRequest,
    MAK080ConsumptionResponse,
    ProcessOrderResponse,
    ALEAUDAcknowledgment,
    SAPErrorResponse
)
from app.services.sap_service import sap_service
from app.core.dependencies import get_current_user, require_role, verify_sap_api_key
from app.core.exceptions import (
    ValidationError,
    ExternalServiceError,
    BusinessLogicError,
    BaseAPIException
)

logger = logging.getLogger(__name__)
router = APIRouter(prefix="/sap", tags=["SAP Integration"])

# Role-based access for SAP operations
sap_access = require_role(["admin", "super_admin", "operator", "analyst", "team_lead"])


# ============================================================================
# INBOUND INTERFACES (Krystal → SAP)
# ============================================================================

@router.post(
    "/material-trigger",
    summary="Material Trigger Request",
    description="""
    **1st Inbound Interface: Material Trigger**
    
    Send material number and plant combination to SAP ECC via BTP to trigger 
    the outbound interface for sending process order list back to Krystal.
    
    **Process Flow:**
    1. Krystal sends material + plant to SAP
    2. SAP validates material/plant combination in MARC table
    3. If valid, SAP triggers process order extraction
    4. SAP sends process orders back via outbound interface
    
    **Validation:**
    - Material number must exist in SAP
    - Plant must be one of: K002, K013, K012
    - Material must be extended to the plant
    
    **Access Control:**
    - Operators: Can trigger for their assigned areas
    - Analysts/Team Leads: Can trigger for their areas
    - Admins: Can trigger for any material/plant
    """,
    response_model=MaterialTriggerResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Material trigger sent successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "Material trigger sent to SAP successfully. Process orders will be sent shortly.",
                        "material_number": "FG001234",
                        "plant": "K013",
                        "request_id": "REQ-2026-01-06-001",
                        "timestamp": "2026-01-06T12:00:00Z"
                    }
                }
            }
        },
        400: {
            "description": "Invalid material/plant combination",
            "model": SAPErrorResponse
        },
        401: {"description": "Authentication required"},
        403: {"description": "Insufficient permissions"},
        500: {"description": "SAP communication error"}
    }
)
async def trigger_material_request(
    request: MaterialTriggerRequest,
    current_user: Dict = Depends(sap_access),
    background_tasks: BackgroundTasks = BackgroundTasks()
):
    """
    Trigger material request to SAP for process orders.
    """
    try:
        logger.info(f"Material trigger request from user {current_user.get('email')} for {request.material_number} in {request.plant}")
        
        # TODO: Add area-based access control
        # Check if user has access to the plant/area
        user_areas = current_user.get('areas', '').split(',')
        user_title = current_user.get('title', '')
        
        # Admins can access all plants
        if user_title not in ['admin', 'super_admin']:
            # For now, allow all authenticated users
            # In production, implement proper area-based filtering
            pass
        
        # Send material trigger to SAP
        response = await sap_service.trigger_material_request(request)
        
        # Log the activity in background
        background_tasks.add_task(
            log_sap_activity,
            user_id=current_user.get('user_id'),
            activity_type="material_trigger",
            description=f"Triggered material request for {request.material_number} in {request.plant}",
            request_id=response.request_id,
            success=response.success
        )
        
        return response
        
    except ValidationError as e:
        logger.warning(f"Material trigger validation error: {e.message}")
        # timestamp is already a string (from BaseAPIException.__init__)
        timestamp = e.timestamp if hasattr(e, 'timestamp') else datetime.utcnow().isoformat()
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail={
                "success": False,
                "error_code": "VALIDATION_ERROR",
                "error_message": e.message,
                "timestamp": timestamp
            }
        )
    except ExternalServiceError as e:
        logger.error(f"SAP communication error: {e.message}")
        # timestamp is already a string (from BaseAPIException.__init__)
        timestamp = e.timestamp if hasattr(e, 'timestamp') else datetime.utcnow().isoformat()
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail={
                "success": False,
                "error_code": "SAP_COMMUNICATION_ERROR",
                "error_message": "Failed to communicate with SAP system",
                "timestamp": timestamp
            }
        )
    except Exception as e:
        logger.error(f"Unexpected error in material trigger: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail={
                "success": False,
                "error_code": "INTERNAL_ERROR",
                "error_message": "An unexpected error occurred",
                "timestamp": None
            }
        )


@router.post(
    "/component-consumption",
    summary="Component Consumption (MAK080)",
    description="""
    **2nd Inbound Interface: MAK080 Process Order Component Consumption**
    
    Send component consumption data to SAP ECC after pre-weighing activities.
    This creates goods issue (movement type 261) transactions in SAP.
    
    **Process Flow:**
    1. Krystal sends component consumption data
    2. BTP transforms JSON to IDOC format
    3. SAP processes MAK080 IDOC
    4. SAP posts goods issue (261 movement)
    5. SAP sends ALEAUD acknowledgment back
    
    **Data Requirements:**
    - Valid material numbers and plants
    - Correct batch numbers
    - Sufficient stock availability
    - Valid production order numbers
    
    **Error Handling:**
    - Status 51: Data issues (incorrect data, insufficient stock)
    - Status 64: IDOC ready but not processed
    - Automatic reprocessing via scheduled jobs
    
    **Access Control:**
    - Operators: Can consume for their orders
    - Analysts/Team Leads: Can consume for approved items
    - Admins: Can consume any components
    """,
    response_model=MAK080ConsumptionResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Component consumption sent successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "Component consumption sent to SAP successfully",
                        "idoc_number": "0000000001234567",
                        "components_count": 3,
                        "request_id": "MAK080-2026-01-06-001",
                        "timestamp": "2026-01-06T12:00:00Z"
                    }
                }
            }
        },
        400: {
            "description": "Invalid consumption data",
            "model": SAPErrorResponse
        },
        401: {"description": "Authentication required"},
        403: {"description": "Insufficient permissions"},
        500: {"description": "SAP processing error"}
    }
)
async def send_component_consumption(
    request: MAK080ConsumptionRequest,
    current_user: Dict = Depends(sap_access),
    background_tasks: BackgroundTasks = BackgroundTasks()
):
    """
    Send component consumption data to SAP (MAK080).
    """
    try:
        logger.info(f"MAK080 consumption request from user {current_user.get('email')} with {len(request.components)} components")
        
        # TODO: Add validation for user access to orders/materials
        # Check if user has access to the production orders
        user_title = current_user.get('title', '')
        
        # Send consumption data to SAP
        response = await sap_service.send_component_consumption(request)
        
        # Log the activity in background
        background_tasks.add_task(
            log_sap_activity,
            user_id=current_user.get('user_id'),
            activity_type="component_consumption",
            description=f"Sent component consumption with {len(request.components)} components",
            request_id=response.request_id,
            success=response.success,
            details={
                "idoc_number": response.idoc_number,
                "components_count": response.components_count
            }
        )
        
        return response
        
    except ValidationError as e:
        logger.warning(f"MAK080 validation error: {e.message}")
        # timestamp is already a string (from BaseAPIException.__init__)
        timestamp = e.timestamp if hasattr(e, 'timestamp') else datetime.utcnow().isoformat()
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail={
                "success": False,
                "error_code": "VALIDATION_ERROR",
                "error_message": e.message,
                "timestamp": timestamp
            }
        )
    except ExternalServiceError as e:
        logger.error(f"SAP processing error: {e.message}")
        # timestamp is already a string (from BaseAPIException.__init__)
        timestamp = e.timestamp if hasattr(e, 'timestamp') else datetime.utcnow().isoformat()
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail={
                "success": False,
                "error_code": "SAP_PROCESSING_ERROR",
                "error_message": "Failed to process consumption in SAP",
                "timestamp": timestamp
            }
        )
    except Exception as e:
        logger.error(f"Unexpected error in MAK080 consumption: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail={
                "success": False,
                "error_code": "INTERNAL_ERROR",
                "error_message": "An unexpected error occurred",
                "timestamp": None
            }
        )


# ============================================================================
# OUTBOUND INTERFACES (SAP → Krystal)
# ============================================================================

@router.post(
    "/receive-process-orders",
    summary="Receive Process Orders from SAP",
    description="""
    **1st Outbound Interface: Process Order Interface**
    
    Receive process orders from SAP ECC via BTP. This endpoint is called by BTP
    when SAP sends process order data in response to a material trigger.
    
    **Authentication**: Requires API key authentication (X-API-Key header)
    
    **Data Received:**
    - Process order headers (AFKO, AUFK tables)
    - Material data and descriptions
    - Order components/reservations (RESB table)
    - Order operations and sequences
    - Order status information
    
    **Filtering Criteria:**
    - Orders created/released in past 2 weeks
    - Not in TECO/CLSD status
    - Plant and order type in Z2XPR1 table
    - Valid material/plant combinations
    
    **Access Control:**
    - This endpoint is called by BTP (system-to-system)
    - Requires SAP API key in X-API-Key header
    - Not accessible to regular users
    """,
    response_model=ProcessOrderResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Process orders received successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "Received 1 process orders successfully",
                        "material_number": "FG001234",
                        "plant": "K013",
                        "process_orders": [
                            {
                                "header": {
                                    "order_number": "000012345678",
                                    "material_number": "FG001234",
                                    "plant": "K013"
                                },
                                "material_data": {
                                    "material_description": "Test Product"
                                },
                                "components": [],
                                "status": ["REL"]
                            }
                        ],
                        "timestamp": "2026-01-06T12:00:00Z"
                    }
                }
            }
        },
        401: {
            "description": "API key authentication failed",
            "content": {
                "application/json": {
                    "example": {
                        "detail": "API key required"
                    }
                }
            }
        },
        500: {"description": "Processing error"}
    },
    tags=["SAP Integration - External"]
)
async def receive_process_orders(
    request: Request,
    background_tasks: BackgroundTasks = BackgroundTasks(),
    api_key_valid: bool = Depends(verify_sap_api_key)
):
    """
    Receive process orders from SAP via BTP.
    This is a system-to-system endpoint called by BTP with API key authentication.
    """
    try:
        payload = await request.json()
        logger.info("Receiving process orders from SAP via BTP (API key authenticated)")
        
        # Process the received data
        response = await sap_service.receive_process_orders(payload)
        
        # Store process orders in database for mobile app consumption
        background_tasks.add_task(
            store_process_orders,
            process_orders=response.process_orders,
            material_number=response.material_number,
            plant=response.plant
        )
        
        return response
        
    except Exception as e:
        logger.error(f"Error receiving process orders: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail={
                "success": False,
                "error_code": "PROCESSING_ERROR",
                "error_message": "Failed to process received process orders"
            }
        )


@router.post(
    "/receive-aleaud-acknowledgment",
    summary="Receive ALEAUD Acknowledgment from SAP",
    description="""
    **2nd Outbound Interface: ALEAUD Acknowledgment**
    
    Receive acknowledgment responses from SAP ECC via BTP for MAK080 submissions.
    This provides success/failure status for component consumption requests.
    
    **Authentication**: Requires API key authentication (X-API-Key header)
    
    **Status Types:**
    - S: Success - Component consumption processed successfully
    - E: Error - Processing failed due to data issues
    - W: Warning - Processed with warnings
    - I: Information - Additional information provided
    
    **Error Scenarios:**
    - Status 51: Data issues (insufficient stock, missing batch, etc.)
    - Status 64: IDOC ready but not processed
    - Material/batch validation failures
    - Stock availability issues
    
    **Access Control:**
    - This endpoint is called by BTP (system-to-system)
    - Requires SAP API key in X-API-Key header
    - Not accessible to regular users
    """,
    response_model=ALEAUDAcknowledgment,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "ALEAUD acknowledgment received successfully",
            "content": {
                "application/json": {
                    "example": {
                        "message_type": "Z2WMPOISSFX080",
                        "idoc_number": "0000000001234567",
                        "status": "53",
                        "status_code": "53",
                        "status_text": "Application document posted",
                        "status_type": "S",
                        "success": True,
                        "plant": "K013"
                    }
                }
            }
        },
        401: {
            "description": "API key authentication failed",
            "content": {
                "application/json": {
                    "example": {
                        "detail": "API key required"
                    }
                }
            }
        },
        500: {"description": "Processing error"}
    },
    tags=["SAP Integration - External"]
)
async def receive_aleaud_acknowledgment(
    request: Request,
    background_tasks: BackgroundTasks = BackgroundTasks(),
    api_key_valid: bool = Depends(verify_sap_api_key)
):
    """
    Receive ALEAUD acknowledgment from SAP via BTP.
    This is a system-to-system endpoint called by BTP with API key authentication.
    """
    try:
        payload = await request.json()
        logger.info("Receiving ALEAUD acknowledgment from SAP via BTP (API key authenticated)")
        
        # Process the acknowledgment
        acknowledgment = await sap_service.receive_aleaud_acknowledgment(payload)
        
        # Update consumption request status in database
        background_tasks.add_task(
            update_consumption_status,
            idoc_number=acknowledgment.idoc_number,
            success=acknowledgment.success,
            status_code=acknowledgment.status_code,
            error_message=acknowledgment.error_message
        )
        
        return acknowledgment
        
    except Exception as e:
        logger.error(f"Error receiving ALEAUD acknowledgment: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail={
                "success": False,
                "error_code": "PROCESSING_ERROR",
                "error_message": "Failed to process ALEAUD acknowledgment"
            }
        )


# ============================================================================
# UTILITY ENDPOINTS
# ============================================================================

@router.get(
    "/plants",
    summary="Get Valid Plants",
    description="Get list of valid plants for SAP integration (K002, K013, K012).",
    response_model=Dict[str, Any]
)
async def get_valid_plants(current_user: Dict = Depends(get_current_user)):
    """Get list of valid plants for SAP integration."""
    return {
        "plants": sap_service.valid_plants,
        "plant_descriptions": {
            "K002": "Kenya Plant 002",
            "K013": "Kenya Plant 013", 
            "K012": "Kenya Plant 012"
        }
    }


@router.get(
    "/order-types/{plant}",
    summary="Get Valid Order Types for Plant",
    description="Get valid order types for a specific plant based on Z2XPR1 configuration.",
    response_model=Dict[str, Any]
)
async def get_order_types_for_plant(
    plant: str,
    current_user: Dict = Depends(get_current_user)
):
    """Get valid order types for a plant."""
    if plant not in sap_service.valid_plants:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail=f"Invalid plant: {plant}. Valid plants: {sap_service.valid_plants}"
        )
    
    order_types = sap_service.get_valid_order_types(plant)
    
    return {
        "plant": plant,
        "order_types": order_types,
        "descriptions": {
            "KE33": "Production Order Type KE33",
            "KE23": "Production Order Type KE23",
            "KE24": "Production Order Type KE24",
            "KE71": "Production Order Type KE71",
            "KE72": "Production Order Type KE72"
        }
    }


# ============================================================================
# BACKGROUND TASKS
# ============================================================================

async def log_sap_activity(
    user_id: str,
    activity_type: str,
    description: str,
    request_id: str,
    success: bool,
    details: Dict[str, Any] = None
):
    """Log SAP integration activity."""
    try:
        # TODO: Implement activity logging to database
        logger.info(f"SAP Activity - User: {user_id}, Type: {activity_type}, Success: {success}, Request: {request_id}")
    except Exception as e:
        logger.error(f"Failed to log SAP activity: {e}")


# In-memory storage for process orders (for mobile app retrieval)
# In production, this would be in database
_process_orders_cache = {}

async def store_process_orders(
    process_orders: list,
    material_number: str,
    plant: str
):
    """Store received process orders in memory cache (for mobile app retrieval)."""
    try:
        cache_key = f"{material_number}:{plant}"
        _process_orders_cache[cache_key] = {
            "process_orders": process_orders,
            "material_number": material_number,
            "plant": plant,
            "timestamp": datetime.utcnow().isoformat()
        }
        logger.info(f"Stored {len(process_orders)} process orders for {material_number} in {plant}")
    except Exception as e:
        logger.error(f"Failed to store process orders: {e}")


async def update_consumption_status(
    idoc_number: str,
    success: bool,
    status_code: str,
    error_message: str = None
):
    """Update consumption request status based on ALEAUD acknowledgment."""
    try:
        # TODO: Implement database update for consumption status
        logger.info(f"Updating consumption status for IDOC {idoc_number}: Success={success}, Status={status_code}")
    except Exception as e:
        logger.error(f"Failed to update consumption status: {e}")


# ============================================================================
# MOBILE APP ENDPOINTS
# ============================================================================

@router.get(
    "/orders",
    summary="Get Process Orders for Mobile App",
    description="""
    Retrieve process orders for a material/plant combination.
    Used by mobile app after material trigger to get available orders.
    
    **Flow:**
    1. Mobile app sends material trigger
    2. SAP sends orders back asynchronously (stored in cache)
    3. Mobile app polls this endpoint to retrieve orders
    
    **Access Control:**
    - Requires authentication
    - Available to operators, analysts, team leads
    """,
    status_code=status.HTTP_200_OK,
    tags=["SAP Integration - Mobile"]
)
async def get_process_orders(
    material_number: str = Query(..., description="Material number"),
    plant: str = Query(..., description="Plant code"),
    current_user: Dict[str, Any] = Depends(sap_access)
):
    """
    Get process orders for material/plant combination.
    Returns orders that were received from SAP after material trigger.
    """
    try:
        cache_key = f"{material_number.upper()}:{plant.upper()}"
        
        # Check cache first
        if cache_key in _process_orders_cache:
            cached_data = _process_orders_cache[cache_key]
            # Return cached orders
            return {
                "success": True,
                "material_number": material_number.upper(),
                "plant": plant.upper(),
                "process_orders": cached_data["process_orders"],
                "count": len(cached_data["process_orders"]),
                "timestamp": cached_data["timestamp"]
            }
        
        # If not in cache, try to get from SAP simulator database (for testing)
        try:
            import sys
            import os
            # Add backend directory to path (go up from app/api/v1/ to backend/)
            current_file = os.path.abspath(__file__)
            backend_dir = os.path.dirname(os.path.dirname(os.path.dirname(os.path.dirname(current_file))))
            if backend_dir not in sys.path:
                sys.path.insert(0, backend_dir)
            
            from sap_simulator_repository import SAPSimulatorRepository
            repo = SAPSimulatorRepository()
            orders = repo.get_process_orders(material_number.upper(), plant.upper())
            
            if orders:
                # Format orders for response
                formatted_orders = []
                for order in orders:
                    formatted_orders.append({
                        "header": {
                            "order_number": order["order_number"],
                            "order_type": order.get("order_type", "KE33"),
                            "material_number": order["material_number"],
                            "plant": order["plant"],
                            "total_order_quantity": float(order["total_order_quantity"]),
                            "base_unit_of_measure": order.get("base_unit_of_measure", "KG"),
                            "basic_start_date": order["basic_start_date"].isoformat() if order.get("basic_start_date") else None,
                            "basic_finish_date": order["basic_finish_date"].isoformat() if order.get("basic_finish_date") else None
                        },
                        "material_data": {
                            "material_description": order.get("material_description", ""),
                            "material_type": order.get("material_type", "FERT")
                        },
                        "components": order.get("components", []),
                        "status": order.get("status", ["REL"]) if isinstance(order.get("status"), list) else ["REL"]
                    })
                
                return {
                    "success": True,
                    "material_number": material_number.upper(),
                    "plant": plant.upper(),
                    "process_orders": formatted_orders,
                    "count": len(formatted_orders),
                    "timestamp": datetime.utcnow().isoformat()
                }
        except ImportError:
            # SAP simulator repository not available
            pass
        except Exception as e:
            logger.warning(f"Could not get orders from SAP simulator DB: {e}")
        
        # No orders found
        return {
            "success": True,
            "material_number": material_number.upper(),
            "plant": plant.upper(),
            "process_orders": [],
            "count": 0,
            "message": "No orders found. Please trigger material request first.",
            "timestamp": datetime.utcnow().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Error getting process orders: {e}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail={
                "success": False,
                "error_code": "PROCESSING_ERROR",
                "error_message": "Failed to retrieve process orders"
            }
        )