"""
Email API endpoints.
Handles email sending functionality including welcome emails and password reset emails.
"""

from fastapi import APIRouter, HTTPException, Depends, status
from typing import Dict, Any
import logging

from app.models.schemas import (
    SendWelcomeEmailRequest,
    ForgotPasswordRequest,
    TestEmailRequest,
    BaseResponse,
    ErrorResponse
)
from app.core.dependencies import (
    get_current_user,
    RoleBasedAccess
)
from app.core.exceptions import (
    ValidationError,
    BusinessLogicError,
    AuthorizationError,
    BaseAPIException,
    EmailServiceError
)

logger = logging.getLogger(__name__)
router = APIRouter(prefix="/emails", tags=["Email Services"])

# Role-based access control
admin_access = RoleBasedAccess(allowed_roles=['admin', 'super_admin'])
management_access = RoleBasedAccess(allowed_roles=['admin', 'super_admin', 'head_of_analyst', 'head_of_team_lead'])


@router.post(
    "/send-welcome",
    summary="Send Welcome Email",
    description="""
    Send welcome email to a user with their account credentials.
    
    **Purpose:**
    - Send welcome email to newly created users
    - Include login credentials and system information
    - Provide secure credential viewing option
    
    **Access Control:**
    - Requires admin or management access
    - Only users who can create accounts can send welcome emails
    
    **Security:**
    - Credentials can be sent via secure one-time link instead of plain text
    - Email sending is logged for audit purposes
    """,
    response_model=BaseResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Welcome email sent successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "Welcome email sent successfully to user@example.com"
                    }
                }
            }
        },
        400: {"description": "Validation error", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse},
        404: {"description": "User not found", "model": ErrorResponse},
        500: {"description": "Email service error", "model": ErrorResponse}
    }
)
async def send_welcome_email(
    request: SendWelcomeEmailRequest,
    current_user: Dict[str, Any] = Depends(admin_access)
):
    """
    Send welcome email to a user.
    
    **Example Request:**
    ```json
    {
        "user_id": "123",
        "send_credentials": true
    }
    ```
    
    **Example Response:**
    ```json
    {
        "success": true,
        "message": "Welcome email sent successfully to user@example.com"
    }
    ```
    """
    try:
        # Import here to avoid circular imports
        from app.services.user_service import user_service
        from backend.email_service import email_service
        from app.core.config import get_settings
        
        settings = get_settings()
        
        # Get user details
        user = user_service.get_user_by_id(request.user_id, current_user)
        if not user:
            raise HTTPException(status_code=404, detail="User not found")
        
        # Prepare email data
        email_data = {
            'email': user['email'],
            'name': user['name'],
            'title': user['title'],
            'shift': user['shift'],
            'login_url': settings.login_url
        }
        
        # Add credentials if requested
        if request.send_credentials:
            # For security, we should generate a secure one-time link
            # instead of sending plain text password
            credential_view_url = f"{settings.frontend_url}/view-credentials?token={user['user_id']}"
            email_data['credential_view_url'] = credential_view_url
        
        # Send welcome email
        result = email_service.send_welcome_email(**email_data)
        
        if result['success']:
            logger.info(f"Welcome email sent to {user['email']} by {current_user.get('email')}")
            return BaseResponse(
                success=True,
                message=f"Welcome email sent successfully to {user['email']}"
            )
        else:
            raise EmailServiceError(result.get('message', 'Failed to send welcome email'))
        
    except HTTPException:
        raise
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error sending welcome email: {e}")
        raise HTTPException(status_code=500, detail="Failed to send welcome email")


@router.post(
    "/forgot-password",
    summary="Send Password Reset Email",
    description="""
    Send password reset email to a user.
    
    **Purpose:**
    - Send password reset link to user's email
    - Generate secure reset token with expiration
    - Provide instructions for password reset
    
    **Security:**
    - Reset tokens expire after 1 hour
    - Tokens are single-use only
    - Email sending is rate-limited
    - No user enumeration (same response for valid/invalid emails)
    """,
    response_model=BaseResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Password reset email sent (if email exists)",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "If the email address exists in our system, a password reset link has been sent."
                    }
                }
            }
        },
        400: {"description": "Validation error", "model": ErrorResponse},
        429: {"description": "Rate limit exceeded", "model": ErrorResponse},
        500: {"description": "Email service error", "model": ErrorResponse}
    }
)
async def forgot_password(request: ForgotPasswordRequest):
    """
    Send password reset email.
    
    **Example Request:**
    ```json
    {
        "email": "user@example.com"
    }
    ```
    
    **Example Response:**
    ```json
    {
        "success": true,
        "message": "If the email address exists in our system, a password reset link has been sent."
    }
    ```
    """
    try:
        # Import here to avoid circular imports
        from app.services.user_service import user_service
        from backend.email_service import email_service
        from app.core.config import get_settings
        import secrets
        from datetime import datetime, timedelta
        
        settings = get_settings()
        
        # Check if user exists (but don't reveal this information)
        user = user_service.find_user_by_email(request.email)
        
        if user:
            # Generate reset token
            reset_token = secrets.token_urlsafe(32)
            expiry_time = datetime.utcnow() + timedelta(hours=1)
            
            # Store reset token (you would implement this in user service)
            # user_service.store_reset_token(user['user_id'], reset_token, expiry_time)
            
            # Create reset URL
            reset_url = f"{settings.frontend_url}/reset-password?token={reset_token}"
            
            # Send password reset email
            result = email_service.send_password_reset_email(
                email=user['email'],
                name=user['name'],
                reset_url=reset_url,
                expiry_hours=1
            )
            
            if result['success']:
                logger.info(f"Password reset email sent to {user['email']}")
            else:
                logger.error(f"Failed to send password reset email to {user['email']}: {result.get('message')}")
        else:
            # Log attempt for security monitoring
            logger.warning(f"Password reset requested for non-existent email: {request.email}")
        
        # Always return the same response to prevent user enumeration
        return BaseResponse(
            success=True,
            message="If the email address exists in our system, a password reset link has been sent."
        )
        
    except Exception as e:
        logger.error(f"Error processing password reset request: {e}")
        # Don't reveal internal errors
        return BaseResponse(
            success=True,
            message="If the email address exists in our system, a password reset link has been sent."
        )


@router.post(
    "/test",
    summary="Send Test Email",
    description="""
    Send a test email to verify email service functionality.
    
    **Purpose:**
    - Test email service configuration
    - Verify SMTP settings
    - Debug email delivery issues
    
    **Access Control:**
    - Requires admin access
    - Only for testing and debugging purposes
    
    **Security:**
    - Admin access required
    - Email sending is logged
    - Rate limiting applied
    """,
    response_model=BaseResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Test email sent successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "Test email sent successfully to test@example.com"
                    }
                }
            }
        },
        400: {"description": "Validation error", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse},
        500: {"description": "Email service error", "model": ErrorResponse}
    }
)
async def send_test_email(
    request: TestEmailRequest,
    current_user: Dict[str, Any] = Depends(admin_access)
):
    """
    Send a test email.
    
    **Example Request:**
    ```json
    {
        "to": "test@example.com",
        "subject": "Test Email from Krystal EA System",
        "message": "This is a test email to verify the email service is working correctly.",
        "is_html": false
    }
    ```
    
    **Example Response:**
    ```json
    {
        "success": true,
        "message": "Test email sent successfully to test@example.com"
    }
    ```
    """
    try:
        # Import here to avoid circular imports
        from backend.email_service import email_service
        
        # Send test email
        result = email_service.send_email(
            to=request.to,
            subject=request.subject,
            message=request.message,
            is_html=request.is_html
        )
        
        if result['success']:
            logger.info(f"Test email sent to {request.to} by {current_user.get('email')}")
            return BaseResponse(
                success=True,
                message=f"Test email sent successfully to {request.to}"
            )
        else:
            raise EmailServiceError(result.get('message', 'Failed to send test email'))
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error sending test email: {e}")
        raise HTTPException(status_code=500, detail="Failed to send test email")


@router.get(
    "/status",
    summary="Get Email Service Status",
    description="""
    Get the current status of the email service.
    
    **Returns:**
    - Email service configuration status
    - SMTP connection status
    - Recent email sending statistics
    
    **Access Control:**
    - Requires admin access
    - For monitoring and debugging purposes
    """,
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Email service status retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "service_enabled": True,
                        "smtp_configured": True,
                        "last_email_sent": "2026-01-04T12:00:00Z",
                        "emails_sent_today": 15,
                        "configuration": {
                            "smtp_host": "smtp.gmail.com",
                            "smtp_port": 587,
                            "from_email": "system@krystal-ea.com"
                        }
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def get_email_service_status(
    current_user: Dict[str, Any] = Depends(admin_access)
):
    """
    Get email service status and configuration.
    
    **Example Response:**
    ```json
    {
        "service_enabled": true,
        "smtp_configured": true,
        "configuration": {
            "smtp_host": "smtp.gmail.com",
            "smtp_port": 587,
            "from_email": "system@krystal-ea.com"
        }
    }
    ```
    """
    try:
        # Import here to avoid circular imports
        from backend.email_service import email_service
        from app.core.config import get_settings
        
        settings = get_settings()
        
        # Get email service status
        status_info = {
            "service_enabled": email_service.mail_enabled,
            "smtp_configured": bool(settings.smtp_host and settings.smtp_port),
            "debug_mode": email_service.debug,
            "configuration": {
                "smtp_host": settings.smtp_host,
                "smtp_port": settings.smtp_port,
                "from_email": email_service.from_email,
                "from_name": email_service.from_name,
                "php_endpoint": email_service.php_endpoint
            }
        }
        
        logger.info(f"Email service status requested by {current_user.get('email')}")
        return status_info
        
    except Exception as e:
        logger.error(f"Error getting email service status: {e}")
        raise HTTPException(status_code=500, detail="Failed to get email service status")