"""
Batch processing API endpoints.
Handles batch data submission, approvals, and batch-related operations.
"""

from fastapi import APIRouter, HTTPException, Request, Query, Depends, status
from typing import Optional, Dict, Any, List
import json
import logging
from app.models.schemas import (
    BackendRequest,
    BatchSubmitRequest,
    BatchSubmitResponse,
    BatchItemsRequest,
    BatchItemsBulkRequest,
    ApproveItemRequest,
    ListItemsRequest,
    UpdateRowRequest,
    GenerateReportRequest,
    BatchStatisticsRequest,
    BatchStatisticsResponse,
    BatchItemResponse,
    ProcessedDataRequest,
    ProcessedDataResponse,
    ProductIngredientResponse,
    UpdateIngredientRequest,
    BaseResponse,
    ErrorResponse,
    PaginatedResponse
)
from app.services.batch_service import batch_service
from app.core.dependencies import (
    get_current_user,
    RoleBasedAccess
)
from app.core.exceptions import (
    BatchNotFoundError,
    ValidationError,
    BusinessLogicError,
    AuthorizationError,
    BaseAPIException
)
from app.core.config import get_settings

logger = logging.getLogger(__name__)
router = APIRouter(prefix="/batches", tags=["Batch Processing"])

# Max rows per batch submit (avoid huge payloads)
MAX_BATCH_ROWS = 1000

# Role-based access control instances
operator_access = RoleBasedAccess(allowed_roles=['operator', 'analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead'])
analyst_access = RoleBasedAccess(allowed_roles=['analyst', 'team_lead', 'head_of_analyst', 'head_of_team_lead'])
management_access = RoleBasedAccess(allowed_roles=['admin', 'super_admin', 'head_of_analyst', 'head_of_team_lead'])


@router.post(
    "/submit",
    summary="Submit Batch Data (v1)",
    description="""
    Submit batch weighing data. **JWT only** (no CSRF). Use this endpoint; ignore legacy /api/backend.

    **Request body (application/json):**
    - **tableData**: list of rows. Each row: **item**, **actual_weight**, **expected_weight** (optional), **comment** (optional), bn, date_manufacturer, expiry_date, mdn.
    - **operator**, **analyst**, **teamLead**: emails.
    - **product**, **factory**: codes.

    **expected_weight** and **comment** in each row are stored in the DB as sent.
    """,
    response_model=BatchSubmitResponse,
    status_code=status.HTTP_201_CREATED,
    responses={
        201: {"description": "Batch submitted"},
        422: {"description": "Invalid request body"},
        401: {"description": "Not authenticated"},
        403: {"description": "Forbidden"},
    },
    openapi_extra={
        "requestBody": {
            "content": {
                "application/json": {
                    "schema": {"$ref": "#/components/schemas/BatchSubmitRequest"},
                    "example": {
                        "tableData": [
                            {
                                "item": "Sodium Carbonate",
                                "actual_weight": "24",
                                "expected_weight": "12.75",
                                "comment": "Operator note",
                                "bn": "",
                                "date_manufacturer": "2026-01-28",
                                "expiry_date": "2027-01-28",
                                "mdn": "jhgv9"
                            }
                        ],
                        "operator": "user@example.com",
                        "analyst": "user@example.com",
                        "teamLead": "user@example.com",
                        "product": "FG001234",
                        "factory": "K013"
                    }
                }
            }
        }
    },
)
async def submit_batch(
    request: Request,
    current_user: Dict[str, Any] = Depends(operator_access),
):
    """Accept raw JSON body only; no Pydantic on request so body is never altered. Validate shape manually."""
    settings = get_settings()
    try:
        body_bytes = await request.body()
        raw_str = body_bytes.decode("utf-8")
        try:
            body = json.loads(raw_str)
        except json.JSONDecodeError as e:
            raise HTTPException(status_code=422, detail=[{"loc": ["body"], "msg": str(e)}])
        if not isinstance(body, dict):
            raise HTTPException(status_code=422, detail=[{"loc": ["body"], "msg": "Body must be a JSON object"}])

        t = body.get("tableData")
        if not t or not isinstance(t, list) or len(t) == 0:
            raise HTTPException(status_code=422, detail=[{"loc": ["body", "tableData"], "msg": "tableData must be a non-empty list"}])
        if len(t) > MAX_BATCH_ROWS:
            raise HTTPException(
                status_code=422,
                detail=[{"loc": ["body", "tableData"], "msg": f"tableData may have at most {MAX_BATCH_ROWS} rows"}],
            )
        for i, row in enumerate(t):
            if not isinstance(row, dict) or not (row.get("item") or "").strip() or not str(row.get("actual_weight") or "").strip():
                raise HTTPException(status_code=422, detail=[{"loc": ["body", "tableData", i], "msg": "Each row must have item and actual_weight"}])

        if not (body.get("product") or "").strip():
            raise HTTPException(status_code=422, detail=[{"loc": ["body", "product"], "msg": "product is required"}])
        if not (body.get("operator") or "").strip():
            raise HTTPException(status_code=422, detail=[{"loc": ["body", "operator"], "msg": "operator is required"}])
        if not (body.get("analyst") or "").strip():
            raise HTTPException(status_code=422, detail=[{"loc": ["body", "analyst"], "msg": "analyst is required"}])
        if not (body.get("teamLead") or "").strip():
            raise HTTPException(status_code=422, detail=[{"loc": ["body", "teamLead"], "msg": "teamLead is required"}])

        if settings.debug:
            logger.info(
                "[BATCH] raw body length=%d, first row keys=%s",
                len(raw_str), list((t[0] or {}).keys()) if t and isinstance(t[0], dict) else [],
            )
        result = batch_service.submit_batch(body, submitted_by=current_user.get("id"))
        if settings.debug:
            first = t[0] if t and isinstance(t[0], dict) else {}
            result["debug_first_row_keys"] = list(first.keys())
            result["debug_raw_body_snippet"] = raw_str[:600] if raw_str else None
            result["debug_parsed_first_row"] = dict(first)
        else:
            result = {k: v for k, v in result.items() if not k.startswith("debug_")}
        logger.info("Batch submitted by %s: %s items", current_user.get("email"), result["items_created"])
        return BatchSubmitResponse(**result)
    except HTTPException:
        raise
    except ValidationError as e:
        raise HTTPException(status_code=e.status_code, detail=e.message or "Validation error")
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error("Error submitting batch: %s", e)
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/debug/recent-data",
    summary="[Debug] Recent data table rows",
    description="Returns the most recent rows from the data table (id, items, actual_weight, expected_weight, comment, etc.) to verify what is stored in the DB.",
    response_model=List[Dict[str, Any]],
    dependencies=[Depends(operator_access)]
)
async def get_recent_data_table(
    limit: int = Query(20, ge=1, le=100, description="Number of rows to return")
):
    """Debug endpoint: pull recent rows from the data table to inspect expected_weight and comment."""
    try:
        rows = batch_service.get_recent_data_table_rows(limit=limit)
        return rows
    except Exception as e:
        logger.error(f"Error fetching recent data: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post(
    "/items",
    summary="Get Batch Items",
    description="""
    Retrieve batch items based on query parameters.
    
    **Access Control:**
    - **Admin/Super Admin**: Can see all batch items
    - **Head roles**: Can see items in their areas
    - **Analysts/Team Leads**: Can see items assigned to them
    - **Operators**: Can see only their own items
    
    **Query Parameters:**
    - Serial number for specific batch
    - Shift filter
    - Operator filter (automatically applied based on user role)
    """,
    response_model=List[BatchItemResponse],
    responses={
        200: {
            "description": "Batch items retrieved successfully",
            "content": {
                "application/json": {
                    "example": [
                        {
                            "id": 123,
                            "product": "vaseline_extra_strength",
                            "items": "Water",
                            "batch_no": "BATCH001",
                            "expected_weight": "50.0",
                            "actual_weight": "50.5",
                            "operator": "operator@example.com",
                            "analyst": "analyst@example.com",
                            "serial_no": "2026_1_5_1",
                            "shift": "Day Shift",
                            "status": "1",
                            "analyst_status": "0",
                            "tl_status": "0"
                        }
                    ]
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Access denied", "model": ErrorResponse}
    }
)
async def get_batch_items(
    batch_query: BatchItemsRequest,
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get batch items based on query parameters.
    
    **Example Request:**
    ```json
    {
        "serial_no": "2026_1_5_1",
        "shift": "Day Shift",
        "operator": "operator@example.com"
    }
    ```
    """
    try:
        items = batch_service.get_batch_items(batch_query, current_user)
        return items
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting batch items: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/items/bulk",
    summary="Get Multiple Batch Items",
    description="""
    Retrieve multiple batch items with different query parameters in a single request.
    
    **Use Cases:**
    - Fetching multiple batches at once
    - Comparing different shifts or operators
    - Bulk data retrieval for reporting
    
    **Limitations:**
    - Maximum 100 queries per request
    - Same access control rules apply to each query
    """,
    response_model=List[List[BatchItemResponse]],
    responses={
        200: {
            "description": "Bulk batch items retrieved successfully"
        },
        400: {"description": "Too many queries or validation error", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse}
    }
)
async def get_batch_items_bulk(
    bulk_request: BatchItemsBulkRequest,
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get multiple batch items with different query parameters.
    
    **Example Request:**
    ```json
    {
        "items": [
            {"serial_no": "2026_1_5_1", "shift": "Day Shift"},
            {"serial_no": "2026_1_5_2", "shift": "Day Shift"}
        ]
    }
    ```
    """
    try:
        results = []
        
        for query in bulk_request.items:
            items = batch_service.get_batch_items(query, current_user)
            results.append(items)
        
        return results
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting bulk batch items: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/pending-approvals",
    summary="Get Pending Approvals",
    description="""
    Get items pending approval based on user role and areas with pagination support.
    
    **Role-Based Access:**
    - **Analyst**: Items assigned to them for analyst approval
    - **Team Lead**: Items assigned to them for TL approval  
    - **Head of Analyst**: All analyst pending items in their areas (NT, Test, BW, PC)
    - **Head of Team Lead**: All TL pending items in their areas (NT, Test, BW, PC)
    - **Admin/Super Admin**: All pending items (VIEW ONLY - cannot approve/reject)
    
    **Approval Types:**
    - **analyst**: Items pending analyst approval
    - **tl**: Items pending team lead approval
    - **all**: All pending items (for head/admin roles)
    
    **Pagination:**
    - Use `page` and `limit` parameters to control pagination
    - Response includes pagination metadata
    
    **Frontend Flags:**
    - Each item includes `can_approve` flag to show/hide approve buttons
    - Response includes `user_permissions` for overall UI control
    """,
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Pending approvals retrieved successfully with pagination",
            "content": {
                "application/json": {
                    "examples": {
                        "paginated_response": {
                            "summary": "Paginated Response",
                            "description": "Response with pagination and permission flags",
                            "value": {
                                "items": [
                                    {
                                        "id": 123,
                                        "serial_no": "2026_1_5_1",
                                        "analyst": "analyst@example.com",
                                        "analyst_status": 0,
                                        "factory": "NT",
                                        "can_approve": True
                                    }
                                ],
                                "pagination": {
                                    "page": 1,
                                    "limit": 20,
                                    "total": 45,
                                    "pages": 3,
                                    "has_next": True,
                                    "has_prev": False
                                },
                                "user_permissions": {
                                    "can_approve_items": True,
                                    "role": "analyst",
                                    "areas": ["NT", "Test"],
                                    "view_only": False
                                }
                            }
                        }
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Access denied", "model": ErrorResponse}
    }
)
async def get_pending_approvals(
    approval_type: Optional[str] = Query(None, description="Filter by approval type: 'analyst', 'tl', or 'all'"),
    page: int = Query(1, ge=1, description="Page number (starts from 1)"),
    limit: int = Query(20, ge=1, le=100, description="Items per page (max 100)"),
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get items pending approval with pagination and permission flags.
    
    **Example Response:**
    ```json
    {
        "items": [
            {
                "id": 123,
                "product": "vaseline_extra_strength",
                "analyst": "analyst@example.com",
                "analyst_status": "0",
                "can_approve": true
            }
        ],
        "pagination": {
            "page": 1,
            "limit": 20,
            "total": 45,
            "pages": 3,
            "has_next": true,
            "has_prev": false
        },
        "user_permissions": {
            "can_approve_items": true,
            "role": "analyst",
            "view_only": false
        }
    }
    ```
    """
    try:
        result = batch_service.get_pending_approvals_paginated(
            requesting_user=current_user,
            approval_type=approval_type,
            page=page,
            limit=limit
        )
        return result
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting pending approvals: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/approve",
    summary="Approve/Reject Batch Item",
    description="""
    Approve or reject a batch item.
    
    **Approval Workflow:**
    1. **Analyst Approval**: Analyst reviews and approves/rejects items
    2. **Team Lead Approval**: Team lead reviews analyst-approved items
    
    **Access Control:**
    - Analysts can approve items assigned to them
    - Team leads can approve items assigned to them (after analyst approval)
    - Head roles can approve items in their areas
    - Admin can approve any items
    
    **Security:**
    - Requires CSRF token validation
    - Validates user is assigned to the item
    - Prevents unauthorized approvals
    """,
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Item approved/rejected successfully",
            "content": {
                "application/json": {
                    "examples": {
                        "approval_success": {
                            "summary": "Successful Approval",
                            "description": "Item approved successfully",
                            "value": {
                                "success": True,
                                "message": "Item approved successfully",
                                "item": {
                                    "id": 123,
                                    "serial_no": "2026_1_5_1",
                                    "analyst_status": "1",
                                    "analyst_approved_by": "analyst@example.com",
                                    "analyst_approved_at": "2026-01-05T11:50:40",
                                    "tl_status": "0"
                                }
                            }
                        },
                        "rejection_success": {
                            "summary": "Successful Rejection",
                            "description": "Item rejected successfully",
                            "value": {
                                "success": True,
                                "message": "Item rejected successfully",
                                "item": {
                                    "id": 123,
                                    "serial_no": "2026_1_5_1",
                                    "analyst_status": "2",
                                    "analyst_rejected_by": "analyst@example.com",
                                    "analyst_rejected_at": "2026-01-05T11:50:40",
                                    "analyst_rejection_reason": "Weight variance exceeds acceptable limits"
                                }
                            }
                        }
                    }
                }
            }
        },
        400: {
            "description": "Validation error or missing rejection reason", 
            "content": {
                "application/json": {
                    "examples": {
                        "missing_rejection_reason": {
                            "summary": "Missing Rejection Reason",
                            "description": "Rejection reason is required when action is reject",
                            "value": {
                                "detail": [
                                    {
                                        "loc": ["body", "rejection_reason"],
                                        "msg": "rejection_reason is required when action is \"reject\"",
                                        "type": "value_error"
                                    }
                                ]
                            }
                        },
                        "invalid_approval_with_reason": {
                            "summary": "Invalid Approval with Rejection Reason",
                            "description": "Rejection reason should not be provided when approving",
                            "value": {
                                "detail": [
                                    {
                                        "loc": ["body", "rejection_reason"],
                                        "msg": "rejection_reason should not be provided when action is \"approve\"",
                                        "type": "value_error"
                                    }
                                ]
                            }
                        }
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Access denied", "model": ErrorResponse},
        404: {"description": "Item not found", "model": ErrorResponse}
    }
)
async def approve_item(
    request: Request,
    approval_data: ApproveItemRequest,
    current_user: Dict[str, Any] = Depends(analyst_access)
):
    """
    Approve or reject a batch item.
    
    **Example Approval Request:**
    ```json
    {
        "serial_no": "2026_1_5_1",
        "operator": "operator@example.com",
        "shift": "Day Shift",
        "analyst": "analyst@example.com",
        "type": "analyst",
        "action": "approve"
    }
    ```
    
    **Example Rejection Request:**
    ```json
    {
        "serial_no": "2026_1_5_1",
        "operator": "operator@example.com",
        "shift": "Day Shift",
        "analyst": "analyst@example.com",
        "type": "analyst",
        "action": "reject",
        "rejection_reason": "Weight variance exceeds acceptable limits"
    }
    ```
    
    **Validation Rules:**
    - When `action` is "approve": `rejection_reason` must NOT be provided
    - When `action` is "reject": `rejection_reason` is REQUIRED
    - Providing `rejection_reason` with "approve" action will result in validation error
    """
    try:
        # Validate CSRF token
        # Process approval
        result = batch_service.approve_item(approval_data, current_user)
        
        logger.info(f"Item approval by {current_user.get('email')}: {approval_data.action} {approval_data.type}")
        return result
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error approving item: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/processed-data",
    summary="Get Processed Data",
    description="""
    Get processed batch data with role-based filtering and pagination.
    
    **Role-Based Access:**
    - **Admin/Super Admin**: Can see all items (approved and not approved)
    - **Head of Analyst/Team Lead**: Can see items in their areas
    - **Analyst/Team Lead**: Can see items assigned to them
    - **Operator**: Can see their own data only
    
    **Optional Filters:**
    - If no filters provided, returns data based on user role
    - `serial_no`, `shift`, `operator`: Filter by specific values
    - `approval_status`: Filter by approval status
    
    **Approval Status Values:**
    - `approved`: Fully approved items (analyst + TL approved)
    - `not_approved`: Items not fully approved
    - `rejected`: Items rejected by analyst or TL
    - `pending`: Items pending approval
    - `all`: All items (default)
    
    **Pagination:**
    - Default: 20 items per page
    - Maximum: 100 items per page
    """,
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Processed data retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "items": [
                            {
                                "id": 123,
                                "serial_no": "2026_1_5_1",
                                "product": "Product A",
                                "operator": "operator@example.com",
                                "analyst": "analyst@example.com",
                                "teamLead": "tl@example.com",
                                "approval_status": "fully_approved",
                                "scan_date": "2026-01-05",
                                "factory": "NT"
                            }
                        ],
                        "pagination": {
                            "page": 1,
                            "limit": 20,
                            "total": 50,
                            "pages": 3,
                            "has_next": True,
                            "has_prev": False
                        },
                        "user_permissions": {
                            "can_approve_items": True,
                            "role": "head_of_analyst",
                            "areas": ["NT", "Test"],
                            "view_only": False
                        },
                        "summary": {
                            "total_items": 50,
                            "filters_applied": {
                                "serial_no": None,
                                "shift": "Day Shift",
                                "operator": None,
                                "approval_status": "approved"
                            }
                        }
                    }
                }
            }
        },
        400: {
            "description": "Invalid request parameters",
            "content": {
                "application/json": {
                    "examples": {
                        "invalid_approval_status": {
                            "summary": "Invalid Approval Status",
                            "description": "Invalid approval_status value provided",
                            "value": {
                                "detail": [
                                    {
                                        "loc": ["body", "approval_status"],
                                        "msg": "string does not match regex \"^(approved|not_approved|rejected|pending|all)$\"",
                                        "type": "value_error.regex"
                                    }
                                ]
                            }
                        }
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Access denied", "model": ErrorResponse}
    }
)
async def get_processed_data(
    request_data: ProcessedDataRequest,
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get processed batch data with role-based filtering and pagination.
    
    **Example Request Body:**
    ```json
    {
        "serial_no": "2026_1_5_1",
        "shift": "Day Shift",
        "operator": "operator@example.com",
        "approval_status": "approved",
        "page": 1,
        "limit": 20
    }
    ```
    
    **Example Request Body (All data for user's role):**
    ```json
    {
        "page": 1,
        "limit": 50
    }
    ```
    
    **Example Request Body (Filter by approval status):**
    ```json
    {
        "approval_status": "pending",
        "page": 1,
        "limit": 10
    }
    ```
    
    **Response includes:**
    - `items`: Array of batch items with approval_status field
    - `pagination`: Pagination metadata
    - `user_permissions`: User's role and permissions
    - `summary`: Applied filters and total count
    """
    try:
        result = batch_service.get_processed_data(
            requesting_user=current_user,
            serial_no=request_data.serial_no,
            shift=request_data.shift,
            operator=request_data.operator,
            approval_status=request_data.approval_status if request_data.approval_status != "all" else None,
            page=request_data.page,
            limit=request_data.limit
        )
        
        return result
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting processed data: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/statistics",
    summary="Get Batch Statistics",
    description="""
    Get batch processing statistics and analytics.
    
    **Access Control:**
    - Requires management access (admin, super_admin, head roles)
    
    **Statistics Include:**
    - Total items processed
    - Items by approval status
    - Items by product
    - Items by operator
    - Time-based filtering available
    
    **Optional Filters:**
    - `date_from`: Start date for filtering (YYYY-MM-DD format)
    - `date_to`: End date for filtering (YYYY-MM-DD format)
    - `product`: Filter by specific product name
    """,
    response_model=BatchStatisticsResponse,
    responses={
        200: {
            "description": "Statistics retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "total_items": 150,
                        "items_by_status": {
                            "pending_analyst": 25,
                            "pending_tl": 15,
                            "approved": 100,
                            "rejected": 10
                        },
                        "items_by_product": {
                            "vaseline_extra_strength": 75,
                            "product_b": 50,
                            "product_c": 25
                        },
                        "items_by_operator": {
                            "operator1@example.com": 60,
                            "operator2@example.com": 45,
                            "operator3@example.com": 45
                        }
                    }
                }
            }
        },
        400: {
            "description": "Invalid request parameters",
            "content": {
                "application/json": {
                    "examples": {
                        "invalid_date_format": {
                            "summary": "Invalid Date Format",
                            "description": "Date must be in YYYY-MM-DD format",
                            "value": {
                                "detail": [
                                    {
                                        "loc": ["body", "date_from"],
                                        "msg": "string does not match regex \"^\\d{4}-\\d{2}-\\d{2}$\"",
                                        "type": "value_error.regex"
                                    }
                                ]
                            }
                        }
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Management access required", "model": ErrorResponse}
    }
)
async def get_batch_statistics(
    request_data: BatchStatisticsRequest,
    current_user: Dict[str, Any] = Depends(management_access)
):
    """
    Get batch processing statistics with optional filtering.
    
    **Example Request Bodies:**
    
    ```json
    // Get all statistics
    {}
    
    // Filter by date range
    {
        "date_from": "2026-01-01",
        "date_to": "2026-01-31"
    }
    
    // Filter by product and date
    {
        "date_from": "2026-01-01",
        "date_to": "2026-01-31",
        "product": "vaseline_extra_strength"
    }
    
    // Filter by product only
    {
        "product": "vaseline_extra_strength"
    }
    ```
    
    **Response includes:**
    - `total_items`: Total number of items matching filters
    - `items_by_status`: Breakdown by approval status
    - `items_by_product`: Breakdown by product
    - `items_by_operator`: Breakdown by operator (top 10)
    """
    try:
        stats = batch_service.get_batch_statistics(
            current_user, 
            request_data.date_from, 
            request_data.date_to, 
            request_data.product
        )
        
        return BatchStatisticsResponse(**stats)
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting batch statistics: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")