# Krystal EA Digital Weighing Scale API

A modern FastAPI backend for the Krystal EA Digital Weighing Scale System, providing comprehensive batch processing, user management, and approval workflows for industrial weighing operations.

## 🚀 Features

- **JWT Authentication** - Secure token-based authentication
- **Role-Based Access Control** - Hierarchical permission system
- **Batch Processing** - Complete weighing data management with approval workflows
- **User Management** - Comprehensive user administration
- **Activity Tracking** - Complete audit trail and session management
- **Email Integration** - Automated notifications and password reset
- **SAP Integration** - Bidirectional communication with SAP ECC via BTP
- **Real-time Statistics** - Analytics and reporting capabilities
- **RESTful API** - Clean, documented API endpoints
- **Interactive Documentation** - Swagger UI with custom branding

## 📋 Table of Contents

- [Quick Start](#quick-start)
- [Installation](#installation)
- [Configuration](#configuration)
- [API Documentation](#api-documentation)
- [Architecture](#architecture)
- [Authentication](#authentication)
- [Database Schema](#database-schema)
- [SAP Integration](#sap-integration)
- [Development](#development)
- [Testing](#testing)
- [Deployment](#deployment)
- [Contributing](#contributing)

## 🚀 Quick Start

1. **Clone and Setup**
   ```bash
   cd backend
   pip install -r requirements.txt
   ```

2. **Configure Environment**
   ```bash
   cp .env.example .env
   # Edit .env with your database and JWT settings
   ```

3. **Run the Server**
   ```bash
   python main.py
   # or
   uvicorn app.main:app --reload --host 0.0.0.0 --port 8000
   ```

4. **Access API Documentation**
   - Swagger UI: http://localhost:8000/docs
   - ReDoc: http://localhost:8000/redoc

## 📦 Installation

### Prerequisites

- Python 3.8+
- MySQL 5.7+ or MariaDB 10.3+
- pip (Python package manager)

### Install Dependencies

```bash
pip install -r requirements.txt
```

### Required Python Packages

```
fastapi>=0.104.1
uvicorn[standard]>=0.24.0
pydantic>=2.5.0
python-jose[cryptography]>=3.3.0
passlib[bcrypt]>=1.7.4
python-multipart>=0.0.6
python-dotenv>=1.0.0
PyMySQL>=1.1.0
cryptography>=41.0.0
email-validator>=2.1.0
```

## ⚙️ Configuration

### Environment Variables

Create a `.env` file in the backend directory:

```env
# Application
APP_NAME=Krystal EA API
APP_VERSION=1.0.0
DEBUG=false

# Database
DB_HOST=localhost
DB_USER=your_db_user
DB_PASSWORD=your_db_password
DB_NAME=your_db_name
DB_PORT=3306

# JWT Configuration
JWT_SECRET_KEY=your-super-secret-jwt-key-at-least-32-characters-long
JWT_ALGORITHM=HS256
JWT_EXPIRATION_HOURS=24

# CORS Origins (comma-separated)
CORS_ORIGINS=http://localhost:8080,http://localhost:3000

# Email Configuration (SMTP)
SMTP_HOST=smtp.mailsafi.com
SMTP_PORT=587
SMTP_SECURE=tls
SMTP_USERNAME=systems@krystal-ea.com
SMTP_PASSWORD=your_smtp_password
SMTP_FROM_EMAIL=systems@krystal-ea.com
SMTP_FROM_NAME=Krystal EA System
MAIL_ENABLED=true

# Application URLs
BASE_URL=http://localhost:8000
FRONTEND_URL=http://localhost:3000
LOGIN_URL=http://localhost:3000/login

# Timezone
TIMEZONE=Africa/Nairobi

# SAP Integration Configuration
SAP_BTP_BASE_URL=https://your-btp-endpoint.com
SAP_BTP_CLIENT_ID=your_btp_client_id
SAP_BTP_CLIENT_SECRET=your_btp_client_secret
SAP_INTEGRATION_ENABLED=false
SAP_TIMEOUT=30

# API Key Configuration for External Systems
API_KEY_HEADER=X-API-Key
SAP_API_KEY=your-super-secure-sap-api-key-at-least-32-characters-long

# Logging
LOG_LEVEL=INFO
```

### Configuration Validation

The application uses Pydantic for configuration validation:
- JWT secret key must be at least 32 characters
- Database connection is tested on startup
- Email settings are validated if mail is enabled

## 📚 API Documentation

### Interactive Documentation

- **Swagger UI**: http://localhost:8000/docs
- **ReDoc**: http://localhost:8000/redoc
- **OpenAPI JSON**: http://localhost:8000/openapi.json

### Documentation Files

- **[API_DOCUMENTATION.md](API_DOCUMENTATION.md)**: Complete API reference for user-facing endpoints
- **[EXTERNAL_API_DOCUMENTATION.md](EXTERNAL_API_DOCUMENTATION.md)**: API reference for external systems (BTP/SAP)
- **[SAP_INTEGRATION_GUIDE.md](SAP_INTEGRATION_GUIDE.md)**: Comprehensive SAP integration guide

### API Endpoints Overview

#### Authentication (`/api/v1/auth`)
- `POST /login` - User authentication
- `GET /logout` - User logout
- `GET /me` - Get current user info
- `GET /protected-test` - Test authentication

#### User Management (`/api/v1/users`)
- `GET /` - List users with filtering
- `GET /{user_id}` - Get specific user
- `POST /` - Create new user
- `PUT /{user_id}` - Update user
- `DELETE /{user_id}` - Delete user
- `GET /statistics/overview` - User statistics

#### Batch Processing (`/api/v1/batches`)
- `POST /submit` - Submit batch data
- `POST /items` - Get batch items
- `POST /items/bulk` - Bulk batch retrieval
- `GET /pending-approvals` - Get pending approvals
- `POST /approve` - Approve/reject items
- `POST /processed-data` - Get processed data
- `POST /statistics` - Batch statistics

#### Products (`/api/v1/products`)
- `GET /` - List products
- `GET /{product_name}/ingredients` - Get ingredients
- `PUT /{product_name}/ingredients/{ingredient_id}` - Update ingredients
- `POST /list-items` - List items by batch

#### Email Service (`/api/v1/emails`)
- `POST /send-welcome` - Send welcome email
- `POST /forgot-password` - Password reset email
- `POST /test` - Test email service

#### Activity Logs (`/api/v1/activity-logs`)
- `POST /user-activities` - Get user activities
- `POST /active-sessions` - Get active sessions
- `POST /statistics` - Activity statistics
- `GET /my-activities` - Personal activity history
- `GET /my-sessions` - Personal session history

#### Login Logs (`/api/v1/login-logs`)
- `POST /` - Get login logs
- `POST /statistics` - Login statistics
- `GET /history` - Login history

#### SAP Integration (`/api/v1/sap`)
**User Endpoints (JWT Authentication):**
- `POST /material-trigger` - Trigger material request to SAP
- `POST /component-consumption` - Send component consumption (MAK080)
- `GET /plants` - Get valid plants
- `GET /order-types/{plant}` - Get order types for plant

**External Endpoints (API Key Authentication):**
- `POST /receive-process-orders` - Receive process orders from SAP
- `POST /receive-aleaud-acknowledgment` - Receive ALEAUD acknowledgment

> **Note**: For external system integration (BTP/SAP), see [EXTERNAL_API_DOCUMENTATION.md](EXTERNAL_API_DOCUMENTATION.md)

### Authentication

All API endpoints (except `/auth/login`) require JWT authentication:

```bash
# Login to get token
curl -X POST "http://localhost:8000/api/v1/auth/login" \
  -H "Content-Type: application/json" \
  -d '{"email": "user@example.com", "password": "password"}'

# Use token in subsequent requests
curl -X GET "http://localhost:8000/api/v1/users" \
  -H "Authorization: Bearer YOUR_JWT_TOKEN"
```

## 🏗️ Architecture

### Project Structure

```
backend/
├── app/
│   ├── api/v1/              # API endpoints
│   │   ├── auth.py          # Authentication
│   │   ├── users.py         # User management
│   │   ├── batches.py       # Batch processing
│   │   ├── products.py      # Product management
│   │   ├── emails.py        # Email service
│   │   ├── activity_logs.py # Activity tracking
│   │   └── login_logs.py    # Login tracking
│   ├── services/            # Business logic
│   │   ├── auth_service.py
│   │   ├── user_service.py
│   │   ├── batch_service.py
│   │   ├── activity_logs_service.py
│   │   └── login_logs_service.py
│   ├── repositories/        # Data access layer
│   │   ├── base.py
│   │   ├── user_repository.py
│   │   ├── batch_repository.py
│   │   ├── active_sessions_repository.py
│   │   ├── user_activities_repository.py
│   │   └── login_logs_repository.py
│   ├── models/              # Data models
│   │   └── schemas.py       # Pydantic schemas
│   ├── core/                # Core functionality
│   │   ├── config.py        # Configuration
│   │   ├── dependencies.py  # FastAPI dependencies
│   │   ├── exceptions.py    # Custom exceptions
│   │   └── error_handlers.py # Error handling
│   ├── utils/               # Utilities
│   │   └── database.py      # Database utilities
│   ├── static/              # Static files
│   └── main.py              # FastAPI application
├── tests/                   # Test suite
├── requirements.txt         # Dependencies
├── main.py                  # Entry point
└── README.md               # This file
```

### Design Patterns

- **Repository Pattern**: Data access abstraction
- **Service Layer**: Business logic separation
- **Dependency Injection**: FastAPI dependencies
- **Factory Pattern**: Role-based access control
- **Observer Pattern**: Activity logging

### Key Components

1. **FastAPI Application**: Modern async web framework
2. **Pydantic Models**: Data validation and serialization
3. **JWT Authentication**: Stateless token-based auth
4. **MySQL Database**: Relational data storage
5. **SAP Integration**: Bidirectional communication with SAP ECC via BTP
6. **CORS Middleware**: Cross-origin request handling
7. **Custom Error Handling**: Structured error responses
8. **Logging System**: Comprehensive activity tracking

## 🔐 Authentication & Authorization

### JWT Authentication

The API uses JWT (JSON Web Tokens) for stateless authentication:

- **Token Expiration**: 24 hours (configurable)
- **Algorithm**: HS256
- **Claims**: User ID, email, name, role, areas

### Role-Based Access Control

Hierarchical permission system:

1. **Super Admin**: Full system access
2. **Admin**: Administrative functions
3. **Head of Analyst**: Area-based management
4. **Head of Team Lead**: Team management
5. **Team Lead**: Item approval authority
6. **Analyst**: Data analysis and initial approval
7. **Operator**: Data entry only
8. **Regular User**: Profile access only

### Area-Based Filtering

Users see data only from their assigned operational areas:
- Food, BW (Body Wash), Soap, etc.
- Configurable per user
- Enforced at the service layer

## 🏭 SAP Integration

The system provides comprehensive integration with SAP ECC via Business Technology Platform (BTP):

### Integration Capabilities

- **Material Triggers**: Request process orders from SAP
- **Component Consumption**: Post goods issue transactions (MAK080)
- **Process Orders**: Receive production order data from SAP
- **Acknowledgments**: Receive transaction status updates (ALEAUD)

### Supported Plants

- **K002**: Kenya Plant 002
- **K013**: Kenya Plant 013  
- **K012**: Kenya Plant 012

### Integration Flow

```
Krystal EA ←→ BTP Middleware ←→ SAP ECC
```

1. Krystal EA sends JSON requests to BTP
2. BTP transforms JSON to IDOC format for SAP
3. SAP ECC processes IDOCs and sends responses
4. BTP transforms SAP responses back to JSON
5. Krystal EA receives and processes responses

### Configuration

```env
SAP_BTP_BASE_URL=https://your-btp-endpoint.com
SAP_BTP_CLIENT_ID=your_btp_client_id
SAP_BTP_CLIENT_SECRET=your_btp_client_secret
SAP_INTEGRATION_ENABLED=true
API_KEY_HEADER=X-API-Key
SAP_API_KEY=your-super-secure-sap-api-key-at-least-32-characters-long
```

### Authentication Methods

The SAP integration uses two authentication methods:

- **JWT Authentication**: For user-facing endpoints (material-trigger, component-consumption)
- **API Key Authentication**: For external endpoints called by BTP (receive-process-orders, receive-aleaud-acknowledgment)

### API Key Generation

Generate a secure API key for SAP integration:

```bash
# Use the provided utility script
python backend/scripts/generate_api_key.py

# Test your API key setup
python backend/scripts/test_external_api.py

# Or generate manually (64+ characters recommended)
python -c "import secrets, string; print(''.join(secrets.choice(string.ascii_letters + string.digits + '-_') for _ in range(64)))"
```

For detailed SAP integration information, see [SAP_INTEGRATION_GUIDE.md](SAP_INTEGRATION_GUIDE.md).

For external system integration, see [EXTERNAL_API_DOCUMENTATION.md](EXTERNAL_API_DOCUMENTATION.md).

## 🗄️ Database Schema

### Core Tables

#### Users Table
```sql
users (
  id INT PRIMARY KEY,
  user_id VARCHAR(50) UNIQUE,
  name VARCHAR(100),
  email VARCHAR(100) UNIQUE,
  password VARCHAR(255),
  title VARCHAR(50),        -- Role/position
  shift VARCHAR(20),        -- Day/Night shift
  areas TEXT,              -- Comma-separated areas
  status TINYINT,          -- 1=active, 0=inactive
  imglink VARCHAR(255),    -- Profile image
  created_at TIMESTAMP,
  updated_at TIMESTAMP
)
```

#### Batch Data Table
```sql
data (
  id INT PRIMARY KEY,
  serial_number VARCHAR(50),
  product_name VARCHAR(100),
  batch_number VARCHAR(50),
  ingredient_name VARCHAR(100),
  target_weight DECIMAL(10,3),
  actual_weight DECIMAL(10,3),
  variance DECIMAL(10,3),
  operator_email VARCHAR(100),
  analyst_email VARCHAR(100),
  tl_email VARCHAR(100),
  shift VARCHAR(20),
  date_created DATE,
  time_created TIME,
  
  -- Approval workflow columns
  analyst_approved_by VARCHAR(100),
  analyst_approved_at TIMESTAMP,
  analyst_rejected_by VARCHAR(100),
  analyst_rejected_at TIMESTAMP,
  analyst_rejection_reason TEXT,
  
  tl_approved_by VARCHAR(100),
  tl_approved_at TIMESTAMP,
  tl_rejected_by VARCHAR(100),
  tl_rejected_at TIMESTAMP,
  tl_rejection_reason TEXT,
  
  created_at TIMESTAMP,
  updated_at TIMESTAMP
)
```

#### Activity Tracking Tables
```sql
user_activities (
  id INT PRIMARY KEY,
  user_id VARCHAR(50),
  activity_type VARCHAR(50),
  description TEXT,
  ip_address VARCHAR(45),
  user_agent TEXT,
  created_at TIMESTAMP
)

active_sessions (
  id INT PRIMARY KEY,
  user_id VARCHAR(50),
  session_token VARCHAR(255),
  ip_address VARCHAR(45),
  user_agent TEXT,
  created_at TIMESTAMP,
  last_activity TIMESTAMP
)

login_logs (
  id INT PRIMARY KEY,
  user_id VARCHAR(50),
  email VARCHAR(100),
  success BOOLEAN,
  failure_reason VARCHAR(255),
  ip_address VARCHAR(45),
  user_agent TEXT,
  created_at TIMESTAMP
)
```

## 🛠️ Development

### Running in Development Mode

```bash
# Install dependencies
pip install -r requirements.txt

# Run with auto-reload
uvicorn app.main:app --reload --host 0.0.0.0 --port 8000

# Or use the main.py entry point
python main.py
```

### Development Tools

- **FastAPI**: Modern web framework
- **Uvicorn**: ASGI server with auto-reload
- **Pydantic**: Data validation
- **PyMySQL**: MySQL database driver
- **python-jose**: JWT handling
- **passlib**: Password hashing

### Code Style

- **Type Hints**: All functions use type annotations
- **Docstrings**: Comprehensive documentation
- **Error Handling**: Custom exceptions with proper HTTP codes
- **Logging**: Strategic logging throughout the application
- **Validation**: Pydantic models for all data

### Adding New Endpoints

1. **Create Pydantic Models** in `app/models/schemas.py`
2. **Add Repository Methods** in appropriate repository
3. **Implement Service Logic** in service layer
4. **Create API Endpoints** in `app/api/v1/`
5. **Add Tests** in `tests/`

Example:
```python
# 1. Schema
class ItemCreate(BaseModel):
    name: str
    description: Optional[str] = None

# 2. Repository method
def create_item(self, item_data: dict) -> dict:
    # Database logic here
    pass

# 3. Service method
def create_item(self, item_data: ItemCreate) -> dict:
    # Business logic here
    return self.repository.create_item(item_data.dict())

# 4. API endpoint
@router.post("/items")
async def create_item(item: ItemCreate, current_user: dict = Depends(get_current_user)):
    return service.create_item(item)
```

## 🧪 Testing

### Running Tests

```bash
# Run all tests
python -m pytest

# Run with coverage
python -m pytest --cov=app

# Run specific test file
python -m pytest tests/test_auth.py

# Run with verbose output
python -m pytest -v
```

### Test Structure

```
tests/
├── test_auth.py         # Authentication tests
├── test_users.py        # User management tests
├── test_batches.py      # Batch processing tests
└── conftest.py          # Test configuration
```

### Test Coverage

Current test coverage includes:
- Authentication service and API
- User management operations
- Batch processing workflows
- Error handling scenarios

### Writing Tests

```python
import pytest
from fastapi.testclient import TestClient
from app.main import app

client = TestClient(app)

def test_login_success():
    response = client.post("/api/v1/auth/login", json={
        "email": "test@example.com",
        "password": "password"
    })
    assert response.status_code == 200
    assert "access_token" in response.json()
```

## 🚀 Deployment

### Production Setup

1. **Environment Configuration**
   ```bash
   # Set production environment variables
   export DEBUG=false
   export JWT_SECRET_KEY="your-production-secret-key"
   export DB_HOST="your-production-db-host"
   # ... other production settings
   ```

2. **Database Setup**
   ```bash
   # Create production database
   mysql -u root -p
   CREATE DATABASE your_production_db;
   # Import schema and data
   ```

3. **Install Dependencies**
   ```bash
   pip install -r requirements.txt
   ```

4. **Run with Gunicorn**
   ```bash
   pip install gunicorn
   gunicorn app.main:app -w 4 -k uvicorn.workers.UvicornWorker --bind 0.0.0.0:8000
   ```

### Docker Deployment (Recommended)

```dockerfile
FROM python:3.9-slim

WORKDIR /app

COPY requirements.txt .
RUN pip install -r requirements.txt

COPY . .

EXPOSE 8000

CMD ["gunicorn", "app.main:app", "-w", "4", "-k", "uvicorn.workers.UvicornWorker", "--bind", "0.0.0.0:8000"]
```

### Environment Variables for Production

```env
DEBUG=false
JWT_SECRET_KEY=your-super-secure-production-key
DB_HOST=your-production-db-host
DB_USER=your-production-db-user
DB_PASSWORD=your-production-db-password
DB_NAME=your-production-db-name
CORS_ORIGINS=https://your-frontend-domain.com
SMTP_HOST=your-production-smtp-host
# ... other production settings
```

### Health Checks

The API provides health check endpoints:
- `GET /health` - Application health status
- `GET /` - Basic API information

### Monitoring

Consider implementing:
- Application performance monitoring (APM)
- Error tracking (Sentry)
- Log aggregation (ELK stack)
- Metrics collection (Prometheus)

## 🤝 Contributing

### Development Workflow

1. **Fork the repository**
2. **Create a feature branch**
   ```bash
   git checkout -b feature/your-feature-name
   ```
3. **Make your changes**
4. **Add tests for new functionality**
5. **Run tests and ensure they pass**
   ```bash
   python -m pytest
   ```
6. **Submit a pull request**

### Code Standards

- Follow PEP 8 style guidelines
- Add type hints to all functions
- Write comprehensive docstrings
- Include tests for new features
- Update documentation as needed

### Commit Message Format

```
type(scope): description

Examples:
feat(auth): add password reset functionality
fix(batches): resolve approval workflow bug
docs(readme): update installation instructions
test(users): add user creation tests
```

## 📄 License

This project is proprietary software developed for Krystal EA. All rights reserved.

## 📞 Support

For support and questions:
- **Email**: systems@krystal-ea.com
- **Documentation**: http://localhost:8000/docs
- **API Reference**: http://localhost:8000/redoc

---

**Krystal EA Digital Weighing Scale API** - Built with ❤️ using FastAPI