# Krystal EA API Documentation

Complete API reference for the Krystal EA Digital Weighing Scale System.

> **For External Systems**: If you're integrating external systems (BTP/SAP) with API key authentication, see [EXTERNAL_API_DOCUMENTATION.md](EXTERNAL_API_DOCUMENTATION.md)

## 📋 Table of Contents

- [Authentication](#authentication)
- [User Management](#user-management)
- [Batch Processing](#batch-processing)
- [Product Management](#product-management)
- [Email Service](#email-service)
- [Activity Logs](#activity-logs)
- [Login Logs](#login-logs)
- [SAP Integration](#sap-integration)
- [Error Handling](#error-handling)
- [Rate Limiting](#rate-limiting)
- [Examples](#examples)

## 🔐 Authentication

All API endpoints (except login) require JWT authentication via the `Authorization` header.

### Login

**POST** `/api/v1/auth/login`

Authenticate a user and receive a JWT token.

**Request Body:**
```json
{
  "email": "user@example.com",
  "password": "password123"
}
```

**Response (200):**
```json
{
  "success": true,
  "redirect": "blue.html",
  "message": "Login successful",
  "access_token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
  "token_type": "bearer"
}
```

**Response (401):**
```json
{
  "error": {
    "code": "INVALID_CREDENTIALS",
    "message": "Invalid email or password",
    "timestamp": "2026-01-06T12:00:00Z"
  }
}
```

### Get Current User

**GET** `/api/v1/auth/me`

Get information about the currently authenticated user.

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "id": "user123",
  "user_id": "user123",
  "name": "John Doe",
  "email": "john@example.com",
  "title": "analyst",
  "shift": "Day Shift",
  "areas": "Food,BW",
  "status": 1
}
```

### Logout

**GET** `/api/v1/auth/logout`

Logout the current user (invalidates session).

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "success": true,
  "message": "Logged out successfully"
}
```

## 👥 User Management

### List Users

**GET** `/api/v1/users`

Get a paginated list of users with filtering options.

**Query Parameters:**
- `page` (int, default: 1) - Page number
- `limit` (int, default: 50, max: 100) - Items per page
- `search` (string) - Search by name or email
- `title` (string) - Filter by user role/title
- `areas` (string) - Filter by operational areas
- `status` (int) - Filter by status (1=active, 0=inactive)
- `include_inactive` (bool, default: false) - Include inactive users

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "items": [
    {
      "id": "user123",
      "user_id": "user123",
      "name": "John Doe",
      "email": "john@example.com",
      "title": "analyst",
      "shift": "Day Shift",
      "areas": "Food,BW",
      "status": 1,
      "created_at": "2026-01-01T00:00:00Z"
    }
  ],
  "total": 25,
  "page": 1,
  "limit": 50,
  "pages": 1
}
```

### Get User by ID

**GET** `/api/v1/users/{user_id}`

Get detailed information about a specific user.

**Path Parameters:**
- `user_id` (string) - User's unique identifier

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "id": "user123",
  "user_id": "user123",
  "name": "John Doe",
  "email": "john@example.com",
  "title": "analyst",
  "shift": "Day Shift",
  "areas": "Food,BW",
  "status": 1,
  "imglink": "/images/user123.jpg",
  "created_at": "2026-01-01T00:00:00Z",
  "updated_at": "2026-01-01T00:00:00Z"
}
```

### Create User

**POST** `/api/v1/users`

Create a new user account.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "name": "Jane Smith",
  "email": "jane@example.com",
  "password": "securepassword123",
  "title": "analyst",
  "shift": "Day Shift",
  "areas": "Food,Soap",
  "status": 1
}
```

**Response (201):**
```json
{
  "success": true,
  "message": "User created successfully",
  "data": {
    "id": "user456",
    "user_id": "user456",
    "name": "Jane Smith",
    "email": "jane@example.com",
    "title": "analyst",
    "shift": "Day Shift",
    "areas": "Food,Soap",
    "status": 1,
    "created_at": "2026-01-06T12:00:00Z"
  }
}
```

### Update User

**PUT** `/api/v1/users/{user_id}`

Update an existing user's information.

**Path Parameters:**
- `user_id` (string) - User's unique identifier

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "name": "Jane Smith Updated",
  "title": "team_lead",
  "shift": "Night Shift",
  "areas": "Food,Soap,BW",
  "status": 1
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "User updated successfully",
  "data": {
    "id": "user456",
    "user_id": "user456",
    "name": "Jane Smith Updated",
    "email": "jane@example.com",
    "title": "team_lead",
    "shift": "Night Shift",
    "areas": "Food,Soap,BW",
    "status": 1,
    "updated_at": "2026-01-06T12:30:00Z"
  }
}
```

### Delete User

**DELETE** `/api/v1/users/{user_id}`

Soft delete a user (sets status to inactive).

**Path Parameters:**
- `user_id` (string) - User's unique identifier

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "success": true,
  "message": "User deleted successfully"
}
```

### User Statistics

**GET** `/api/v1/users/statistics/overview`

Get user statistics and analytics.

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "total_users": 150,
  "active_users": 142,
  "inactive_users": 8,
  "users_by_role": {
    "operator": 80,
    "analyst": 35,
    "team_lead": 20,
    "admin": 15
  },
  "users_by_shift": {
    "Day Shift": 95,
    "Night Shift": 55
  },
  "users_by_area": {
    "Food": 60,
    "BW": 45,
    "Soap": 40,
    "Other": 5
  },
  "recent_registrations": 12
}
```

## 📦 Batch Processing

### Submit Batch Data

**POST** `/api/v1/batches/submit`

Submit new batch weighing data from the frontend.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "product_name": "Omo Washing Powder",
  "batch_number": "BATCH001",
  "items": [
    {
      "ingredient_name": "Sodium Carbonate",
      "target_weight": 25.500,
      "actual_weight": 25.485,
      "variance": -0.015
    },
    {
      "ingredient_name": "Linear Alkylbenzene",
      "target_weight": 15.200,
      "actual_weight": 15.210,
      "variance": 0.010
    }
  ]
}
```

**Response (201):**
```json
{
  "success": true,
  "message": "Batch data submitted successfully",
  "data": {
    "serial_number": "26010612345",
    "product_name": "Omo Washing Powder",
    "batch_number": "BATCH001",
    "items_count": 2,
    "operator_email": "operator@example.com",
    "analyst_email": "analyst@example.com",
    "tl_email": "teamlead@example.com",
    "shift": "Day Shift",
    "created_at": "2026-01-06T12:00:00Z"
  }
}
```

### Get Batch Items

**POST** `/api/v1/batches/items`

Retrieve batch items with filtering and pagination.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "page": 1,
  "limit": 50,
  "filters": {
    "product_name": "Omo Washing Powder",
    "batch_number": "BATCH001",
    "date_from": "2026-01-01",
    "date_to": "2026-01-06",
    "shift": "Day Shift",
    "operator_email": "operator@example.com"
  }
}
```

**Response (200):**
```json
{
  "items": [
    {
      "id": 123,
      "serial_number": "26010612345",
      "product_name": "Omo Washing Powder",
      "batch_number": "BATCH001",
      "ingredient_name": "Sodium Carbonate",
      "target_weight": 25.500,
      "actual_weight": 25.485,
      "variance": -0.015,
      "operator_email": "operator@example.com",
      "analyst_email": "analyst@example.com",
      "tl_email": "teamlead@example.com",
      "shift": "Day Shift",
      "date_created": "2026-01-06",
      "time_created": "12:00:00",
      "analyst_approved_by": null,
      "analyst_approved_at": null,
      "tl_approved_by": null,
      "tl_approved_at": null,
      "created_at": "2026-01-06T12:00:00Z"
    }
  ],
  "total": 1,
  "page": 1,
  "limit": 50,
  "pages": 1
}
```

### Get Pending Approvals

**GET** `/api/v1/batches/pending-approvals`

Get items pending approval for the current user.

**Query Parameters:**
- `page` (int, default: 1) - Page number
- `limit` (int, default: 50) - Items per page

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "items": [
    {
      "id": 123,
      "serial_number": "26010612345",
      "product_name": "Omo Washing Powder",
      "batch_number": "BATCH001",
      "ingredient_name": "Sodium Carbonate",
      "target_weight": 25.500,
      "actual_weight": 25.485,
      "variance": -0.015,
      "operator_email": "operator@example.com",
      "shift": "Day Shift",
      "date_created": "2026-01-06",
      "time_created": "12:00:00",
      "approval_level": "analyst",
      "days_pending": 1
    }
  ],
  "total": 5,
  "page": 1,
  "limit": 50,
  "pages": 1
}
```

### Approve/Reject Items

**POST** `/api/v1/batches/approve`

Approve or reject batch items.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "item_ids": [123, 124, 125],
  "action": "approve",
  "rejection_reason": null
}
```

**For rejection:**
```json
{
  "item_ids": [126],
  "action": "reject",
  "rejection_reason": "Weight variance exceeds acceptable limits"
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Items processed successfully",
  "data": {
    "approved_count": 3,
    "rejected_count": 0,
    "processed_items": [123, 124, 125]
  }
}
```

### Get Processed Data

**POST** `/api/v1/batches/processed-data`

Get fully approved batch data for reporting.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "page": 1,
  "limit": 100,
  "filters": {
    "product_name": "Omo Washing Powder",
    "date_from": "2026-01-01",
    "date_to": "2026-01-06",
    "shift": "Day Shift"
  }
}
```

**Response (200):**
```json
{
  "items": [
    {
      "id": 123,
      "serial_number": "26010612345",
      "product_name": "Omo Washing Powder",
      "batch_number": "BATCH001",
      "ingredient_name": "Sodium Carbonate",
      "target_weight": 25.500,
      "actual_weight": 25.485,
      "variance": -0.015,
      "operator_email": "operator@example.com",
      "analyst_approved_by": "analyst@example.com",
      "analyst_approved_at": "2026-01-06T13:00:00Z",
      "tl_approved_by": "teamlead@example.com",
      "tl_approved_at": "2026-01-06T14:00:00Z",
      "shift": "Day Shift",
      "date_created": "2026-01-06",
      "time_created": "12:00:00"
    }
  ],
  "total": 1,
  "page": 1,
  "limit": 100,
  "pages": 1
}
```

### Batch Statistics

**POST** `/api/v1/batches/statistics`

Get batch processing statistics and analytics.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "date_from": "2026-01-01",
  "date_to": "2026-01-06",
  "group_by": "product"
}
```

**Response (200):**
```json
{
  "summary": {
    "total_items": 1250,
    "pending_analyst_approval": 45,
    "pending_tl_approval": 23,
    "fully_approved": 1182,
    "rejected_items": 15
  },
  "by_product": {
    "Omo Washing Powder": {
      "total": 450,
      "approved": 425,
      "pending": 20,
      "rejected": 5
    },
    "Sunlight Soap": {
      "total": 380,
      "approved": 370,
      "pending": 8,
      "rejected": 2
    }
  },
  "by_shift": {
    "Day Shift": {
      "total": 750,
      "approved": 720,
      "pending": 25,
      "rejected": 5
    },
    "Night Shift": {
      "total": 500,
      "approved": 462,
      "pending": 28,
      "rejected": 10
    }
  },
  "approval_times": {
    "avg_analyst_approval_hours": 2.5,
    "avg_tl_approval_hours": 1.8,
    "avg_total_approval_hours": 4.3
  }
}
```

## 🏭 SAP Integration

The API provides comprehensive integration with SAP ECC via Business Technology Platform (BTP) for enterprise resource planning and manufacturing execution.

### Material Trigger Request

**POST** `/api/v1/sap/material-trigger`

Send material number and plant combination to SAP to trigger process order extraction.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "material_number": "FG001234",
  "plant": "K013"
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Material trigger sent to SAP successfully. Process orders will be sent shortly.",
  "material_number": "FG001234",
  "plant": "K013",
  "request_id": "REQ-2026-01-06-001",
  "timestamp": "2026-01-06T12:00:00Z"
}
```

**Valid Plants:**
- `K002` - Kenya Plant 002
- `K013` - Kenya Plant 013
- `K012` - Kenya Plant 012

### Component Consumption (MAK080)

**POST** `/api/v1/sap/component-consumption`

Send component consumption data to SAP after pre-weighing activities.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "document_date": "2026-01-06",
  "posting_date": "2026-01-06",
  "transaction_code": "MB1A",
  "description": "Component consumption for batch processing",
  "components": [
    {
      "material_number": "RM001234",
      "plant": "K013",
      "storage_location": "0001",
      "batch_number": "BATCH001",
      "movement_type": "261",
      "quantity": 25.500,
      "unit_of_measure": "KG",
      "order_number": "000012345678"
    }
  ]
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Component consumption sent to SAP successfully",
  "idoc_number": "0000000001234567",
  "components_count": 1,
  "request_id": "MAK080-2026-01-06-001",
  "timestamp": "2026-01-06T12:00:00Z"
}
```

### Get Valid Plants

**GET** `/api/v1/sap/plants`

Get list of valid plants for SAP integration.

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "plants": ["K002", "K013", "K012"],
  "plant_descriptions": {
    "K002": "Kenya Plant 002",
    "K013": "Kenya Plant 013",
    "K012": "Kenya Plant 012"
  }
}
```

### Get Order Types for Plant

**GET** `/api/v1/sap/order-types/{plant}`

Get valid order types for a specific plant.

**Path Parameters:**
- `plant` (string) - Plant code (K002, K013, or K012)

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "plant": "K013",
  "order_types": ["KE33"],
  "descriptions": {
    "KE33": "Production Order Type KE33"
  }
}
```

### SAP Integration Flow

#### Inbound Interfaces (Krystal → SAP)

1. **Material Trigger**
   - **Authentication**: JWT Bearer token (user authentication)
   - Krystal sends material + plant to SAP
   - SAP validates material/plant combination
   - SAP triggers process order extraction
   - SAP sends process orders back via outbound interface

2. **Component Consumption (MAK080)**
   - **Authentication**: JWT Bearer token (user authentication)
   - Krystal sends component consumption data
   - BTP transforms JSON to IDOC format
   - SAP processes MAK080 IDOC
   - SAP posts goods issue (261 movement)
   - SAP sends ALEAUD acknowledgment back

#### Outbound Interfaces (SAP → Krystal)

1. **Process Order Interface**
   - **Authentication**: API Key (X-API-Key header)
   - SAP sends process orders in response to material trigger
   - Includes order headers, material data, and components
   - Filtered by plant and order type configuration

2. **ALEAUD Acknowledgment**
   - **Authentication**: API Key (X-API-Key header)
   - SAP sends acknowledgment for MAK080 submissions
   - Provides success/failure status
   - Includes error details for failed transactions

### Authentication Methods

#### JWT Authentication (User Endpoints)
For user-facing endpoints (material-trigger, component-consumption):
```
Authorization: Bearer <jwt_token>
```

#### API Key Authentication (External Endpoints)
For system-to-system endpoints (receive-process-orders, receive-aleaud-acknowledgment):
```
X-API-Key: <sap_api_key>
```

### Receive Process Orders (External)

**POST** `/api/v1/sap/receive-process-orders`

**⚠️ External Endpoint**: This endpoint is called by SAP BTP, not by users.

Receive process orders from SAP ECC via BTP.

**Headers:**
```
X-API-Key: <sap_api_key>
Content-Type: application/json
```

**Request Body:**
```json
{
  "material_number": "FG001234",
  "plant": "K013",
  "process_orders": [
    {
      "header": {
        "order_number": "000012345678",
        "order_type": "KE33",
        "material_number": "FG001234",
        "plant": "K013",
        "total_order_quantity": 1000.0,
        "base_unit_of_measure": "KG",
        "basic_start_date": "2026-01-06",
        "basic_finish_date": "2026-01-10"
      },
      "material_data": {
        "material_description": "Finished Good Product",
        "material_type": "FERT"
      },
      "components": [
        {
          "material_number": "RM001234",
          "requirements_quantity": 25.5,
          "base_unit_of_measure": "KG",
          "requirements_date": "2026-01-06",
          "plant": "K013"
        }
      ],
      "status": ["REL"]
    }
  ]
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Received 1 process orders successfully",
  "material_number": "FG001234",
  "plant": "K013",
  "process_orders": [...],
  "timestamp": "2026-01-06T12:00:00Z"
}
```

### Receive ALEAUD Acknowledgment (External)

**POST** `/api/v1/sap/receive-aleaud-acknowledgment`

**⚠️ External Endpoint**: This endpoint is called by SAP BTP, not by users.

Receive ALEAUD acknowledgment from SAP ECC via BTP.

**Headers:**
```
X-API-Key: <sap_api_key>
Content-Type: application/json
```

**Request Body:**
```json
{
  "message_type": "Z2WMPOISSFX080",
  "idoc_number": "0000000001234567",
  "status": "53",
  "status_code": "53",
  "status_text": "Application document posted",
  "status_type": "S",
  "status_message_qualifier": "SAP",
  "status_message_id": "Z2MAK080",
  "status_message_number": "064",
  "parameters": {},
  "plant": "K013"
}
```

**Response (200):**
```json
{
  "message_type": "Z2WMPOISSFX080",
  "idoc_number": "0000000001234567",
  "status": "53",
  "status_code": "53",
  "status_text": "Application document posted",
  "status_type": "S",
  "success": true,
  "plant": "K013"
}
```

**SAP Error Response (400/500):**
```json
{
  "success": false,
  "error_code": "MATERIAL_PLANT_NOT_FOUND",
  "error_message": "Material and Plant combination does not exist",
  "details": {
    "material_number": "INVALID123",
    "plant": "K013"
  },
  "timestamp": "2026-01-06T12:00:00Z"
}
```

**Common SAP Error Codes:**
- `MATERIAL_PLANT_NOT_FOUND` - Invalid material/plant combination
- `VALIDATION_ERROR` - Request data validation failed
- `SAP_COMMUNICATION_ERROR` - Failed to communicate with SAP
- `SAP_PROCESSING_ERROR` - SAP processing failed
- `INSUFFICIENT_STOCK` - Not enough stock for consumption
- `INVALID_BATCH` - Batch number not found or expired

## 🏭 Product Management

### List Products

**GET** `/api/v1/products`

Get a list of all products.

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "products": [
    {
      "name": "Omo Washing Powder",
      "category": "Detergent",
      "ingredients_count": 8
    },
    {
      "name": "Sunlight Soap",
      "category": "Soap",
      "ingredients_count": 5
    }
  ]
}
```

### Get Product Ingredients

**GET** `/api/v1/products/{product_name}/ingredients`

Get ingredients and specifications for a specific product.

**Path Parameters:**
- `product_name` (string) - Product name

**Headers:**
```
Authorization: Bearer <jwt_token>
```

**Response (200):**
```json
{
  "product_name": "Omo Washing Powder",
  "ingredients": [
    {
      "id": 1,
      "name": "Sodium Carbonate",
      "target_weight": 25.500,
      "tolerance_plus": 0.050,
      "tolerance_minus": 0.050,
      "unit": "kg"
    },
    {
      "id": 2,
      "name": "Linear Alkylbenzene",
      "target_weight": 15.200,
      "tolerance_plus": 0.030,
      "tolerance_minus": 0.030,
      "unit": "kg"
    }
  ]
}
```

### Update Product Ingredient

**PUT** `/api/v1/products/{product_name}/ingredients/{ingredient_id}`

Update ingredient specifications.

**Path Parameters:**
- `product_name` (string) - Product name
- `ingredient_id` (int) - Ingredient ID

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "target_weight": 25.600,
  "tolerance_plus": 0.060,
  "tolerance_minus": 0.040
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Ingredient updated successfully",
  "data": {
    "id": 1,
    "name": "Sodium Carbonate",
    "target_weight": 25.600,
    "tolerance_plus": 0.060,
    "tolerance_minus": 0.040,
    "unit": "kg",
    "updated_at": "2026-01-06T12:00:00Z"
  }
}
```

## 📧 Email Service

### Send Welcome Email

**POST** `/api/v1/emails/send-welcome`

Send a welcome email to a new user.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "user_email": "newuser@example.com",
  "user_name": "New User",
  "temporary_password": "temp123456"
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Welcome email sent successfully",
  "email_sent_to": "newuser@example.com"
}
```

### Send Password Reset Email

**POST** `/api/v1/emails/forgot-password`

Send a password reset email.

**Headers:**
```
Content-Type: application/json
```

**Request Body:**
```json
{
  "email": "user@example.com"
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Password reset email sent successfully",
  "email_sent_to": "user@example.com"
}
```

### Test Email Service

**POST** `/api/v1/emails/test`

Test the email service configuration.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "test_email": "admin@example.com"
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Test email sent successfully",
  "smtp_config": {
    "host": "smtp.mailsafi.com",
    "port": 587,
    "secure": "tls",
    "from": "systems@krystal-ea.com"
  }
}
```

## 📊 Activity Logs

### Get User Activities

**POST** `/api/v1/activity-logs/user-activities`

Get user activity logs with filtering.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "page": 1,
  "limit": 50,
  "filters": {
    "user_id": "user123",
    "activity_type": "login",
    "date_from": "2026-01-01",
    "date_to": "2026-01-06"
  }
}
```

**Response (200):**
```json
{
  "items": [
    {
      "id": 1,
      "user_id": "user123",
      "activity_type": "login",
      "description": "User logged in successfully",
      "ip_address": "192.168.1.100",
      "user_agent": "Mozilla/5.0...",
      "created_at": "2026-01-06T12:00:00Z"
    }
  ],
  "total": 1,
  "page": 1,
  "limit": 50,
  "pages": 1
}
```

### Get Active Sessions

**POST** `/api/v1/activity-logs/active-sessions`

Get active user sessions.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "page": 1,
  "limit": 50,
  "filters": {
    "user_id": "user123"
  }
}
```

**Response (200):**
```json
{
  "items": [
    {
      "id": 1,
      "user_id": "user123",
      "session_token": "sess_abc123...",
      "ip_address": "192.168.1.100",
      "user_agent": "Mozilla/5.0...",
      "created_at": "2026-01-06T12:00:00Z",
      "last_activity": "2026-01-06T12:30:00Z"
    }
  ],
  "total": 1,
  "page": 1,
  "limit": 50,
  "pages": 1
}
```

### Activity Statistics

**POST** `/api/v1/activity-logs/statistics`

Get activity statistics and analytics.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "date_from": "2026-01-01",
  "date_to": "2026-01-06",
  "group_by": "activity_type"
}
```

**Response (200):**
```json
{
  "summary": {
    "total_activities": 1250,
    "unique_users": 85,
    "active_sessions": 42
  },
  "by_activity_type": {
    "login": 320,
    "logout": 315,
    "batch_submit": 180,
    "batch_approve": 95,
    "user_create": 12,
    "user_update": 28
  },
  "by_hour": {
    "08": 45,
    "09": 78,
    "10": 92,
    "11": 85,
    "12": 67,
    "13": 89,
    "14": 76,
    "15": 82,
    "16": 71,
    "17": 45
  }
}
```

## 🔐 Login Logs

### Get Login Logs

**POST** `/api/v1/login-logs`

Get login attempt logs with filtering.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "page": 1,
  "limit": 50,
  "filters": {
    "user_id": "user123",
    "success": true,
    "date_from": "2026-01-01",
    "date_to": "2026-01-06"
  }
}
```

**Response (200):**
```json
{
  "items": [
    {
      "id": 1,
      "user_id": "user123",
      "email": "user@example.com",
      "success": true,
      "failure_reason": null,
      "ip_address": "192.168.1.100",
      "user_agent": "Mozilla/5.0...",
      "created_at": "2026-01-06T12:00:00Z"
    }
  ],
  "total": 1,
  "page": 1,
  "limit": 50,
  "pages": 1
}
```

### Login Statistics

**POST** `/api/v1/login-logs/statistics`

Get login statistics and analytics.

**Headers:**
```
Authorization: Bearer <jwt_token>
Content-Type: application/json
```

**Request Body:**
```json
{
  "date_from": "2026-01-01",
  "date_to": "2026-01-06"
}
```

**Response (200):**
```json
{
  "summary": {
    "total_attempts": 450,
    "successful_logins": 425,
    "failed_attempts": 25,
    "success_rate": 94.4,
    "unique_users": 85
  },
  "by_day": {
    "2026-01-01": {"successful": 68, "failed": 3},
    "2026-01-02": {"successful": 72, "failed": 5},
    "2026-01-03": {"successful": 85, "failed": 4},
    "2026-01-04": {"successful": 78, "failed": 6},
    "2026-01-05": {"successful": 82, "failed": 4},
    "2026-01-06": {"successful": 40, "failed": 3}
  },
  "failure_reasons": {
    "Invalid password": 18,
    "User not found": 5,
    "Account inactive": 2
  }
}
```

## ❌ Error Handling

All API endpoints return structured error responses with consistent format.

### Error Response Format

```json
{
  "error": {
    "code": "ERROR_CODE",
    "message": "Human-readable error message",
    "details": {
      "field": "specific_field",
      "value": "invalid_value"
    },
    "timestamp": "2026-01-06T12:00:00Z"
  }
}
```

### Common Error Codes

| HTTP Status | Error Code | Description |
|-------------|------------|-------------|
| 400 | `VALIDATION_ERROR` | Request data validation failed |
| 400 | `MISSING_FIELD` | Required field is missing |
| 401 | `AUTH_FAILED` | Authentication failed |
| 401 | `TOKEN_EXPIRED` | JWT token has expired |
| 401 | `INVALID_CREDENTIALS` | Invalid email or password |
| 403 | `ACCESS_DENIED` | Insufficient permissions |
| 403 | `ROLE_REQUIRED` | Specific role required |
| 404 | `NOT_FOUND` | Resource not found |
| 404 | `USER_NOT_FOUND` | User does not exist |
| 409 | `EMAIL_EXISTS` | Email already registered |
| 422 | `BUSINESS_LOGIC_ERROR` | Business rule violation |
| 429 | `RATE_LIMIT_EXCEEDED` | Too many requests |
| 500 | `INTERNAL_ERROR` | Internal server error |
| 500 | `DATABASE_ERROR` | Database operation failed |

### Example Error Responses

**Validation Error (400):**
```json
{
  "error": {
    "code": "VALIDATION_ERROR",
    "message": "Invalid email format",
    "details": {
      "field": "email",
      "value": "invalid-email"
    },
    "timestamp": "2026-01-06T12:00:00Z"
  }
}
```

**Authentication Error (401):**
```json
{
  "error": {
    "code": "TOKEN_EXPIRED",
    "message": "JWT token has expired",
    "timestamp": "2026-01-06T12:00:00Z"
  }
}
```

**Authorization Error (403):**
```json
{
  "error": {
    "code": "ROLE_REQUIRED",
    "message": "Admin role required for this operation",
    "timestamp": "2026-01-06T12:00:00Z"
  }
}
```

## 🚦 Rate Limiting

Rate limiting is implemented to prevent abuse and ensure fair usage.

### Rate Limits

| Endpoint Category | Limit | Window |
|------------------|-------|--------|
| Authentication | 5 requests | 1 minute |
| User Management | 100 requests | 1 hour |
| Batch Operations | 200 requests | 1 hour |
| General API | 1000 requests | 1 hour |

### Rate Limit Headers

```
X-RateLimit-Limit: 100
X-RateLimit-Remaining: 95
X-RateLimit-Reset: 1641484800
```

### Rate Limit Exceeded Response (429)

```json
{
  "error": {
    "code": "RATE_LIMIT_EXCEEDED",
    "message": "Rate limit exceeded. Try again later.",
    "details": {
      "limit": 100,
      "window": "1 hour",
      "reset_at": "2026-01-06T13:00:00Z"
    },
    "timestamp": "2026-01-06T12:00:00Z"
  }
}
```

## 📝 Examples

### Complete Authentication Flow

```bash
# 1. Login
curl -X POST "http://localhost:8000/api/v1/auth/login" \
  -H "Content-Type: application/json" \
  -d '{
    "email": "user@example.com",
    "password": "password123"
  }'

# Response: {"access_token": "eyJ...", "token_type": "bearer"}

# 2. Use token for authenticated requests
curl -X GET "http://localhost:8000/api/v1/users" \
  -H "Authorization: Bearer eyJ..."

# 3. Get current user info
curl -X GET "http://localhost:8000/api/v1/auth/me" \
  -H "Authorization: Bearer eyJ..."

# 4. Logout
curl -X GET "http://localhost:8000/api/v1/auth/logout" \
  -H "Authorization: Bearer eyJ..."
```

### Batch Processing Workflow

```bash
# 1. Submit batch data
curl -X POST "http://localhost:8000/api/v1/batches/submit" \
  -H "Authorization: Bearer eyJ..." \
  -H "Content-Type: application/json" \
  -d '{
    "product_name": "Omo Washing Powder",
    "batch_number": "BATCH001",
    "items": [
      {
        "ingredient_name": "Sodium Carbonate",
        "target_weight": 25.500,
        "actual_weight": 25.485,
        "variance": -0.015
      }
    ]
  }'

# 2. Check pending approvals (as analyst)
curl -X GET "http://localhost:8000/api/v1/batches/pending-approvals" \
  -H "Authorization: Bearer eyJ..."

# 3. Approve items
curl -X POST "http://localhost:8000/api/v1/batches/approve" \
  -H "Authorization: Bearer eyJ..." \
  -H "Content-Type: application/json" \
  -d '{
    "item_ids": [123, 124],
    "action": "approve"
  }'

# 4. Get processed data for reporting
curl -X POST "http://localhost:8000/api/v1/batches/processed-data" \
  -H "Authorization: Bearer eyJ..." \
  -H "Content-Type: application/json" \
  -d '{
    "page": 1,
    "limit": 100,
    "filters": {
      "date_from": "2026-01-01",
      "date_to": "2026-01-06"
    }
  }'
```

### User Management Example

```bash
# 1. List users with filtering
curl -X GET "http://localhost:8000/api/v1/users?search=john&title=analyst&page=1&limit=10" \
  -H "Authorization: Bearer eyJ..."

# 2. Create new user
curl -X POST "http://localhost:8000/api/v1/users" \
  -H "Authorization: Bearer eyJ..." \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Jane Smith",
    "email": "jane@example.com",
    "password": "securepassword123",
    "title": "analyst",
    "shift": "Day Shift",
    "areas": "Food,Soap"
  }'

# 3. Update user
curl -X PUT "http://localhost:8000/api/v1/users/user456" \
  -H "Authorization: Bearer eyJ..." \
  -H "Content-Type: application/json" \
  -d '{
    "title": "team_lead",
    "areas": "Food,Soap,BW"
  }'

# 4. Get user statistics
curl -X GET "http://localhost:8000/api/v1/users/statistics/overview" \
  -H "Authorization: Bearer eyJ..."
```

---

For more information, visit the interactive API documentation at `/docs` when running the server.